#include "WCSimDetectorConstruction.hh"
#include "WCSimDetectorMessenger.hh"
#include "WCSimTuningParameters.hh"

#include "G4Material.hh"
#include "G4Element.hh"
#include "G4Box.hh"
#include "G4LogicalVolume.hh"
#include "G4VPhysicalVolume.hh"
#include "G4PVPlacement.hh"
#include "G4ThreeVector.hh"
#include "globals.hh"
#include "G4VisAttributes.hh"

#include "G4RunManager.hh"
#include "G4PhysicalVolumeStore.hh"
#include "G4LogicalVolumeStore.hh"
#include "G4SolidStore.hh"

std::map<int, G4Transform3D> WCSimDetectorConstruction::tubeIDMap;
//std::map<int, cyl_location>  WCSimDetectorConstruction::tubeCylLocation;
hash_map<std::string, int, hash<std::string> > 
WCSimDetectorConstruction::tubeLocationMap;

WCSimDetectorConstruction::WCSimDetectorConstruction(G4int DetConfig,WCSimTuningParameters* WCSimTuningPars):WCSimTuningParams(WCSimTuningPars)
{
	
  // Decide if (only for the case of !1kT detector) should be upright or horizontal
  isUpright = false;
  isHyperK  = false;
  isNuPrism  = false;

  debugMode = false;
//-----------------------------------------------------
// Initilize SD pointers
//-----------------------------------------------------

      aWCPMT     = NULL;

  myConfiguration = DetConfig;

  //-----------------------------------------------------
  // Create Materials
  //-----------------------------------------------------
    
  ConstructMaterials();

  //-----------------------------------------------------
  // Initialize things related to the tubeID
  //-----------------------------------------------------

  WCSimDetectorConstruction::tubeIDMap.clear();
  //WCSimDetectorConstruction::tubeCylLocation.clear();// (JF) Removed
  WCSimDetectorConstruction::tubeLocationMap.clear();
  WCSimDetectorConstruction::PMTLogicalVolumes.clear();
  totalNumPMTs = 0;
  WCPMTExposeHeight= 0.;
  //-----------------------------------------------------
  // Set the default WC geometry.  This can be changed later.
  //-----------------------------------------------------

  SetSuperKGeometry();
  //SetHyperKGeometry();
  //DUSEL_100kton_10inch_40perCent();
  //DUSEL_100kton_10inch_HQE_12perCent();
  //DUSEL_100kton_10inch_HQE_30perCent();
  //DUSEL_100kton_10inch_HQE_30perCent_Gd();
  //DUSEL_150kton_10inch_HQE_30perCent();
  //DUSEL_200kton_10inch_HQE_12perCent();
  //DUSEL_200kton_12inch_HQE_10perCent();
  //DUSEL_200kton_12inch_HQE_14perCent();


  //----------------------------------------------------- 
  // Set whether or not Pi0-specific info is saved
  //-----------------------------------------------------

  SavePi0Info(false);
  
  //-----------------------------------------------------
  // Set the default method for implementing the PMT QE
  //-----------------------------------------------------
  SetPMT_QE_Method(1);
   //default NOT to use collection efficiency
  SetPMT_Coll_Eff(0);

  //----------------------------------------------------- 
  // Make the detector messenger to allow changing geometry
  //-----------------------------------------------------

  messenger = new WCSimDetectorMessenger(this);
}

#include "G4GeometryManager.hh"
#include "G4LogicalBorderSurface.hh"
#include "G4LogicalSkinSurface.hh"

void WCSimDetectorConstruction::UpdateGeometry()
{
 
  
  G4bool geomChanged = true;
  G4RunManager::GetRunManager()->DefineWorldVolume(Construct(), geomChanged);
 
 }



WCSimDetectorConstruction::~WCSimDetectorConstruction(){
  for (int i=0;i<fpmts.size();i++){
    delete fpmts.at(i);
  }
  fpmts.clear();
}

//G4ThreeVector WCSimDetectorConstruction::GetTranslationFromSettings()
//{
//    TFile* inFile = TFile::Open(fInputSettingsFilename);
//    if (!inFile){
//        G4ThreeVector trans;
//        return trans;
//    }
//
//    fSettingsTree = (TTree*) inFile->Get("Settings");
//    if (!fSettingsTree){
//        G4ThreeVector trans;
//        return trans;
//    }
//
//    float NuBeamAng;
//    float NuIdfdPos[3];
//    for(int i = 0; i < 3; ++i) NuIdfdPos[i] = 0;
//    fSettingsTree->SetBranchAddress("NuIdfdPos",NuIdfdPos); 
//    fSettingsTree->SetBranchAddress("NuBeamAng",&NuBeamAng); 
//
//    fSettingsTree->GetEntry(0);
//
//    G4ThreeVector trans(0, (NuIdfdPos[1] + TMath::Tan(NuBeamAng)*600.0), NuIdfdPos[2] + 600.0);
//
//    inFile->Close();
//    fInputSettingsFilename = "";
//    fSettingsTree = NULL;
//
//    return trans;
//
//}

G4VPhysicalVolume* WCSimDetectorConstruction::Construct()
{  
  G4GeometryManager::GetInstance()->OpenGeometry();

  G4PhysicalVolumeStore::GetInstance()->Clean();
  G4LogicalVolumeStore::GetInstance()->Clean();
  G4SolidStore::GetInstance()->Clean();
  G4LogicalBorderSurface::CleanSurfaceTable();
  G4LogicalSkinSurface::CleanSurfaceTable();
  WCSimDetectorConstruction::PMTLogicalVolumes.clear();

  totalNumPMTs = 0;
  
  //-----------------------------------------------------
  // Create Logical Volumes
  //-----------------------------------------------------

  // First create the logical volumes of the sub detectors.  After they are 
  // created their size will be used to make the world volume.
  // Note the order is important because they rearrange themselves depending
  // on their size and detector ordering.

  G4LogicalVolume* logicWCBox;
  // Select between HyperK and cylinder
  if (isHyperK) logicWCBox = ConstructHyperK();
  else logicWCBox = ConstructCylinder(); 

  G4cout << " WCLength       = " << WCLength/m << " m"<< G4endl;

  //-------------------------------

  // Now make the detector Hall.  The lengths of the subdectors 
  // were set above.

  G4ThreeVector position(0,0,0);// = GetTranslationFromSettings();

  if(isNuPrism) position.setY(WCIDVerticalPosition);

  std::cout << "position Y = " << position.y() << std::endl;


  G4double expHallLength = 3.*WCLength; //jl145 - extra space to simulate cosmic muons more easily

  G4cout << " expHallLength = " << expHallLength / m << G4endl;
  G4double expHallHalfLength = 0.5*expHallLength;

  G4Box* solidExpHall = new G4Box("expHall",
				  expHallHalfLength + fabs(position.x()),
				  expHallHalfLength + fabs(position.y()),
				  expHallHalfLength + fabs(position.z()));
  
  G4LogicalVolume* logicExpHall = 
    new G4LogicalVolume(solidExpHall,
			G4Material::GetMaterial("Vacuum"),
			"expHall",
			0,0,0);

  // Now set the visualization attributes of the logical volumes.

  //   logicWCBox->SetVisAttributes(G4VisAttributes::Invisible);
  logicExpHall->SetVisAttributes(G4VisAttributes::Invisible);

  //-----------------------------------------------------
  // Create and place the physical Volumes
  //-----------------------------------------------------

  // Experimental Hall
  G4VPhysicalVolume* physiExpHall = 
    new G4PVPlacement(0,G4ThreeVector(),
  		      logicExpHall,
  		      "expHall",
  		      0,false,0,true);

  // Water Cherenkov Detector (WC) mother volume
  // WC Box, nice to turn on for x and y views to provide a frame:

  G4RotationMatrix* rotationMatrix = new G4RotationMatrix;
  G4ThreeVector genPosition = G4ThreeVector(0., 0., WCPosition);

  if(isNuPrism){
      rotationMatrix->rotateX(90.*deg);
      rotationMatrix->rotateY(0.*deg);
      rotationMatrix->rotateZ(0.*deg);
      genPosition.setX(position.x());
      genPosition.setY(position.y());
      genPosition.setZ(position.z());
  }

  G4Transform3D transform(*rotationMatrix, genPosition);

  G4VPhysicalVolume* physiWCBox = 
    new G4PVPlacement(transform,
		      logicWCBox,
		      "WCBox",
		      logicExpHall,
		      false,
		      0);

  // Reset the tubeID and tubeLocation maps before refiling them
  tubeIDMap.clear();
  tubeLocationMap.clear();


  // Traverse and print the geometry Tree
  
  //  TraverseReplicas(physiWCBox, 0, G4Transform3D(), 
  //	   &WCSimDetectorConstruction::PrintGeometryTree) ;
  
  TraverseReplicas(physiWCBox, 0, G4Transform3D(), 
	           &WCSimDetectorConstruction::DescribeAndRegisterPMT) ;
  
  
  TraverseReplicas(physiWCBox, 0, G4Transform3D(), 
		   &WCSimDetectorConstruction::GetWCGeom) ;
  
  DumpGeometryTableToFile();
  
  // Return the pointer to the physical experimental hall
  return physiExpHall;
}

WCSimPMTObject *WCSimDetectorConstruction::CreatePMTObject(G4String PMTType)
{
  if (PMTType == "PMT20inch"){
     WCSimPMTObject* PMT = new PMT20inch;
      WCSimDetectorConstruction::SetPMTPointer(PMT);
      return PMT;
  }
  else if (PMTType == "PMT8inch"){
    WCSimPMTObject* PMT = new PMT8inch;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "PMT5inch"){
    WCSimPMTObject* PMT = new PMT5inch;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "PMT3inch"){
    WCSimPMTObject* PMT = new PMT3inch;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "PMT3inchGT"){
    WCSimPMTObject* PMT = new PMT3inchGT;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "PMT3inchR12199_02"){
    WCSimPMTObject* PMT = new PMT3inchR12199_02;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "PMT10inch"){
    WCSimPMTObject* PMT = new PMT10inch;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "PMT10inchHQE"){
    WCSimPMTObject* PMT = new PMT10inchHQE;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "PMT12inchHQE"){
    WCSimPMTObject* PMT = new PMT12inchHQE;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "HPD20inchHQE"){
    WCSimPMTObject* PMT = new HPD20inchHQE;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "HPD12inchHQE"){
    WCSimPMTObject* PMT = new HPD12inchHQE;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "BoxandLine20inchHQE"){
    WCSimPMTObject* PMT = new BoxandLine20inchHQE;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }
  else if (PMTType == "BoxandLine12inchHQE"){
    WCSimPMTObject* PMT = new BoxandLine12inchHQE;
    WCSimDetectorConstruction::SetPMTPointer(PMT);
    return PMT;
  }

  else { G4cout << PMTType << " is not a recognized PMT Type. Exiting WCSim." << G4endl; exit(1);}
}
