#include "WCSimWCPMT.hh"
#include "WCSimPMTObject.hh"
#include "WCSimWCDigi.hh"
#include "WCSimWCHit.hh"

#include "G4EventManager.hh"
#include "G4Event.hh"
#include "G4SDManager.hh"
#include "G4DigiManager.hh"
#include "G4ios.hh"
#include "G4RotationMatrix.hh"
#include "G4ThreeVector.hh"

#include "WCSimDetectorConstruction.hh"
#include "WCSimPmtInfo.hh"


#include <vector>
// for memset
#include <cstring>
#include <iostream>


////////////////////////////////////////////////////////////////////////////////////////////////
// PMT Base Class

G4float  WCSimPMTObject::GetCollectionEfficiency(float angle)
{
    return Interpolate_func(angle, 10, GetCollectionEfficiencyAngle(), GetCollectionEfficiencyArray())/100.;
}

G4float WCSimPMTObject::Interpolate_func(G4float x, G4int ncount, G4float *angle, G4float *quantity){
  // linear interpolate the quantity function versus angle                                                                                                                        
  if (x < *angle || x >=*(angle+ncount-1)){
    return 0;
  }else{
    for (Int_t i=0;i!=ncount;i++){
      if (x>=*(angle+i) && x < *(angle+i+1)){
        return (x-*(angle+i))/(*(angle+i+1)-*(angle+i))* (*(quantity+i+1)) + (*(angle+i+1)-x)/(*(angle+i+1)-*(angle+i)) * (*(quantity+i));
      }
    }
  }

  // Error Condition
  G4cerr << "Interpolation failure." << G4endl;
  assert(false);
  return -999.;
}


// By default, collection efficiency is binned in 10-degree angular bins from 0 to 90
// This can be overridden by setting GetCE in the derived class
G4float* WCSimPMTObject::GetCollectionEfficiencyAngle(){
  static G4float angle[10] = { 0., 10., 20., 30., 40., 50., 60., 70., 80., 90.};
  return angle;
}


// By default, each PMT has 100% collection efficiency at all angles
// This can be overridden by setting GetCE in the derived class
G4float* WCSimPMTObject::GetCollectionEfficiencyArray(){
  static G4float CE[10] = { 100., 100., 100., 100., 100., 100., 100., 100., 100., 100.};
  // CollectionEfficiency before modification on 2015-03-27 (Different from SKDetSim)
  // static G4float CE[10]={100,100,99,95,90,85,80,69,35,13}; 
  return CE;
}



////////////////////////////////////////////////////////////////////////////////////////////////
// 20 inch


PMT20inch::PMT20inch() {}
PMT20inch::~PMT20inch(){}

G4String PMT20inch::GetPMTName() {G4String PMTName = "20inch"; return PMTName;}
G4double PMT20inch::GetExposeHeight() {return .18*m;}
G4double PMT20inch::GetRadius() {return .254*m;}
G4double PMT20inch::GetPMTGlassThickness() {return 0.4*cm;}
float PMT20inch::HitTimeSmearing(float Q) {
  float timingConstant = 10.0; 
  float timingResolution = 0.33 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 0.58) timingResolution=0.58;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* PMT20inch::Getqpe()
   {
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000129, 0.000754, 0.004060, 0.028471,
    // 2
    0.068449, 0.115679, 0.164646, 0.203466, 0.235631,
    0.262351, 0.282064, 0.303341, 0.320618, 0.338317,
    0.357825, 0.371980, 0.385820, 0.398838, 0.413595,
    0.428590, 0.444387, 0.461685, 0.482383, 0.502369,
    0.520779, 0.540011, 0.559293, 0.579354, 0.599337,
    0.619580, 0.639859, 0.659807, 0.679810, 0.699620,
    0.718792, 0.737382, 0.755309, 0.772042, 0.788232,
    0.803316, 0.817861, 0.831148, 0.844339, 0.855532,
    0.866693, 0.876604, 0.886067, 0.894473, 0.902150,
    0.909515, 0.915983, 0.922050, 0.927418, 0.932492,
    // 3
    0.936951, 0.940941, 0.944660, 0.948004, 0.951090,
    0.953833, 0.956576, 0.958886, 0.961134, 0.963116,
    0.964930, 0.966562, 0.968008, 0.969424, 0.970687,
    0.971783, 0.972867, 0.973903, 0.974906, 0.975784,
    0.976632, 0.977438, 0.978190, 0.978891, 0.979543,
    0.980124, 0.980666, 0.981255, 0.981770, 0.982227,
    0.982701, 0.983146, 0.983566, 0.983975, 0.984357,
    0.984713, 0.985094, 0.985404, 0.985739, 0.986049,
    0.986339, 0.986630, 0.986922, 0.987176, 0.987431,
    0.987655, 0.987922, 0.988173, 0.988414, 0.988639,
    // 4
    0.988856, 0.989065, 0.989273, 0.989475, 0.989662,
    0.989828, 0.990007, 0.990172, 0.990327, 0.990497,
    0.990645, 0.990797, 0.990981, 0.991135, 0.991272,
    0.991413, 0.991550, 0.991673, 0.991805, 0.991928,
    0.992063, 0.992173, 0.992296, 0.992406, 0.992514,
    0.992632, 0.992733, 0.992837, 0.992954, 0.993046,
    0.993148, 0.993246, 0.993354, 0.993458, 0.993549,
    0.993656, 0.993744, 0.993836, 0.993936, 0.994033,
    0.994134, 0.994222, 0.994307, 0.994413, 0.994495,
    0.994572, 0.994659, 0.994739, 0.994816, 0.994886,
    // 5
    0.994970, 0.995032, 0.995110, 0.995178, 0.995250,
    0.995321, 0.995383, 0.995464, 0.995532, 0.995609,
    0.995674, 0.995750, 0.995821, 0.995889, 0.995952,
    0.996010, 0.996071, 0.996153, 0.996218, 0.996283,
    0.996335, 0.996384, 0.996431, 0.996484, 0.996537,
    0.996597, 0.996655, 0.996701, 0.996745, 0.996802,
    0.996860, 0.996917, 0.996962, 0.997014, 0.997079,
    0.997114, 0.997165, 0.997204, 0.997250, 0.997295,
    0.997335, 0.997379, 0.997418, 0.997454, 0.997488,
    0.997530, 0.997573, 0.997606, 0.997648, 0.997685,
    // 6
    0.997725, 0.997762, 0.997795, 0.997835, 0.997866,
    0.997898, 0.997941, 0.997966, 0.997997, 0.998039,
    0.998065, 0.998104, 0.998128, 0.998153, 0.998179,
    0.998205, 0.998223, 0.998254, 0.998293, 0.998319,
    0.998346, 0.998374, 0.998397, 0.998414, 0.998432,
    0.998456, 0.998482, 0.998511, 0.998532, 0.998553,
    0.998571, 0.998594, 0.998614, 0.998638, 0.998669,
    0.998693, 0.998715, 0.998743, 0.998762, 0.998793,
    0.998812, 0.998834, 0.998857, 0.998872, 0.998888,
    0.998904, 0.998926, 0.998946, 0.998963, 0.998983,
    // 7
    0.999007, 0.999027, 0.999044, 0.999064, 0.999079,
    0.999096, 0.999120, 0.999133, 0.999152, 0.999160,
    0.999174, 0.999188, 0.999206, 0.999221, 0.999234,
    0.999248, 0.999263, 0.999276, 0.999286, 0.999300,
    0.999313, 0.999321, 0.999331, 0.999347, 0.999356,
    0.999369, 0.999381, 0.999394, 0.999402, 0.999415,
    0.999427, 0.999433, 0.999446, 0.999458, 0.999472,
    0.999484, 0.999499, 0.999513, 0.999522, 0.999532,
    0.999540, 0.999550, 0.999559, 0.999567, 0.999574,
    0.999588, 0.999599, 0.999613, 0.999618, 0.999627,
    // 8
    0.999635, 0.999639, 0.999652, 0.999662, 0.999667,
    0.999671, 0.999678, 0.999682, 0.999688, 0.999693,
    0.999698, 0.999701, 0.999706, 0.999711, 0.999718,
    0.999722, 0.999727, 0.999732, 0.999737, 0.999740,
    0.999746, 0.999750, 0.999754, 0.999763, 0.999766,
    0.999769, 0.999774, 0.999780, 0.999784, 0.999788,
    0.999796, 0.999803, 0.999807, 0.999809, 0.999815,
    0.999820, 0.999827, 0.999830, 0.999833, 0.999833,
    0.999836, 0.999839, 0.999842, 0.999845, 0.999850,
    0.999853, 0.999857, 0.999860, 0.999865, 0.999870,
    // 9
    0.999873, 0.999877, 0.999880, 0.999882, 0.999883,
    0.999886, 0.999888, 0.999889, 0.999895, 0.999896,
    0.999897, 0.999901, 0.999902, 0.999905, 0.999907,
    0.999907, 0.999909, 0.999911, 0.999911, 0.999912,
    0.999913, 0.999914, 0.999917, 0.999919, 0.999921,
    0.999923, 0.999927, 0.999929, 0.999931, 0.999933,
    0.999936, 0.999942, 0.999942, 0.999944, 0.999947,
    0.999947, 0.999948, 0.999949, 0.999952, 0.999955,
    0.999957, 0.999957, 0.999961, 0.999962, 0.999963,
    0.999963, 0.999963, 0.999964, 0.999965, 0.999965,
    // 10
    0.999965, 0.999965, 0.999966, 0.999968, 0.999969,
    0.999971, 0.999972, 0.999972, 0.999973, 0.999975,
    0.999975, 0.999975, 0.999975, 0.999975, 0.999975,
    0.999975, 0.999979, 0.999979, 0.999980, 0.999982,
    0.999983, 0.999985, 0.999986, 0.999987, 0.999987,
    0.999988, 0.999989, 0.999989, 0.999989, 0.999989,
    0.999990, 0.999990, 0.999992, 0.999993, 0.999994,
    0.999994, 0.999994, 0.999994, 0.999994, 0.999995,
    0.999995, 0.999995, 0.999996, 0.999996, 0.999996,
    0.999996, 0.999998, 0.999999, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array
    0.0 
  };
   return qpe0;
  }

G4float* PMT20inch::GetQEWavelength(){
  static G4float wavelength_value[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength_value;
}

G4float* PMT20inch::GetQE(){  
  static G4float QE[20] = { 0.00, .0139, .0854, .169, .203, .206, .211, .202,.188, .167, .140, .116, .0806, .0432, .0265, .0146, .00756, .00508, .00158, 0.00};
  return QE;
}
G4float PMT20inch::GetmaxQE(){
  const G4float maxQE = 0.211;
  return maxQE;
}



////////////////////////////////////////////////////////////////////////////////////////////////
// 8 inch

PMT8inch::PMT8inch(){}
PMT8inch::~PMT8inch(){}

G4String PMT8inch::GetPMTName() {G4String PMTName = "8inch"; return PMTName;}
G4double PMT8inch::GetExposeHeight() {return 91.6*mm;}
G4double PMT8inch::GetRadius() {return 101.6*mm;}
G4double PMT8inch::GetPMTGlassThickness() {return 0.55*cm;} //currently the same as 10inch
G4float PMT8inch::HitTimeSmearing(float Q) { 
  float timingConstant = 1.890; 
  float timingResolution = 0.33 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 0.58) timingResolution=0.58;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* PMT8inch::Getqpe() //currently uses the same as 20inch
   {
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000129, 0.000754, 0.004060, 0.028471,
    // 2
    0.068449, 0.115679, 0.164646, 0.203466, 0.235631,
    0.262351, 0.282064, 0.303341, 0.320618, 0.338317,
    0.357825, 0.371980, 0.385820, 0.398838, 0.413595,
    0.428590, 0.444387, 0.461685, 0.482383, 0.502369,
    0.520779, 0.540011, 0.559293, 0.579354, 0.599337,
    0.619580, 0.639859, 0.659807, 0.679810, 0.699620,
    0.718792, 0.737382, 0.755309, 0.772042, 0.788232,
    0.803316, 0.817861, 0.831148, 0.844339, 0.855532,
    0.866693, 0.876604, 0.886067, 0.894473, 0.902150,
    0.909515, 0.915983, 0.922050, 0.927418, 0.932492,
    // 3
    0.936951, 0.940941, 0.944660, 0.948004, 0.951090,
    0.953833, 0.956576, 0.958886, 0.961134, 0.963116,
    0.964930, 0.966562, 0.968008, 0.969424, 0.970687,
    0.971783, 0.972867, 0.973903, 0.974906, 0.975784,
    0.976632, 0.977438, 0.978190, 0.978891, 0.979543,
    0.980124, 0.980666, 0.981255, 0.981770, 0.982227,
    0.982701, 0.983146, 0.983566, 0.983975, 0.984357,
    0.984713, 0.985094, 0.985404, 0.985739, 0.986049,
    0.986339, 0.986630, 0.986922, 0.987176, 0.987431,
    0.987655, 0.987922, 0.988173, 0.988414, 0.988639,
    // 4
    0.988856, 0.989065, 0.989273, 0.989475, 0.989662,
    0.989828, 0.990007, 0.990172, 0.990327, 0.990497,
    0.990645, 0.990797, 0.990981, 0.991135, 0.991272,
    0.991413, 0.991550, 0.991673, 0.991805, 0.991928,
    0.992063, 0.992173, 0.992296, 0.992406, 0.992514,
    0.992632, 0.992733, 0.992837, 0.992954, 0.993046,
    0.993148, 0.993246, 0.993354, 0.993458, 0.993549,
    0.993656, 0.993744, 0.993836, 0.993936, 0.994033,
    0.994134, 0.994222, 0.994307, 0.994413, 0.994495,
    0.994572, 0.994659, 0.994739, 0.994816, 0.994886,
    // 5
    0.994970, 0.995032, 0.995110, 0.995178, 0.995250,
    0.995321, 0.995383, 0.995464, 0.995532, 0.995609,
    0.995674, 0.995750, 0.995821, 0.995889, 0.995952,
    0.996010, 0.996071, 0.996153, 0.996218, 0.996283,
    0.996335, 0.996384, 0.996431, 0.996484, 0.996537,
    0.996597, 0.996655, 0.996701, 0.996745, 0.996802,
    0.996860, 0.996917, 0.996962, 0.997014, 0.997079,
    0.997114, 0.997165, 0.997204, 0.997250, 0.997295,
    0.997335, 0.997379, 0.997418, 0.997454, 0.997488,
    0.997530, 0.997573, 0.997606, 0.997648, 0.997685,
    // 6
    0.997725, 0.997762, 0.997795, 0.997835, 0.997866,
    0.997898, 0.997941, 0.997966, 0.997997, 0.998039,
    0.998065, 0.998104, 0.998128, 0.998153, 0.998179,
    0.998205, 0.998223, 0.998254, 0.998293, 0.998319,
    0.998346, 0.998374, 0.998397, 0.998414, 0.998432,
    0.998456, 0.998482, 0.998511, 0.998532, 0.998553,
    0.998571, 0.998594, 0.998614, 0.998638, 0.998669,
    0.998693, 0.998715, 0.998743, 0.998762, 0.998793,
    0.998812, 0.998834, 0.998857, 0.998872, 0.998888,
    0.998904, 0.998926, 0.998946, 0.998963, 0.998983,
    // 7
    0.999007, 0.999027, 0.999044, 0.999064, 0.999079,
    0.999096, 0.999120, 0.999133, 0.999152, 0.999160,
    0.999174, 0.999188, 0.999206, 0.999221, 0.999234,
    0.999248, 0.999263, 0.999276, 0.999286, 0.999300,
    0.999313, 0.999321, 0.999331, 0.999347, 0.999356,
    0.999369, 0.999381, 0.999394, 0.999402, 0.999415,
    0.999427, 0.999433, 0.999446, 0.999458, 0.999472,
    0.999484, 0.999499, 0.999513, 0.999522, 0.999532,
    0.999540, 0.999550, 0.999559, 0.999567, 0.999574,
    0.999588, 0.999599, 0.999613, 0.999618, 0.999627,
    // 8
    0.999635, 0.999639, 0.999652, 0.999662, 0.999667,
    0.999671, 0.999678, 0.999682, 0.999688, 0.999693,
    0.999698, 0.999701, 0.999706, 0.999711, 0.999718,
    0.999722, 0.999727, 0.999732, 0.999737, 0.999740,
    0.999746, 0.999750, 0.999754, 0.999763, 0.999766,
    0.999769, 0.999774, 0.999780, 0.999784, 0.999788,
    0.999796, 0.999803, 0.999807, 0.999809, 0.999815,
    0.999820, 0.999827, 0.999830, 0.999833, 0.999833,
    0.999836, 0.999839, 0.999842, 0.999845, 0.999850,
    0.999853, 0.999857, 0.999860, 0.999865, 0.999870,
    // 9
    0.999873, 0.999877, 0.999880, 0.999882, 0.999883,
    0.999886, 0.999888, 0.999889, 0.999895, 0.999896,
    0.999897, 0.999901, 0.999902, 0.999905, 0.999907,
    0.999907, 0.999909, 0.999911, 0.999911, 0.999912,
    0.999913, 0.999914, 0.999917, 0.999919, 0.999921,
    0.999923, 0.999927, 0.999929, 0.999931, 0.999933,
    0.999936, 0.999942, 0.999942, 0.999944, 0.999947,
    0.999947, 0.999948, 0.999949, 0.999952, 0.999955,
    0.999957, 0.999957, 0.999961, 0.999962, 0.999963,
    0.999963, 0.999963, 0.999964, 0.999965, 0.999965,
    // 10
    0.999965, 0.999965, 0.999966, 0.999968, 0.999969,
    0.999971, 0.999972, 0.999972, 0.999973, 0.999975,
    0.999975, 0.999975, 0.999975, 0.999975, 0.999975,
    0.999975, 0.999979, 0.999979, 0.999980, 0.999982,
    0.999983, 0.999985, 0.999986, 0.999987, 0.999987,
    0.999988, 0.999989, 0.999989, 0.999989, 0.999989,
    0.999990, 0.999990, 0.999992, 0.999993, 0.999994,
    0.999994, 0.999994, 0.999994, 0.999994, 0.999995,
    0.999995, 0.999995, 0.999996, 0.999996, 0.999996,
    0.999996, 0.999998, 0.999999, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array
    0.0 
  };
   return qpe0;
  }

//Currenly the PMT QE info is the same as 20 inch.
G4float* PMT8inch::GetQE(){
  static G4float QE[20] = { 0.00, .0139, .0854, .169, .203, .206, .211, .202,.188, .167, .140, .116, .0806, .0432, .0265, .0146, .00756, .00508, .00158, 0.00};
  return QE;
}
G4float* PMT8inch::GetQEWavelength(){static G4float wavelength[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength;}

G4float  PMT8inch::GetmaxQE(){
    const G4float maxQE = 0.211;
  return maxQE;
}



////////////////////////////////////////////////////////////////////////////////////////////////
// 5 inch

PMT5inch::PMT5inch(){}
PMT5inch::~PMT5inch(){}

G4String PMT5inch::GetPMTName() {G4String PMTName = "5inch"; return PMTName;}
G4double PMT5inch::GetExposeHeight() {return 57.*mm;} //rough estimation
G4double PMT5inch::GetRadius() {return 63.5*mm;}
G4double PMT5inch::GetPMTGlassThickness() {return 0.55*cm;} //currently the same as 10inch
G4float PMT5inch::HitTimeSmearing(float Q) { 
  float timingConstant = 1.890;  //currently the same as 8inch
  float timingResolution = 0.33 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 0.58) timingResolution=0.58;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* PMT5inch::Getqpe() //currently uses the same as 20inch
{
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000129, 0.000754, 0.004060, 0.028471,
    // 2
    0.068449, 0.115679, 0.164646, 0.203466, 0.235631,
    0.262351, 0.282064, 0.303341, 0.320618, 0.338317,
    0.357825, 0.371980, 0.385820, 0.398838, 0.413595,
    0.428590, 0.444387, 0.461685, 0.482383, 0.502369,
    0.520779, 0.540011, 0.559293, 0.579354, 0.599337,
    0.619580, 0.639859, 0.659807, 0.679810, 0.699620,
    0.718792, 0.737382, 0.755309, 0.772042, 0.788232,
    0.803316, 0.817861, 0.831148, 0.844339, 0.855532,
    0.866693, 0.876604, 0.886067, 0.894473, 0.902150,
    0.909515, 0.915983, 0.922050, 0.927418, 0.932492,
    // 3
    0.936951, 0.940941, 0.944660, 0.948004, 0.951090,
    0.953833, 0.956576, 0.958886, 0.961134, 0.963116,
    0.964930, 0.966562, 0.968008, 0.969424, 0.970687,
    0.971783, 0.972867, 0.973903, 0.974906, 0.975784,
    0.976632, 0.977438, 0.978190, 0.978891, 0.979543,
    0.980124, 0.980666, 0.981255, 0.981770, 0.982227,
    0.982701, 0.983146, 0.983566, 0.983975, 0.984357,
    0.984713, 0.985094, 0.985404, 0.985739, 0.986049,
    0.986339, 0.986630, 0.986922, 0.987176, 0.987431,
    0.987655, 0.987922, 0.988173, 0.988414, 0.988639,
    // 4
    0.988856, 0.989065, 0.989273, 0.989475, 0.989662,
    0.989828, 0.990007, 0.990172, 0.990327, 0.990497,
    0.990645, 0.990797, 0.990981, 0.991135, 0.991272,
    0.991413, 0.991550, 0.991673, 0.991805, 0.991928,
    0.992063, 0.992173, 0.992296, 0.992406, 0.992514,
    0.992632, 0.992733, 0.992837, 0.992954, 0.993046,
    0.993148, 0.993246, 0.993354, 0.993458, 0.993549,
    0.993656, 0.993744, 0.993836, 0.993936, 0.994033,
    0.994134, 0.994222, 0.994307, 0.994413, 0.994495,
    0.994572, 0.994659, 0.994739, 0.994816, 0.994886,
    // 5
    0.994970, 0.995032, 0.995110, 0.995178, 0.995250,
    0.995321, 0.995383, 0.995464, 0.995532, 0.995609,
    0.995674, 0.995750, 0.995821, 0.995889, 0.995952,
    0.996010, 0.996071, 0.996153, 0.996218, 0.996283,
    0.996335, 0.996384, 0.996431, 0.996484, 0.996537,
    0.996597, 0.996655, 0.996701, 0.996745, 0.996802,
    0.996860, 0.996917, 0.996962, 0.997014, 0.997079,
    0.997114, 0.997165, 0.997204, 0.997250, 0.997295,
    0.997335, 0.997379, 0.997418, 0.997454, 0.997488,
    0.997530, 0.997573, 0.997606, 0.997648, 0.997685,
    // 6
    0.997725, 0.997762, 0.997795, 0.997835, 0.997866,
    0.997898, 0.997941, 0.997966, 0.997997, 0.998039,
    0.998065, 0.998104, 0.998128, 0.998153, 0.998179,
    0.998205, 0.998223, 0.998254, 0.998293, 0.998319,
    0.998346, 0.998374, 0.998397, 0.998414, 0.998432,
    0.998456, 0.998482, 0.998511, 0.998532, 0.998553,
    0.998571, 0.998594, 0.998614, 0.998638, 0.998669,
    0.998693, 0.998715, 0.998743, 0.998762, 0.998793,
    0.998812, 0.998834, 0.998857, 0.998872, 0.998888,
    0.998904, 0.998926, 0.998946, 0.998963, 0.998983,
    // 7
    0.999007, 0.999027, 0.999044, 0.999064, 0.999079,
    0.999096, 0.999120, 0.999133, 0.999152, 0.999160,
    0.999174, 0.999188, 0.999206, 0.999221, 0.999234,
    0.999248, 0.999263, 0.999276, 0.999286, 0.999300,
    0.999313, 0.999321, 0.999331, 0.999347, 0.999356,
    0.999369, 0.999381, 0.999394, 0.999402, 0.999415,
    0.999427, 0.999433, 0.999446, 0.999458, 0.999472,
    0.999484, 0.999499, 0.999513, 0.999522, 0.999532,
    0.999540, 0.999550, 0.999559, 0.999567, 0.999574,
    0.999588, 0.999599, 0.999613, 0.999618, 0.999627,
    // 8
    0.999635, 0.999639, 0.999652, 0.999662, 0.999667,
    0.999671, 0.999678, 0.999682, 0.999688, 0.999693,
    0.999698, 0.999701, 0.999706, 0.999711, 0.999718,
    0.999722, 0.999727, 0.999732, 0.999737, 0.999740,
    0.999746, 0.999750, 0.999754, 0.999763, 0.999766,
    0.999769, 0.999774, 0.999780, 0.999784, 0.999788,
    0.999796, 0.999803, 0.999807, 0.999809, 0.999815,
    0.999820, 0.999827, 0.999830, 0.999833, 0.999833,
    0.999836, 0.999839, 0.999842, 0.999845, 0.999850,
    0.999853, 0.999857, 0.999860, 0.999865, 0.999870,
    // 9
    0.999873, 0.999877, 0.999880, 0.999882, 0.999883,
    0.999886, 0.999888, 0.999889, 0.999895, 0.999896,
    0.999897, 0.999901, 0.999902, 0.999905, 0.999907,
    0.999907, 0.999909, 0.999911, 0.999911, 0.999912,
    0.999913, 0.999914, 0.999917, 0.999919, 0.999921,
    0.999923, 0.999927, 0.999929, 0.999931, 0.999933,
    0.999936, 0.999942, 0.999942, 0.999944, 0.999947,
    0.999947, 0.999948, 0.999949, 0.999952, 0.999955,
    0.999957, 0.999957, 0.999961, 0.999962, 0.999963,
    0.999963, 0.999963, 0.999964, 0.999965, 0.999965,
    // 10
    0.999965, 0.999965, 0.999966, 0.999968, 0.999969,
    0.999971, 0.999972, 0.999972, 0.999973, 0.999975,
    0.999975, 0.999975, 0.999975, 0.999975, 0.999975,
    0.999975, 0.999979, 0.999979, 0.999980, 0.999982,
    0.999983, 0.999985, 0.999986, 0.999987, 0.999987,
    0.999988, 0.999989, 0.999989, 0.999989, 0.999989,
    0.999990, 0.999990, 0.999992, 0.999993, 0.999994,
    0.999994, 0.999994, 0.999994, 0.999994, 0.999995,
    0.999995, 0.999995, 0.999996, 0.999996, 0.999996,
    0.999996, 0.999998, 0.999999, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array
    0.0 
  };
  return qpe0;
}

//Currenly the PMT QE info is the same as 20 inch.
G4float* PMT5inch::GetQE(){
  static G4float QE[20] = { 0.00, .0139, .0854, .169, .203, .206, .211, .202,.188, .167, .140, .116, .0806, .0432, .0265, .0146, .00756, .00508, .00158, 0.00};
  return QE;
}
G4float* PMT5inch::GetQEWavelength(){static G4float wavelength[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength;}

G4float  PMT5inch::GetmaxQE(){
  const G4float maxQE = 0.211;
  return maxQE;
}



////////////////////////////////////////////////////////////////////////////////////////////////
// 3 inch (transit time spread of 4ns FWHM @1p.e.)

PMT3inch::PMT3inch(){}
PMT3inch::~PMT3inch(){}

G4String PMT3inch::GetPMTName() {G4String PMTName = "3inch"; return PMTName;}
G4double PMT3inch::GetExposeHeight() {return 20.*mm;}
G4double PMT3inch::GetRadius() {return 40.*mm;}
G4double PMT3inch::GetPMTGlassThickness() {return 0.40*cm;}
G4float PMT3inch::HitTimeSmearing(float Q) { 
  float timingConstant = 1.890; // 4ns FWHM when Q=1.0 
  float timingResolution = 0.33 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 0.58) timingResolution=0.58;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* PMT3inch::Getqpe() //currently uses the same as 20inch
{
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000129, 0.000754, 0.004060, 0.028471,
    // 2
    0.068449, 0.115679, 0.164646, 0.203466, 0.235631,
    0.262351, 0.282064, 0.303341, 0.320618, 0.338317,
    0.357825, 0.371980, 0.385820, 0.398838, 0.413595,
    0.428590, 0.444387, 0.461685, 0.482383, 0.502369,
    0.520779, 0.540011, 0.559293, 0.579354, 0.599337,
    0.619580, 0.639859, 0.659807, 0.679810, 0.699620,
    0.718792, 0.737382, 0.755309, 0.772042, 0.788232,
    0.803316, 0.817861, 0.831148, 0.844339, 0.855532,
    0.866693, 0.876604, 0.886067, 0.894473, 0.902150,
    0.909515, 0.915983, 0.922050, 0.927418, 0.932492,
    // 3
    0.936951, 0.940941, 0.944660, 0.948004, 0.951090,
    0.953833, 0.956576, 0.958886, 0.961134, 0.963116,
    0.964930, 0.966562, 0.968008, 0.969424, 0.970687,
    0.971783, 0.972867, 0.973903, 0.974906, 0.975784,
    0.976632, 0.977438, 0.978190, 0.978891, 0.979543,
    0.980124, 0.980666, 0.981255, 0.981770, 0.982227,
    0.982701, 0.983146, 0.983566, 0.983975, 0.984357,
    0.984713, 0.985094, 0.985404, 0.985739, 0.986049,
    0.986339, 0.986630, 0.986922, 0.987176, 0.987431,
    0.987655, 0.987922, 0.988173, 0.988414, 0.988639,
    // 4
    0.988856, 0.989065, 0.989273, 0.989475, 0.989662,
    0.989828, 0.990007, 0.990172, 0.990327, 0.990497,
    0.990645, 0.990797, 0.990981, 0.991135, 0.991272,
    0.991413, 0.991550, 0.991673, 0.991805, 0.991928,
    0.992063, 0.992173, 0.992296, 0.992406, 0.992514,
    0.992632, 0.992733, 0.992837, 0.992954, 0.993046,
    0.993148, 0.993246, 0.993354, 0.993458, 0.993549,
    0.993656, 0.993744, 0.993836, 0.993936, 0.994033,
    0.994134, 0.994222, 0.994307, 0.994413, 0.994495,
    0.994572, 0.994659, 0.994739, 0.994816, 0.994886,
    // 5
    0.994970, 0.995032, 0.995110, 0.995178, 0.995250,
    0.995321, 0.995383, 0.995464, 0.995532, 0.995609,
    0.995674, 0.995750, 0.995821, 0.995889, 0.995952,
    0.996010, 0.996071, 0.996153, 0.996218, 0.996283,
    0.996335, 0.996384, 0.996431, 0.996484, 0.996537,
    0.996597, 0.996655, 0.996701, 0.996745, 0.996802,
    0.996860, 0.996917, 0.996962, 0.997014, 0.997079,
    0.997114, 0.997165, 0.997204, 0.997250, 0.997295,
    0.997335, 0.997379, 0.997418, 0.997454, 0.997488,
    0.997530, 0.997573, 0.997606, 0.997648, 0.997685,
    // 6
    0.997725, 0.997762, 0.997795, 0.997835, 0.997866,
    0.997898, 0.997941, 0.997966, 0.997997, 0.998039,
    0.998065, 0.998104, 0.998128, 0.998153, 0.998179,
    0.998205, 0.998223, 0.998254, 0.998293, 0.998319,
    0.998346, 0.998374, 0.998397, 0.998414, 0.998432,
    0.998456, 0.998482, 0.998511, 0.998532, 0.998553,
    0.998571, 0.998594, 0.998614, 0.998638, 0.998669,
    0.998693, 0.998715, 0.998743, 0.998762, 0.998793,
    0.998812, 0.998834, 0.998857, 0.998872, 0.998888,
    0.998904, 0.998926, 0.998946, 0.998963, 0.998983,
    // 7
    0.999007, 0.999027, 0.999044, 0.999064, 0.999079,
    0.999096, 0.999120, 0.999133, 0.999152, 0.999160,
    0.999174, 0.999188, 0.999206, 0.999221, 0.999234,
    0.999248, 0.999263, 0.999276, 0.999286, 0.999300,
    0.999313, 0.999321, 0.999331, 0.999347, 0.999356,
    0.999369, 0.999381, 0.999394, 0.999402, 0.999415,
    0.999427, 0.999433, 0.999446, 0.999458, 0.999472,
    0.999484, 0.999499, 0.999513, 0.999522, 0.999532,
    0.999540, 0.999550, 0.999559, 0.999567, 0.999574,
    0.999588, 0.999599, 0.999613, 0.999618, 0.999627,
    // 8
    0.999635, 0.999639, 0.999652, 0.999662, 0.999667,
    0.999671, 0.999678, 0.999682, 0.999688, 0.999693,
    0.999698, 0.999701, 0.999706, 0.999711, 0.999718,
    0.999722, 0.999727, 0.999732, 0.999737, 0.999740,
    0.999746, 0.999750, 0.999754, 0.999763, 0.999766,
    0.999769, 0.999774, 0.999780, 0.999784, 0.999788,
    0.999796, 0.999803, 0.999807, 0.999809, 0.999815,
    0.999820, 0.999827, 0.999830, 0.999833, 0.999833,
    0.999836, 0.999839, 0.999842, 0.999845, 0.999850,
    0.999853, 0.999857, 0.999860, 0.999865, 0.999870,
    // 9
    0.999873, 0.999877, 0.999880, 0.999882, 0.999883,
    0.999886, 0.999888, 0.999889, 0.999895, 0.999896,
    0.999897, 0.999901, 0.999902, 0.999905, 0.999907,
    0.999907, 0.999909, 0.999911, 0.999911, 0.999912,
    0.999913, 0.999914, 0.999917, 0.999919, 0.999921,
    0.999923, 0.999927, 0.999929, 0.999931, 0.999933,
    0.999936, 0.999942, 0.999942, 0.999944, 0.999947,
    0.999947, 0.999948, 0.999949, 0.999952, 0.999955,
    0.999957, 0.999957, 0.999961, 0.999962, 0.999963,
    0.999963, 0.999963, 0.999964, 0.999965, 0.999965,
    // 10
    0.999965, 0.999965, 0.999966, 0.999968, 0.999969,
    0.999971, 0.999972, 0.999972, 0.999973, 0.999975,
    0.999975, 0.999975, 0.999975, 0.999975, 0.999975,
    0.999975, 0.999979, 0.999979, 0.999980, 0.999982,
    0.999983, 0.999985, 0.999986, 0.999987, 0.999987,
    0.999988, 0.999989, 0.999989, 0.999989, 0.999989,
    0.999990, 0.999990, 0.999992, 0.999993, 0.999994,
    0.999994, 0.999994, 0.999994, 0.999994, 0.999995,
    0.999995, 0.999995, 0.999996, 0.999996, 0.999996,
    0.999996, 0.999998, 0.999999, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array
    0.0 
  };
  return qpe0;
}

//Currenly the PMT QE info is the same as 20 inch.
G4float* PMT3inch::GetQE(){
  static G4float QE[20] = { 0.00, .0139, .0854, .169, .203, .206, .211, .202,.188, .167, .140, .116, .0806, .0432, .0265, .0146, .00756, .00508, .00158, 0.00};
  return QE;
}
G4float* PMT3inch::GetQEWavelength(){static G4float wavelength[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength;}

G4float  PMT3inch::GetmaxQE(){
  const G4float maxQE = 0.211;
  return maxQE;
}



////////////////////////////////////////////////////////////////////////////////////////////////
// 3 inch, better timing resolution (transit time spread of 2.5ns FWHM @1p.e.)

PMT3inchGT::PMT3inchGT(){}
PMT3inchGT::~PMT3inchGT(){}

G4String PMT3inchGT::GetPMTName() {G4String PMTName = "3inchGT"; return PMTName;}
G4double PMT3inchGT::GetExposeHeight() {return 20.*mm;}
G4double PMT3inchGT::GetRadius() {return 40.*mm;}
G4double PMT3inchGT::GetPMTGlassThickness() {return 0.40*cm;}
G4float PMT3inchGT::HitTimeSmearing(float Q) { 
  float timingConstant = 0.535; // 2.5ns FWHM when Q=1.0
  float timingResolution = 0.33 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 0.58) timingResolution=0.58;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* PMT3inchGT::Getqpe() //currently uses the same as 20inch
{
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000129, 0.000754, 0.004060, 0.028471,
    // 2
    0.068449, 0.115679, 0.164646, 0.203466, 0.235631,
    0.262351, 0.282064, 0.303341, 0.320618, 0.338317,
    0.357825, 0.371980, 0.385820, 0.398838, 0.413595,
    0.428590, 0.444387, 0.461685, 0.482383, 0.502369,
    0.520779, 0.540011, 0.559293, 0.579354, 0.599337,
    0.619580, 0.639859, 0.659807, 0.679810, 0.699620,
    0.718792, 0.737382, 0.755309, 0.772042, 0.788232,
    0.803316, 0.817861, 0.831148, 0.844339, 0.855532,
    0.866693, 0.876604, 0.886067, 0.894473, 0.902150,
    0.909515, 0.915983, 0.922050, 0.927418, 0.932492,
    // 3
    0.936951, 0.940941, 0.944660, 0.948004, 0.951090,
    0.953833, 0.956576, 0.958886, 0.961134, 0.963116,
    0.964930, 0.966562, 0.968008, 0.969424, 0.970687,
    0.971783, 0.972867, 0.973903, 0.974906, 0.975784,
    0.976632, 0.977438, 0.978190, 0.978891, 0.979543,
    0.980124, 0.980666, 0.981255, 0.981770, 0.982227,
    0.982701, 0.983146, 0.983566, 0.983975, 0.984357,
    0.984713, 0.985094, 0.985404, 0.985739, 0.986049,
    0.986339, 0.986630, 0.986922, 0.987176, 0.987431,
    0.987655, 0.987922, 0.988173, 0.988414, 0.988639,
    // 4
    0.988856, 0.989065, 0.989273, 0.989475, 0.989662,
    0.989828, 0.990007, 0.990172, 0.990327, 0.990497,
    0.990645, 0.990797, 0.990981, 0.991135, 0.991272,
    0.991413, 0.991550, 0.991673, 0.991805, 0.991928,
    0.992063, 0.992173, 0.992296, 0.992406, 0.992514,
    0.992632, 0.992733, 0.992837, 0.992954, 0.993046,
    0.993148, 0.993246, 0.993354, 0.993458, 0.993549,
    0.993656, 0.993744, 0.993836, 0.993936, 0.994033,
    0.994134, 0.994222, 0.994307, 0.994413, 0.994495,
    0.994572, 0.994659, 0.994739, 0.994816, 0.994886,
    // 5
    0.994970, 0.995032, 0.995110, 0.995178, 0.995250,
    0.995321, 0.995383, 0.995464, 0.995532, 0.995609,
    0.995674, 0.995750, 0.995821, 0.995889, 0.995952,
    0.996010, 0.996071, 0.996153, 0.996218, 0.996283,
    0.996335, 0.996384, 0.996431, 0.996484, 0.996537,
    0.996597, 0.996655, 0.996701, 0.996745, 0.996802,
    0.996860, 0.996917, 0.996962, 0.997014, 0.997079,
    0.997114, 0.997165, 0.997204, 0.997250, 0.997295,
    0.997335, 0.997379, 0.997418, 0.997454, 0.997488,
    0.997530, 0.997573, 0.997606, 0.997648, 0.997685,
    // 6
    0.997725, 0.997762, 0.997795, 0.997835, 0.997866,
    0.997898, 0.997941, 0.997966, 0.997997, 0.998039,
    0.998065, 0.998104, 0.998128, 0.998153, 0.998179,
    0.998205, 0.998223, 0.998254, 0.998293, 0.998319,
    0.998346, 0.998374, 0.998397, 0.998414, 0.998432,
    0.998456, 0.998482, 0.998511, 0.998532, 0.998553,
    0.998571, 0.998594, 0.998614, 0.998638, 0.998669,
    0.998693, 0.998715, 0.998743, 0.998762, 0.998793,
    0.998812, 0.998834, 0.998857, 0.998872, 0.998888,
    0.998904, 0.998926, 0.998946, 0.998963, 0.998983,
    // 7
    0.999007, 0.999027, 0.999044, 0.999064, 0.999079,
    0.999096, 0.999120, 0.999133, 0.999152, 0.999160,
    0.999174, 0.999188, 0.999206, 0.999221, 0.999234,
    0.999248, 0.999263, 0.999276, 0.999286, 0.999300,
    0.999313, 0.999321, 0.999331, 0.999347, 0.999356,
    0.999369, 0.999381, 0.999394, 0.999402, 0.999415,
    0.999427, 0.999433, 0.999446, 0.999458, 0.999472,
    0.999484, 0.999499, 0.999513, 0.999522, 0.999532,
    0.999540, 0.999550, 0.999559, 0.999567, 0.999574,
    0.999588, 0.999599, 0.999613, 0.999618, 0.999627,
    // 8
    0.999635, 0.999639, 0.999652, 0.999662, 0.999667,
    0.999671, 0.999678, 0.999682, 0.999688, 0.999693,
    0.999698, 0.999701, 0.999706, 0.999711, 0.999718,
    0.999722, 0.999727, 0.999732, 0.999737, 0.999740,
    0.999746, 0.999750, 0.999754, 0.999763, 0.999766,
    0.999769, 0.999774, 0.999780, 0.999784, 0.999788,
    0.999796, 0.999803, 0.999807, 0.999809, 0.999815,
    0.999820, 0.999827, 0.999830, 0.999833, 0.999833,
    0.999836, 0.999839, 0.999842, 0.999845, 0.999850,
    0.999853, 0.999857, 0.999860, 0.999865, 0.999870,
    // 9
    0.999873, 0.999877, 0.999880, 0.999882, 0.999883,
    0.999886, 0.999888, 0.999889, 0.999895, 0.999896,
    0.999897, 0.999901, 0.999902, 0.999905, 0.999907,
    0.999907, 0.999909, 0.999911, 0.999911, 0.999912,
    0.999913, 0.999914, 0.999917, 0.999919, 0.999921,
    0.999923, 0.999927, 0.999929, 0.999931, 0.999933,
    0.999936, 0.999942, 0.999942, 0.999944, 0.999947,
    0.999947, 0.999948, 0.999949, 0.999952, 0.999955,
    0.999957, 0.999957, 0.999961, 0.999962, 0.999963,
    0.999963, 0.999963, 0.999964, 0.999965, 0.999965,
    // 10
    0.999965, 0.999965, 0.999966, 0.999968, 0.999969,
    0.999971, 0.999972, 0.999972, 0.999973, 0.999975,
    0.999975, 0.999975, 0.999975, 0.999975, 0.999975,
    0.999975, 0.999979, 0.999979, 0.999980, 0.999982,
    0.999983, 0.999985, 0.999986, 0.999987, 0.999987,
    0.999988, 0.999989, 0.999989, 0.999989, 0.999989,
    0.999990, 0.999990, 0.999992, 0.999993, 0.999994,
    0.999994, 0.999994, 0.999994, 0.999994, 0.999995,
    0.999995, 0.999995, 0.999996, 0.999996, 0.999996,
    0.999996, 0.999998, 0.999999, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array
    0.0 
  };
  return qpe0;
}

//Currenly the PMT QE info is the same as 20 inch.
G4float* PMT3inchGT::GetQE(){
  static G4float QE[20] = { 0.00, .0139, .0854, .169, .203, .206, .211, .202,.188, .167, .140, .116, .0806, .0432, .0265, .0146, .00756, .00508, .00158, 0.00};
  return QE;
}
G4float* PMT3inchGT::GetQEWavelength(){static G4float wavelength[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength;}

G4float  PMT3inchGT::GetmaxQE(){
  const G4float maxQE = 0.211;
  return maxQE;
}



////////////////////////////////////////////////////////////////////////////////////////////////
// TF: Hacked 20 inch HQE Box and Line PMT and adapted to 3inch R12199-02
//     Will update it later with real properties and add more 3in PMTs after reorganizing this
//     file first.
//     Once I'm updating these values, immediately test with grabbing from a (txt/ROOT) file as example
//     for cleaning up the rest. Then making these objects of a class and then making them Enums.
//
//   Use full area of half sphere, instead of photocathode area from drawings (dotted line) as we do for 20"
//   next: use effective CE/QE as function of theta to correctly simulate edge effect (increase in QE, decrease in CE)
//
PMT3inchR12199_02::PMT3inchR12199_02(){}
PMT3inchR12199_02::~PMT3inchR12199_02(){}
G4String PMT3inchR12199_02::GetPMTName() {G4String PMTName = "PMT3inchR12199_02"; return PMTName;}
G4double PMT3inchR12199_02::GetExposeHeight() {return .0200*m;}  //from TechSheet for 3in (only photocathode would be 15.3mm h, for a radius as photocathode of 36 mm)
G4double PMT3inchR12199_02::GetRadius() {return .0400*m;}   //radius at z = exposeheight of photocathode. In ConstructPMT, we use sphereRadius for the radius of curvature
G4double PMT3inchR12199_02::GetPMTGlassThickness() {return 0.4*cm;}
float PMT3inchR12199_02::HitTimeSmearing(float Q) {
  G4float sig_param[4]={0.6314,0.06260,0.5711,23.96};
  G4float lambda_param[2]={0.4113,0.07827};
  G4float sigma_lowcharge = sig_param[0]*(exp(-sig_param[1]*Q)+sig_param[2]);
  G4float highcharge_param[2];
  highcharge_param[0]=2*sig_param[0]*sig_param[1]*sig_param[3]*sqrt(sig_param[3])*exp(-sig_param[1]*sig_param[3]);
  highcharge_param[1]=sig_param[0]*((1-2*sig_param[1]*sig_param[3])*exp(-sig_param[1]*sig_param[3])+sig_param[2]);
  G4float sigma_highcharge = highcharge_param[0]/sqrt(Q)+highcharge_param[1];
  G4float sigma = sigma_lowcharge*(Q<sig_param[3])+sigma_highcharge*(Q>sig_param[3]);
  G4float lambda = lambda_param[0]+lambda_param[1]*Q;
  G4float Smearing_factor = G4RandGauss::shoot(-0.2,sigma)-1/lambda*log(1-G4UniformRand());
  return Smearing_factor;
}
G4float* PMT3inchR12199_02::Getqpe()
{
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000001, 0.000001, 0.000002, 0.000004,
    0.000008, 0.000014, 0.000025, 0.000044, 0.000486,
    // 2
    0.007195, 0.019406, 0.031920, 0.044503, 0.057189,
    0.070020, 0.083060, 0.096388, 0.110108, 0.124351,
    0.139276, 0.155072, 0.171956, 0.190167, 0.209961,
    0.231594, 0.255310, 0.281319, 0.309777, 0.340762,
    0.374259, 0.410142, 0.448167, 0.487976, 0.529101,
    0.570993, 0.613041, 0.654608, 0.695067, 0.733833,
    0.770390, 0.804317, 0.835304, 0.863151, 0.887777,
    0.909203, 0.927543, 0.942987, 0.955778, 0.966198,
    0.974543, 0.981116, 0.986205, 0.990078, 0.992974,
    0.995104, 0.996642, 0.997734, 0.998495, 0.999017,
    // 3
    0.999369, 0.999601, 0.999752, 0.999848, 0.999909,
    0.999946, 0.999969, 0.999982, 0.999990, 0.999994,
    0.999997, 0.999998, 0.999999, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 4
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 5
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 6
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 7
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 8
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 9
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 10
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array                        
    0.0  };
  return qpe0;
}
// TF: Used WebPlotDigitizer on Fig.2a from VLVnT13 proceedings, red curve (average)
G4float* PMT3inchR12199_02::GetQEWavelength(){
  //TEST: make QE same!!
  //static G4float wavelength_value[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  static G4float wavelength_value[21] = { 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660., 680., 700.};
  return wavelength_value;
}
G4float* PMT3inchR12199_02::GetQE(){
  G4float correctionFactor = 1./0.73;//Correction factor added in July 2015 to scale the output of B&L PDs to 2.27 times the 20" PMTS based on Hamamatsu simulation. This was done in Pull Request #98 and will be removed once a more permanent solution is found.
  // TF: While the main reason is the 20" SK PMT, this correction factor has been applied
  // to the B&L PMT. Therefore all PMTs have to corrected in a similar way, unfortunately.
static G4float QE[21] =
  { .0787*correctionFactor, .1838*correctionFactor, .2401*correctionFactor, .2521*correctionFactor, .2695*correctionFactor, .2676*correctionFactor, .2593*correctionFactor, .2472*correctionFactor, .2276*correctionFactor,
    .1970*correctionFactor,  .1777*correctionFactor, .1547*correctionFactor, .1033*correctionFactor, .0727*correctionFactor, .0587*correctionFactor, .0470*correctionFactor, .0372*correctionFactor, .0285*correctionFactor, .0220*correctionFactor, .0130*correctionFactor, .0084*correctionFactor};
/* TEST: make QE the same!! 
  static G4float QE[20] =
    { 0.00*correctionFactor, .0008*correctionFactor, .1255*correctionFactor, .254962*correctionFactor, .2930*correctionFactor, .3127*correctionFactor, .3130*correctionFactor, .2994*correctionFactor, .2791*correctionFactor, .2491*correctionFactor,
    .2070*correctionFactor,  .1758*correctionFactor, .1384*correctionFactor, .0779*correctionFactor, .0473*correctionFactor, .0288*correctionFactor, .0149*correctionFactor, .0062*correctionFactor, .0002*correctionFactor, .0001*correctionFactor}; 
*/
 return QE;
}
G4float PMT3inchR12199_02::GetmaxQE(){
  G4float correctionFactor = 1./0.73;//Correction factor added in July 2015 to scale the output of B&L PDs to 2.27 times the 20" PMTS based on Hamamatsu simulation. This was done in Pull Request #98 and will be removed once a more permanent solution is found.
  // TEST: make QE the same!!
  const G4float maxQE = 0.271*correctionFactor; //red curve from VLVnT13 proc on R12199-02.
  //const G4float maxQE = 0.315*correctionFactor;
  return maxQE;
}
G4float* PMT3inchR12199_02::GetCollectionEfficiencyArray(){  
  static G4float CE[10] = { 95., 95., 95., 95., 95., 95., 95., 95., 95., 95.};
  return CE;
}



////////////////////////////////////////////////////////////////////////////////////////////////
// 10 inch

PMT10inch::PMT10inch(){}
PMT10inch::~PMT10inch(){}

G4String PMT10inch::GetPMTName() {G4String PMTName = "10inch"; return PMTName;}
G4double PMT10inch::GetExposeHeight() {return 117.*mm;}
G4double PMT10inch::GetRadius() {return 127.*mm;}
G4double PMT10inch::GetPMTGlassThickness() {return 0.55*cm;}
float PMT10inch::HitTimeSmearing(float Q) { 
  float timingConstant = 2.0; 
  float timingResolution = 0.33 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 0.58) timingResolution=0.58;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* PMT10inch::Getqpe() //currently uses the same as 20inch
   {
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000129, 0.000754, 0.004060, 0.028471,
    // 2
    0.068449, 0.115679, 0.164646, 0.203466, 0.235631,
    0.262351, 0.282064, 0.303341, 0.320618, 0.338317,
    0.357825, 0.371980, 0.385820, 0.398838, 0.413595,
    0.428590, 0.444387, 0.461685, 0.482383, 0.502369,
    0.520779, 0.540011, 0.559293, 0.579354, 0.599337,
    0.619580, 0.639859, 0.659807, 0.679810, 0.699620,
    0.718792, 0.737382, 0.755309, 0.772042, 0.788232,
    0.803316, 0.817861, 0.831148, 0.844339, 0.855532,
    0.866693, 0.876604, 0.886067, 0.894473, 0.902150,
    0.909515, 0.915983, 0.922050, 0.927418, 0.932492,
    // 3
    0.936951, 0.940941, 0.944660, 0.948004, 0.951090,
    0.953833, 0.956576, 0.958886, 0.961134, 0.963116,
    0.964930, 0.966562, 0.968008, 0.969424, 0.970687,
    0.971783, 0.972867, 0.973903, 0.974906, 0.975784,
    0.976632, 0.977438, 0.978190, 0.978891, 0.979543,
    0.980124, 0.980666, 0.981255, 0.981770, 0.982227,
    0.982701, 0.983146, 0.983566, 0.983975, 0.984357,
    0.984713, 0.985094, 0.985404, 0.985739, 0.986049,
    0.986339, 0.986630, 0.986922, 0.987176, 0.987431,
    0.987655, 0.987922, 0.988173, 0.988414, 0.988639,
    // 4
    0.988856, 0.989065, 0.989273, 0.989475, 0.989662,
    0.989828, 0.990007, 0.990172, 0.990327, 0.990497,
    0.990645, 0.990797, 0.990981, 0.991135, 0.991272,
    0.991413, 0.991550, 0.991673, 0.991805, 0.991928,
    0.992063, 0.992173, 0.992296, 0.992406, 0.992514,
    0.992632, 0.992733, 0.992837, 0.992954, 0.993046,
    0.993148, 0.993246, 0.993354, 0.993458, 0.993549,
    0.993656, 0.993744, 0.993836, 0.993936, 0.994033,
    0.994134, 0.994222, 0.994307, 0.994413, 0.994495,
    0.994572, 0.994659, 0.994739, 0.994816, 0.994886,
    // 5
    0.994970, 0.995032, 0.995110, 0.995178, 0.995250,
    0.995321, 0.995383, 0.995464, 0.995532, 0.995609,
    0.995674, 0.995750, 0.995821, 0.995889, 0.995952,
    0.996010, 0.996071, 0.996153, 0.996218, 0.996283,
    0.996335, 0.996384, 0.996431, 0.996484, 0.996537,
    0.996597, 0.996655, 0.996701, 0.996745, 0.996802,
    0.996860, 0.996917, 0.996962, 0.997014, 0.997079,
    0.997114, 0.997165, 0.997204, 0.997250, 0.997295,
    0.997335, 0.997379, 0.997418, 0.997454, 0.997488,
    0.997530, 0.997573, 0.997606, 0.997648, 0.997685,
    // 6
    0.997725, 0.997762, 0.997795, 0.997835, 0.997866,
    0.997898, 0.997941, 0.997966, 0.997997, 0.998039,
    0.998065, 0.998104, 0.998128, 0.998153, 0.998179,
    0.998205, 0.998223, 0.998254, 0.998293, 0.998319,
    0.998346, 0.998374, 0.998397, 0.998414, 0.998432,
    0.998456, 0.998482, 0.998511, 0.998532, 0.998553,
    0.998571, 0.998594, 0.998614, 0.998638, 0.998669,
    0.998693, 0.998715, 0.998743, 0.998762, 0.998793,
    0.998812, 0.998834, 0.998857, 0.998872, 0.998888,
    0.998904, 0.998926, 0.998946, 0.998963, 0.998983,
    // 7
    0.999007, 0.999027, 0.999044, 0.999064, 0.999079,
    0.999096, 0.999120, 0.999133, 0.999152, 0.999160,
    0.999174, 0.999188, 0.999206, 0.999221, 0.999234,
    0.999248, 0.999263, 0.999276, 0.999286, 0.999300,
    0.999313, 0.999321, 0.999331, 0.999347, 0.999356,
    0.999369, 0.999381, 0.999394, 0.999402, 0.999415,
    0.999427, 0.999433, 0.999446, 0.999458, 0.999472,
    0.999484, 0.999499, 0.999513, 0.999522, 0.999532,
    0.999540, 0.999550, 0.999559, 0.999567, 0.999574,
    0.999588, 0.999599, 0.999613, 0.999618, 0.999627,
    // 8
    0.999635, 0.999639, 0.999652, 0.999662, 0.999667,
    0.999671, 0.999678, 0.999682, 0.999688, 0.999693,
    0.999698, 0.999701, 0.999706, 0.999711, 0.999718,
    0.999722, 0.999727, 0.999732, 0.999737, 0.999740,
    0.999746, 0.999750, 0.999754, 0.999763, 0.999766,
    0.999769, 0.999774, 0.999780, 0.999784, 0.999788,
    0.999796, 0.999803, 0.999807, 0.999809, 0.999815,
    0.999820, 0.999827, 0.999830, 0.999833, 0.999833,
    0.999836, 0.999839, 0.999842, 0.999845, 0.999850,
    0.999853, 0.999857, 0.999860, 0.999865, 0.999870,
    // 9
    0.999873, 0.999877, 0.999880, 0.999882, 0.999883,
    0.999886, 0.999888, 0.999889, 0.999895, 0.999896,
    0.999897, 0.999901, 0.999902, 0.999905, 0.999907,
    0.999907, 0.999909, 0.999911, 0.999911, 0.999912,
    0.999913, 0.999914, 0.999917, 0.999919, 0.999921,
    0.999923, 0.999927, 0.999929, 0.999931, 0.999933,
    0.999936, 0.999942, 0.999942, 0.999944, 0.999947,
    0.999947, 0.999948, 0.999949, 0.999952, 0.999955,
    0.999957, 0.999957, 0.999961, 0.999962, 0.999963,
    0.999963, 0.999963, 0.999964, 0.999965, 0.999965,
    // 10
    0.999965, 0.999965, 0.999966, 0.999968, 0.999969,
    0.999971, 0.999972, 0.999972, 0.999973, 0.999975,
    0.999975, 0.999975, 0.999975, 0.999975, 0.999975,
    0.999975, 0.999979, 0.999979, 0.999980, 0.999982,
    0.999983, 0.999985, 0.999986, 0.999987, 0.999987,
    0.999988, 0.999989, 0.999989, 0.999989, 0.999989,
    0.999990, 0.999990, 0.999992, 0.999993, 0.999994,
    0.999994, 0.999994, 0.999994, 0.999994, 0.999995,
    0.999995, 0.999995, 0.999996, 0.999996, 0.999996,
    0.999996, 0.999998, 0.999999, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array
    0.0 
  };
   return qpe0;
  }

G4float* PMT10inch::GetQE(){
 static G4float QE[20] =
   { 0.00, .0375, .13, .195, .22, .23, .24, .24, .225, .205,
     .18, .16, .14, .085, .065, .05, .035, .02, .005, 0.0};
 return QE;
}
G4float* PMT10inch::GetQEWavelength(){static G4float wavelength[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength;}

G4float  PMT10inch::GetmaxQE(){
  const G4float maxQE = 0.24;
  return maxQE;
}


////////////////////////////////////////////////////////////////////////////////////////////////
// 10 inch HQE

PMT10inchHQE::PMT10inchHQE() {}
PMT10inchHQE::~PMT10inchHQE(){}

G4String PMT10inchHQE::GetPMTName() {G4String PMTName = "10inch"; return PMTName;}
G4double PMT10inchHQE::GetExposeHeight() {return 117.*mm;}
G4double PMT10inchHQE::GetRadius() {return 127.*mm;}
G4double PMT10inchHQE::GetPMTGlassThickness() {return 0.55*cm;}
G4float PMT10inchHQE::HitTimeSmearing(float Q) {
  float timingConstant = 2.0; 
  float timingResolution = 0.33 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 0.58) timingResolution=0.58;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* PMT10inchHQE::Getqpe() //currently uses the same as 20inch
   {
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000129, 0.000754, 0.004060, 0.028471,
    // 2
    0.068449, 0.115679, 0.164646, 0.203466, 0.235631,
    0.262351, 0.282064, 0.303341, 0.320618, 0.338317,
    0.357825, 0.371980, 0.385820, 0.398838, 0.413595,
    0.428590, 0.444387, 0.461685, 0.482383, 0.502369,
    0.520779, 0.540011, 0.559293, 0.579354, 0.599337,
    0.619580, 0.639859, 0.659807, 0.679810, 0.699620,
    0.718792, 0.737382, 0.755309, 0.772042, 0.788232,
    0.803316, 0.817861, 0.831148, 0.844339, 0.855532,
    0.866693, 0.876604, 0.886067, 0.894473, 0.902150,
    0.909515, 0.915983, 0.922050, 0.927418, 0.932492,
    // 3
    0.936951, 0.940941, 0.944660, 0.948004, 0.951090,
    0.953833, 0.956576, 0.958886, 0.961134, 0.963116,
    0.964930, 0.966562, 0.968008, 0.969424, 0.970687,
    0.971783, 0.972867, 0.973903, 0.974906, 0.975784,
    0.976632, 0.977438, 0.978190, 0.978891, 0.979543,
    0.980124, 0.980666, 0.981255, 0.981770, 0.982227,
    0.982701, 0.983146, 0.983566, 0.983975, 0.984357,
    0.984713, 0.985094, 0.985404, 0.985739, 0.986049,
    0.986339, 0.986630, 0.986922, 0.987176, 0.987431,
    0.987655, 0.987922, 0.988173, 0.988414, 0.988639,
    // 4
    0.988856, 0.989065, 0.989273, 0.989475, 0.989662,
    0.989828, 0.990007, 0.990172, 0.990327, 0.990497,
    0.990645, 0.990797, 0.990981, 0.991135, 0.991272,
    0.991413, 0.991550, 0.991673, 0.991805, 0.991928,
    0.992063, 0.992173, 0.992296, 0.992406, 0.992514,
    0.992632, 0.992733, 0.992837, 0.992954, 0.993046,
    0.993148, 0.993246, 0.993354, 0.993458, 0.993549,
    0.993656, 0.993744, 0.993836, 0.993936, 0.994033,
    0.994134, 0.994222, 0.994307, 0.994413, 0.994495,
    0.994572, 0.994659, 0.994739, 0.994816, 0.994886,
    // 5
    0.994970, 0.995032, 0.995110, 0.995178, 0.995250,
    0.995321, 0.995383, 0.995464, 0.995532, 0.995609,
    0.995674, 0.995750, 0.995821, 0.995889, 0.995952,
    0.996010, 0.996071, 0.996153, 0.996218, 0.996283,
    0.996335, 0.996384, 0.996431, 0.996484, 0.996537,
    0.996597, 0.996655, 0.996701, 0.996745, 0.996802,
    0.996860, 0.996917, 0.996962, 0.997014, 0.997079,
    0.997114, 0.997165, 0.997204, 0.997250, 0.997295,
    0.997335, 0.997379, 0.997418, 0.997454, 0.997488,
    0.997530, 0.997573, 0.997606, 0.997648, 0.997685,
    // 6
    0.997725, 0.997762, 0.997795, 0.997835, 0.997866,
    0.997898, 0.997941, 0.997966, 0.997997, 0.998039,
    0.998065, 0.998104, 0.998128, 0.998153, 0.998179,
    0.998205, 0.998223, 0.998254, 0.998293, 0.998319,
    0.998346, 0.998374, 0.998397, 0.998414, 0.998432,
    0.998456, 0.998482, 0.998511, 0.998532, 0.998553,
    0.998571, 0.998594, 0.998614, 0.998638, 0.998669,
    0.998693, 0.998715, 0.998743, 0.998762, 0.998793,
    0.998812, 0.998834, 0.998857, 0.998872, 0.998888,
    0.998904, 0.998926, 0.998946, 0.998963, 0.998983,
    // 7
    0.999007, 0.999027, 0.999044, 0.999064, 0.999079,
    0.999096, 0.999120, 0.999133, 0.999152, 0.999160,
    0.999174, 0.999188, 0.999206, 0.999221, 0.999234,
    0.999248, 0.999263, 0.999276, 0.999286, 0.999300,
    0.999313, 0.999321, 0.999331, 0.999347, 0.999356,
    0.999369, 0.999381, 0.999394, 0.999402, 0.999415,
    0.999427, 0.999433, 0.999446, 0.999458, 0.999472,
    0.999484, 0.999499, 0.999513, 0.999522, 0.999532,
    0.999540, 0.999550, 0.999559, 0.999567, 0.999574,
    0.999588, 0.999599, 0.999613, 0.999618, 0.999627,
    // 8
    0.999635, 0.999639, 0.999652, 0.999662, 0.999667,
    0.999671, 0.999678, 0.999682, 0.999688, 0.999693,
    0.999698, 0.999701, 0.999706, 0.999711, 0.999718,
    0.999722, 0.999727, 0.999732, 0.999737, 0.999740,
    0.999746, 0.999750, 0.999754, 0.999763, 0.999766,
    0.999769, 0.999774, 0.999780, 0.999784, 0.999788,
    0.999796, 0.999803, 0.999807, 0.999809, 0.999815,
    0.999820, 0.999827, 0.999830, 0.999833, 0.999833,
    0.999836, 0.999839, 0.999842, 0.999845, 0.999850,
    0.999853, 0.999857, 0.999860, 0.999865, 0.999870,
    // 9
    0.999873, 0.999877, 0.999880, 0.999882, 0.999883,
    0.999886, 0.999888, 0.999889, 0.999895, 0.999896,
    0.999897, 0.999901, 0.999902, 0.999905, 0.999907,
    0.999907, 0.999909, 0.999911, 0.999911, 0.999912,
    0.999913, 0.999914, 0.999917, 0.999919, 0.999921,
    0.999923, 0.999927, 0.999929, 0.999931, 0.999933,
    0.999936, 0.999942, 0.999942, 0.999944, 0.999947,
    0.999947, 0.999948, 0.999949, 0.999952, 0.999955,
    0.999957, 0.999957, 0.999961, 0.999962, 0.999963,
    0.999963, 0.999963, 0.999964, 0.999965, 0.999965,
    // 10
    0.999965, 0.999965, 0.999966, 0.999968, 0.999969,
    0.999971, 0.999972, 0.999972, 0.999973, 0.999975,
    0.999975, 0.999975, 0.999975, 0.999975, 0.999975,
    0.999975, 0.999979, 0.999979, 0.999980, 0.999982,
    0.999983, 0.999985, 0.999986, 0.999987, 0.999987,
    0.999988, 0.999989, 0.999989, 0.999989, 0.999989,
    0.999990, 0.999990, 0.999992, 0.999993, 0.999994,
    0.999994, 0.999994, 0.999994, 0.999994, 0.999995,
    0.999995, 0.999995, 0.999996, 0.999996, 0.999996,
    0.999996, 0.999998, 0.999999, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array
    0.0 
  };
   return qpe0;
  }

G4float* PMT10inchHQE::GetQE(){
 static G4float QE[20] =
   { 0.00, .0502, .2017, .2933, .3306, .3396, .3320, .3168, .2915, .2655, 
      .2268,  .1971, .1641, .1102, .0727, .0499, .0323, .0178, .0061, 0.00};
 return QE;
}
G4float* PMT10inchHQE::GetQEWavelength(){static G4float wavelength[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength;}

G4float  PMT10inchHQE::GetmaxQE(){
  const G4float maxQE = 0.3396;
  return maxQE;
}


////////////////////////////////////////////////////////////////////////////////////////////////
// 12 inch HQE

PMT12inchHQE::PMT12inchHQE(){}
PMT12inchHQE::~PMT12inchHQE(){}

G4String PMT12inchHQE::GetPMTName() {G4String PMTName = "12inch"; return PMTName;}
G4double PMT12inchHQE::GetExposeHeight() {return 118.*mm;}
G4double PMT12inchHQE::GetRadius() {return 152.4*mm;}
G4double PMT12inchHQE::GetPMTGlassThickness() {return 0.55*cm;}
G4float PMT12inchHQE::HitTimeSmearing(float Q) {
  float timingConstant = 2.0; 
  float timingResolution = 0.33 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 0.58) timingResolution=0.58;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* PMT12inchHQE::Getqpe() //currently uses the same as 20inch
   {
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000129, 0.000754, 0.004060, 0.028471,
    // 2
    0.068449, 0.115679, 0.164646, 0.203466, 0.235631,
    0.262351, 0.282064, 0.303341, 0.320618, 0.338317,
    0.357825, 0.371980, 0.385820, 0.398838, 0.413595,
    0.428590, 0.444387, 0.461685, 0.482383, 0.502369,
    0.520779, 0.540011, 0.559293, 0.579354, 0.599337,
    0.619580, 0.639859, 0.659807, 0.679810, 0.699620,
    0.718792, 0.737382, 0.755309, 0.772042, 0.788232,
    0.803316, 0.817861, 0.831148, 0.844339, 0.855532,
    0.866693, 0.876604, 0.886067, 0.894473, 0.902150,
    0.909515, 0.915983, 0.922050, 0.927418, 0.932492,
    // 3
    0.936951, 0.940941, 0.944660, 0.948004, 0.951090,
    0.953833, 0.956576, 0.958886, 0.961134, 0.963116,
    0.964930, 0.966562, 0.968008, 0.969424, 0.970687,
    0.971783, 0.972867, 0.973903, 0.974906, 0.975784,
    0.976632, 0.977438, 0.978190, 0.978891, 0.979543,
    0.980124, 0.980666, 0.981255, 0.981770, 0.982227,
    0.982701, 0.983146, 0.983566, 0.983975, 0.984357,
    0.984713, 0.985094, 0.985404, 0.985739, 0.986049,
    0.986339, 0.986630, 0.986922, 0.987176, 0.987431,
    0.987655, 0.987922, 0.988173, 0.988414, 0.988639,
    // 4
    0.988856, 0.989065, 0.989273, 0.989475, 0.989662,
    0.989828, 0.990007, 0.990172, 0.990327, 0.990497,
    0.990645, 0.990797, 0.990981, 0.991135, 0.991272,
    0.991413, 0.991550, 0.991673, 0.991805, 0.991928,
    0.992063, 0.992173, 0.992296, 0.992406, 0.992514,
    0.992632, 0.992733, 0.992837, 0.992954, 0.993046,
    0.993148, 0.993246, 0.993354, 0.993458, 0.993549,
    0.993656, 0.993744, 0.993836, 0.993936, 0.994033,
    0.994134, 0.994222, 0.994307, 0.994413, 0.994495,
    0.994572, 0.994659, 0.994739, 0.994816, 0.994886,
    // 5
    0.994970, 0.995032, 0.995110, 0.995178, 0.995250,
    0.995321, 0.995383, 0.995464, 0.995532, 0.995609,
    0.995674, 0.995750, 0.995821, 0.995889, 0.995952,
    0.996010, 0.996071, 0.996153, 0.996218, 0.996283,
    0.996335, 0.996384, 0.996431, 0.996484, 0.996537,
    0.996597, 0.996655, 0.996701, 0.996745, 0.996802,
    0.996860, 0.996917, 0.996962, 0.997014, 0.997079,
    0.997114, 0.997165, 0.997204, 0.997250, 0.997295,
    0.997335, 0.997379, 0.997418, 0.997454, 0.997488,
    0.997530, 0.997573, 0.997606, 0.997648, 0.997685,
    // 6
    0.997725, 0.997762, 0.997795, 0.997835, 0.997866,
    0.997898, 0.997941, 0.997966, 0.997997, 0.998039,
    0.998065, 0.998104, 0.998128, 0.998153, 0.998179,
    0.998205, 0.998223, 0.998254, 0.998293, 0.998319,
    0.998346, 0.998374, 0.998397, 0.998414, 0.998432,
    0.998456, 0.998482, 0.998511, 0.998532, 0.998553,
    0.998571, 0.998594, 0.998614, 0.998638, 0.998669,
    0.998693, 0.998715, 0.998743, 0.998762, 0.998793,
    0.998812, 0.998834, 0.998857, 0.998872, 0.998888,
    0.998904, 0.998926, 0.998946, 0.998963, 0.998983,
    // 7
    0.999007, 0.999027, 0.999044, 0.999064, 0.999079,
    0.999096, 0.999120, 0.999133, 0.999152, 0.999160,
    0.999174, 0.999188, 0.999206, 0.999221, 0.999234,
    0.999248, 0.999263, 0.999276, 0.999286, 0.999300,
    0.999313, 0.999321, 0.999331, 0.999347, 0.999356,
    0.999369, 0.999381, 0.999394, 0.999402, 0.999415,
    0.999427, 0.999433, 0.999446, 0.999458, 0.999472,
    0.999484, 0.999499, 0.999513, 0.999522, 0.999532,
    0.999540, 0.999550, 0.999559, 0.999567, 0.999574,
    0.999588, 0.999599, 0.999613, 0.999618, 0.999627,
    // 8
    0.999635, 0.999639, 0.999652, 0.999662, 0.999667,
    0.999671, 0.999678, 0.999682, 0.999688, 0.999693,
    0.999698, 0.999701, 0.999706, 0.999711, 0.999718,
    0.999722, 0.999727, 0.999732, 0.999737, 0.999740,
    0.999746, 0.999750, 0.999754, 0.999763, 0.999766,
    0.999769, 0.999774, 0.999780, 0.999784, 0.999788,
    0.999796, 0.999803, 0.999807, 0.999809, 0.999815,
    0.999820, 0.999827, 0.999830, 0.999833, 0.999833,
    0.999836, 0.999839, 0.999842, 0.999845, 0.999850,
    0.999853, 0.999857, 0.999860, 0.999865, 0.999870,
    // 9
    0.999873, 0.999877, 0.999880, 0.999882, 0.999883,
    0.999886, 0.999888, 0.999889, 0.999895, 0.999896,
    0.999897, 0.999901, 0.999902, 0.999905, 0.999907,
    0.999907, 0.999909, 0.999911, 0.999911, 0.999912,
    0.999913, 0.999914, 0.999917, 0.999919, 0.999921,
    0.999923, 0.999927, 0.999929, 0.999931, 0.999933,
    0.999936, 0.999942, 0.999942, 0.999944, 0.999947,
    0.999947, 0.999948, 0.999949, 0.999952, 0.999955,
    0.999957, 0.999957, 0.999961, 0.999962, 0.999963,
    0.999963, 0.999963, 0.999964, 0.999965, 0.999965,
    // 10
    0.999965, 0.999965, 0.999966, 0.999968, 0.999969,
    0.999971, 0.999972, 0.999972, 0.999973, 0.999975,
    0.999975, 0.999975, 0.999975, 0.999975, 0.999975,
    0.999975, 0.999979, 0.999979, 0.999980, 0.999982,
    0.999983, 0.999985, 0.999986, 0.999987, 0.999987,
    0.999988, 0.999989, 0.999989, 0.999989, 0.999989,
    0.999990, 0.999990, 0.999992, 0.999993, 0.999994,
    0.999994, 0.999994, 0.999994, 0.999994, 0.999995,
    0.999995, 0.999995, 0.999996, 0.999996, 0.999996,
    0.999996, 0.999998, 0.999999, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array
    0.0 
  };
   return qpe0;
  }

G4float* PMT12inchHQE::GetQE()//currently uses the same as the 10inchHQE
{
 static G4float QE[20] =
   { 0.00, .0502, .2017, .2933, .3306, .3396, .3320, .3168, .2915, .2655, 
      .2268,  .1971, .1641, .1102, .0727, .0499, .0323, .0178, .0061, 0.00};
 return QE;
}
G4float* PMT12inchHQE::GetQEWavelength(){static G4float wavelength[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength;}

G4float  PMT12inchHQE::GetmaxQE()//currently uses the same as the 10inchHQE
{
  const G4float maxQE = 0.3396;
  return maxQE;
}




////////////////////////////////////////////////////////////////////////////////////////////////
// 20 inch HPD


//////////////////////////////////////////////
//20" HPD Parameter Source                  //
//                                          //
// 1pe resolution: EGADS data (8" HPD)      //
// Timing Resolution: EGADS data (8" HPD)   //
// QE: HQE R3600 PMT data from Hamamatsu    //
//                                          //
//////////////////////////////////////////////

HPD20inchHQE::HPD20inchHQE(){}
HPD20inchHQE::~HPD20inchHQE(){}

G4String HPD20inchHQE::GetPMTName() {G4String PMTName = "HPD20inchHQE"; return PMTName;}
G4double HPD20inchHQE::GetExposeHeight() {return .192*m;}
G4double HPD20inchHQE::GetRadius() {return .254*m;}
G4double HPD20inchHQE::GetPMTGlassThickness() {return 0.3*cm;}
float HPD20inchHQE::HitTimeSmearing(float Q) {
  float timingConstant = 5.0; 
  float timingResolution = 0.47 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 1.2) timingResolution=1.2;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* HPD20inchHQE::Getqpe()
   {
  static G4float qpe0[501]= {
    // 1                                                                                           
    0.000381, 0.000402, 0.000402, 0.000423, 0.000444,
    0.000444, 0.000444, 0.000465, 0.000465, 0.000465,
    0.000465, 0.000465, 0.000486, 0.000486, 0.000529,
    0.000550, 0.000550, 0.000613, 0.000634, 0.000656,
    0.000656, 0.000677, 0.000740, 0.000782, 0.000782,
    0.000782, 0.000782, 0.000782, 0.000782, 0.000782,
    0.000782, 0.000782, 0.000804, 0.000825, 0.000909,
    0.000930, 0.000930, 0.001015, 0.001100, 0.001205,
    0.001374, 0.001480, 0.001649, 0.002136, 0.002855,
    0.004039, 0.005752, 0.009071, 0.012582, 0.016917,
    // 2                                                                                           
    0.023620, 0.031845, 0.042714, 0.048593, 0.059863,
    0.070013, 0.078577, 0.089933, 0.101584, 0.115138,
    0.129221, 0.143368, 0.157895, 0.175573, 0.194012,
    0.214121, 0.236578, 0.264236, 0.293861, 0.326362,
    0.360174, 0.398406, 0.439069, 0.480831, 0.526231,
    0.572731, 0.618603, 0.664208, 0.708600, 0.750897,
    0.790346, 0.826361, 0.858545, 0.886698, 0.910804,
    0.931009, 0.947585, 0.960897, 0.971362, 0.979414,
    0.985479, 0.989951, 0.993178, 0.995457, 0.997034,
    0.998101, 0.998808, 0.999266, 0.999557, 0.999738,
    // 3                                                                                           
    0.999848, 0.999914, 0.999952, 0.999974, 0.999986,
    0.999993, 0.999996, 0.999998, 0.999999, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 4                                                                                           
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 5                                                                                           
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 6                                                                                           
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 7                                                                                          
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 8                                                                                          
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 9                                                                                          
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 10                                                                                         
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array                        
  0.0  };
   return qpe0;
  }

G4float* HPD20inchHQE::GetQEWavelength(){
  static G4float wavelength_value[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength_value;
}

G4float* HPD20inchHQE::GetQE(){
  static G4float QE[20] =
    { 0.00, .0008, .1255, .254962, .2930, .3127, .3130, .2994, .2791, .2491,
      .2070,  .1758, .1384, .0779, .0473, .0288, .0149, .0062, .0002, .0001};  
  return QE;
}
G4float HPD20inchHQE::GetmaxQE(){
  const G4float maxQE = 0.315;
  return maxQE;
}

G4float* HPD20inchHQE::GetCollectionEfficiencyArray(){
  static G4float CE[10] = { 98., 98., 98., 98., 98., 98., 98., 98., 98., 98.};
  return CE;
}


////////////////////////////////////////////////////////////////////////////////////////////////
// 12 inch HPD

//Information for the HPD12inchHQE is identical to the HPD20inchHQE, except for GetPMTName, GetRadius and GetExposeHeight. 

HPD12inchHQE::HPD12inchHQE(){}
HPD12inchHQE::~HPD12inchHQE(){}

G4String HPD12inchHQE::GetPMTName() {G4String PMTName = "HPD12inchHQE"; return PMTName;}
G4double HPD12inchHQE::GetExposeHeight() {return 118.*mm;} //Assumed to be the same as the PMT12inchHQE.
G4double HPD12inchHQE::GetRadius() {return 152.4*mm;} //12 inches
G4double HPD12inchHQE::GetPMTGlassThickness() {return 0.3*cm;} 
float HPD12inchHQE::HitTimeSmearing(float Q) {
  float timingConstant = 5.0; 
  float timingResolution = 0.47 + sqrt(timingConstant/Q); 
  // looking at SK's jitter function for 20" tubes
  if (timingResolution < 1.2) timingResolution=1.2;
  float Smearing_factor = G4RandGauss::shoot(0.0,timingResolution);
  return Smearing_factor;
}

G4float* HPD12inchHQE::Getqpe()
   {
  static G4float qpe0[501]= {
    // 1                                                                                           
    0.000381, 0.000402, 0.000402, 0.000423, 0.000444,
    0.000444, 0.000444, 0.000465, 0.000465, 0.000465,
    0.000465, 0.000465, 0.000486, 0.000486, 0.000529,
    0.000550, 0.000550, 0.000613, 0.000634, 0.000656,
    0.000656, 0.000677, 0.000740, 0.000782, 0.000782,
    0.000782, 0.000782, 0.000782, 0.000782, 0.000782,
    0.000782, 0.000782, 0.000804, 0.000825, 0.000909,
    0.000930, 0.000930, 0.001015, 0.001100, 0.001205,
    0.001374, 0.001480, 0.001649, 0.002136, 0.002855,
    0.004039, 0.005752, 0.009071, 0.012582, 0.016917,
    // 2                                                                                           
    0.023620, 0.031845, 0.042714, 0.048593, 0.059863,
    0.070013, 0.078577, 0.089933, 0.101584, 0.115138,
    0.129221, 0.143368, 0.157895, 0.175573, 0.194012,
    0.214121, 0.236578, 0.264236, 0.293861, 0.326362,
    0.360174, 0.398406, 0.439069, 0.480831, 0.526231,
    0.572731, 0.618603, 0.664208, 0.708600, 0.750897,
    0.790346, 0.826361, 0.858545, 0.886698, 0.910804,
    0.931009, 0.947585, 0.960897, 0.971362, 0.979414,
    0.985479, 0.989951, 0.993178, 0.995457, 0.997034,
    0.998101, 0.998808, 0.999266, 0.999557, 0.999738,
    // 3                                                                                           
    0.999848, 0.999914, 0.999952, 0.999974, 0.999986,
    0.999993, 0.999996, 0.999998, 0.999999, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 4                                                                                           
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 5                                                                                           
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 6                                                                                           
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 7                                                                                          
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 8                                                                                          
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 9                                                                                          
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 10                                                                                         
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array                        
  0.0  };
   return qpe0;
  }

G4float* HPD12inchHQE::GetQEWavelength(){
  static G4float wavelength_value[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength_value;
}

G4float* HPD12inchHQE::GetQE(){
  static G4float QE[20] =
    { 0.00, .0008, .1255, .254962, .2930, .3127, .3130, .2994, .2791, .2491,
      .2070,  .1758, .1384, .0779, .0473, .0288, .0149, .0062, .0002, .0001};  
  return QE;
}
G4float HPD12inchHQE::GetmaxQE(){
  const G4float maxQE = 0.315;
  return maxQE;
}

G4float* HPD12inchHQE::GetCollectionEfficiencyArray(){
  static G4float CE[10] = { 98., 98., 98., 98., 98., 98., 98., 98., 98., 98.};
  return CE;
}


////////////////////////////////////////////////////////////////////////////////////////////////
// 20 inch HQE Box and Line PMT

//////////////////////////////////////////////////////////////
// BoxandLine20inchHQE Parameter Source                     //
//                                                          //
// 1pe resolution: Measured data in dark box (20" B&L)      //
// Timing Resolution: Measured data in dark box (20" B&L)   //
// QE: HQE R3600 PMT data from Hamamatsu                    //
// CE: Simulation result of Hamamatsu (<460mm phi area)     //
//////////////////////////////////////////////////////////////

 

BoxandLine20inchHQE::BoxandLine20inchHQE(){}
BoxandLine20inchHQE::~BoxandLine20inchHQE(){}

G4String BoxandLine20inchHQE::GetPMTName() {G4String PMTName = "BoxandLine20inchHQE"; return PMTName;}
G4double BoxandLine20inchHQE::GetExposeHeight() {return .18*m;}
G4double BoxandLine20inchHQE::GetRadius() {return .254*m;}
G4double BoxandLine20inchHQE::GetPMTGlassThickness() {return 0.4*cm;}

float BoxandLine20inchHQE::HitTimeSmearing(float Q) {
  G4float sig_param[4]={0.8928,0.03278,0.07054,16.4};
  G4float lambda_param[2]={0.4094,0.06852};

  G4float sigma = (sig_param[0]*(exp(-sig_param[1]*Q)+sig_param[2])*(Q<sig_param[3])+(2*sig_param[0]*sig_param[1]*sig_param[3]*sqrt(sig_param[3])*exp(-sig_param[1]*sig_param[3])/sqrt(Q)+(exp(-sig_param[1]*sig_param[3])+sig_param[2])-2*sig_param[0]*sig_param[1]*sig_param[3]*exp(-sig_param[1]*sig_param[3]))*(Q>sig_param[3]));
  G4float lambda = lambda_param[0]+lambda_param[1]*Q;
  G4float Smearing_factor = G4RandGauss::shoot(-0.2,sigma)-1/lambda*log(1-G4UniformRand());
  return Smearing_factor;
}

G4float* BoxandLine20inchHQE::Getqpe()
{
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000001, 0.000001, 0.000002, 0.000004,
    0.000008, 0.000014, 0.000025, 0.000044, 0.000486,
    // 2
    0.007195, 0.019406, 0.031920, 0.044503, 0.057189,
    0.070020, 0.083060, 0.096388, 0.110108, 0.124351,
    0.139276, 0.155072, 0.171956, 0.190167, 0.209961,
    0.231594, 0.255310, 0.281319, 0.309777, 0.340762,
    0.374259, 0.410142, 0.448167, 0.487976, 0.529101,
    0.570993, 0.613041, 0.654608, 0.695067, 0.733833,
    0.770390, 0.804317, 0.835304, 0.863151, 0.887777,
    0.909203, 0.927543, 0.942987, 0.955778, 0.966198,
    0.974543, 0.981116, 0.986205, 0.990078, 0.992974,
    0.995104, 0.996642, 0.997734, 0.998495, 0.999017,
    // 3
    0.999369, 0.999601, 0.999752, 0.999848, 0.999909,
    0.999946, 0.999969, 0.999982, 0.999990, 0.999994,
    0.999997, 0.999998, 0.999999, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 4
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 5
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 6
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 7
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 8
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 9
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 10
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array                        
    0.0  };
  return qpe0;
}
G4float* BoxandLine20inchHQE::GetQEWavelength(){
  static G4float wavelength_value[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength_value;
}

G4float* BoxandLine20inchHQE::GetQE(){
  static G4float QE[20] =
    { 0.00, .0008, .1255, .254962, .2930, .3127, .3130, .2994, .2791, .2491,
      .2070,  .1758, .1384, .0779, .0473, .0288, .0149, .0062, .0002, .0001};  
  return QE;
}
G4float BoxandLine20inchHQE::GetmaxQE(){
  const G4float maxQE = 0.315;
  return maxQE;
}
G4float* BoxandLine20inchHQE::GetCollectionEfficiencyArray(){  
  static G4float CE[10] = { 95., 95., 95., 95., 95., 95., 95., 95., 95., 95.};
  return CE;
}




////////////////////////////////////////////////////////////////////////////////////////////////
// 12 inch HQE Box and Line PMT

//Information for the BoxandLine12inchHQE is identical to the BoxandLine20inchHQE, except for GetPMTName, GetRadius and GetExposeHeight.

BoxandLine12inchHQE::BoxandLine12inchHQE(){}
BoxandLine12inchHQE::~BoxandLine12inchHQE(){}

G4String BoxandLine12inchHQE::GetPMTName() {G4String PMTName = "BoxandLine12inchHQE"; return PMTName;}
G4double BoxandLine12inchHQE::GetExposeHeight() {return 118.*mm;}
G4double BoxandLine12inchHQE::GetRadius() {return 152.4*mm;}
G4double BoxandLine12inchHQE::GetPMTGlassThickness() {return 0.4*cm;}

float BoxandLine12inchHQE::HitTimeSmearing(float Q) {
  G4float sig_param[4]={0.8928,0.03278,0.07054,16.4};
  G4float lambda_param[2]={0.4094,0.06852};

  G4float sigma = (sig_param[0]*(exp(-sig_param[1]*Q)+sig_param[2])*(Q<sig_param[3])+(2*sig_param[0]*sig_param[1]*sig_param[3]*sqrt(sig_param[3])*exp(-sig_param[1]*sig_param[3])/sqrt(Q)+(exp(-sig_param[1]*sig_param[3])+sig_param[2])-2*sig_param[0]*sig_param[1]*sig_param[3]*exp(-sig_param[1]*sig_param[3]))*(Q>sig_param[3]));
  G4float lambda = lambda_param[0]+lambda_param[1]*Q;
  G4float Smearing_factor = G4RandGauss::shoot(-0.2,sigma)-1/lambda*log(1-G4UniformRand());
  return Smearing_factor;
}

G4float* BoxandLine12inchHQE::Getqpe()
{
  static G4float qpe0[501]= {
    // 1
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000001, 0.000001, 0.000002, 0.000004,
    0.000008, 0.000014, 0.000025, 0.000044, 0.000486,
    // 2
    0.007195, 0.019406, 0.031920, 0.044503, 0.057189,
    0.070020, 0.083060, 0.096388, 0.110108, 0.124351,
    0.139276, 0.155072, 0.171956, 0.190167, 0.209961,
    0.231594, 0.255310, 0.281319, 0.309777, 0.340762,
    0.374259, 0.410142, 0.448167, 0.487976, 0.529101,
    0.570993, 0.613041, 0.654608, 0.695067, 0.733833,
    0.770390, 0.804317, 0.835304, 0.863151, 0.887777,
    0.909203, 0.927543, 0.942987, 0.955778, 0.966198,
    0.974543, 0.981116, 0.986205, 0.990078, 0.992974,
    0.995104, 0.996642, 0.997734, 0.998495, 0.999017,
    // 3
    0.999369, 0.999601, 0.999752, 0.999848, 0.999909,
    0.999946, 0.999969, 0.999982, 0.999990, 0.999994,
    0.999997, 0.999998, 0.999999, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 4
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 5
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 6
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 7
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 8
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 9
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // 10
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    // Dummy element for noticing if the loop reached the end of the array                        
    0.0  };
  return qpe0;
}
G4float* BoxandLine12inchHQE::GetQEWavelength(){
  static G4float wavelength_value[20] = { 280., 300., 320., 340., 360., 380., 400., 420., 440., 460., 480., 500., 520., 540., 560., 580., 600., 620., 640., 660.};
  return wavelength_value;
}

G4float* BoxandLine12inchHQE::GetQE(){
  static G4float QE[20] =
    { 0.00, .0008, .1255, .254962, .2930, .3127, .3130, .2994, .2791, .2491,
      .2070,  .1758, .1384, .0779, .0473, .0288, .0149, .0062, .0002, .0001};  
  return QE;
}
G4float BoxandLine12inchHQE::GetmaxQE(){
  const G4float maxQE = 0.315;
  return maxQE;
}
G4float* BoxandLine12inchHQE::GetCollectionEfficiencyArray(){  
  static G4float CE[10] = { 95., 95., 95., 95., 95., 95., 95., 95., 95., 95.};
  return CE;
}
