# Jsonnet Rules

## Rules

* [`jsonnet_library`](#jsonnet_library)
* [`jsonnet_to_json`](#jsonnet_to_json)

## Overview

These are build rules for working with [Jsonnet][jsonnet] files with Bazel.

[jsonnet]: http://google.github.io/jsonnet/doc/

## Setup

To use the Jsonnet rules, simply copy the contents of `jsonnet.WORKSPACE` into
your `WORKSPACE` file.

<a name="#jsonnet_library"></a>
## jsonnet_library

```python
jsonnet_library(name, srcs, deps, imports)
```

<table>
  <thead>
    <tr>
      <th>Attribute</th>
      <th>Description</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <td><code>name</code></td>
      <td>
        <code>Name, required</code>
        <p>A unique name for this rule.</p>
      </td>
    </tr>
    <tr>
      <td><code>srcs</code></td>
      <td>
        <code>List of Labels, required</code>
        <p>
          List of <code>.jsonnet</code> files that comprises this Jsonnet
          library.
        </p>
      </td>
    </tr>
    <tr>
      <td><code>deps</code></td>
      <td>
        <code>List of labels, optional</code>
        <p>
          List of targets that are required by the <code>srcs</code> Jsonnet
          files.
        </p>
      </td>
    </tr>
    <tr>
      <td><code>imports</code></td>
      <td>
        <code>List of strings, optional</code>
        <p>
          List of import <code>-J</code> flags to be passed to the
          <code>jsonnet</code> compiler.
        </p>
      </td>
    </tr>
  </tbody>
</table>

### Example

Suppose you have the following directory structure:

```
[workspace]/
    WORKSPACE
    configs/
      BUILD
      backend.jsonnet
      frontend.jsonnet
```

You can use the `jsonnet_library` rule to build a collection of `.jsonnet`
files that can be imported by other `.jsonnet` files as dependencies:

`configs/BUILD`:

```python
load("/tools/build_defs/jsonnet/jsonnet", "jsonnet_library")

jsonnet_library(
    name = "configs",
    srcs = [
        "backend.jsonnet",
        "frontend.jsonnet",
    ],
)
```

<a name="#jsonnet_to_json"></a>
## jsonnet_to_json

```python
jsonnet_to_json(name, src, deps, outs, multiple_outputs, imports)
```

<table>
  <thead>
    <tr>
      <th>Attribute</th>
      <th>Description</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <td><code>name</code></td>
      <td>
        <code>Name, required</code>
        <p>A unique name for this rule.</p>
        <p>
          This name will be used as the name of the JSON file generated by this
          rule.
        </p>
      </td>
    </tr>
    <tr>
      <td><code>src</code></td>
      <td>
        <code>Label, required</code>
        <p>
          The <code>.jsonnet</code> file to convert to JSON.
        </p>
      </td>
    </tr>
    <tr>
      <td><code>deps</code></td>
      <td>
        <code>List of labels, optional</code>
        <p>
          List of targets that are required by the <code>src</code> Jsonnet
          file.
        </p>
      </td>
    </tr>
    <tr>
      <td><code>outs</code></td>
      <td>
        <code>List of Filenames, optional</code>
        <p>
          Names of the output .json files to be generated by this rule.
        </p>
        <p>
          If you are generating only a single JSON file and are not using
          jsonnet multiple output files, then this attribute should only
          contain the file name of the JSON file you are generating.
        </p>
        <p>
          If you are generating multiple JSON files using jsonnet multiple file
          output (<code>jsonnet -m</code>), then list the file names of all the
          JSON files to be generated. The file names specified here must match
          the file names specified in your <code>src</code> Jsonnet file.
        </p>
        <p>
          For the case where multiple file output is used but only for
          generating one output file, set the <code>multiple_outputs</code>
          attribute to 1 to explicitly enable the <code>-m</code> flag for
          multiple file output.
        </p>
      </td>
    </tr>
    <tr>
      <td><code>multiple_outputs</code></td>
      <td>
        <code>bool, optional, default 0</code>
        <p>
          Set to 1 to explicitly enable multiple file output via the
          <code>jsonnet -m</code> flag.
        </p>
        <p>
          This is used for the case where multiple file output is used but only
          for generating a single output file. For example:
        </p>
<pre>
local foo = import "foo.jsonnet";

{
  "foo.json": foo,
}
</pre>
        </p>
      </td>
    </tr>
    <tr>
      <td><code>imports</code></td>
      <td>
        <code>List of strings, optional</code>
        <p>
          List of import <code>-J</code> flags to be passed to the
          <code>jsonnet</code> compiler.
        </p>
      </td>
    </tr>
  </tbody>
</table>

### Example

Suppose you have the following directory structure:

```
[workspace]/
    WORKSPACE
    workflows/
        BUILD
        workflow.jsonnet
        wordcount.jsonnet
        intersection.jsonnet
```

Say that `workflow.jsonnet` is a base configuration library for a workflow
scheduling system and `wordcount.jsonnet` and `intersection.jsonnet` both
import `workflow.jsonnet` to define workflows for performing a wordcount and
intersection of two files, respectively.

First, create a `jsonnet_library` target with `workflow.jsonnet`:

`workflows/BUILD`:

```python
load("/tools/build_defs/jsonnet/jsonnet", "jsonnet_library")

jsonnet_library(
    name = "workflow",
    srcs = ["workflow.jsonnet"],
)
```

To compile `wordcount.jsonnet` and `intersection.jsonnet` to JSON, define two
`jsonnet_to_json` targets:

```python
jsonnet_to_json(
    name = "wordcount",
    src = "wordcount.jsonnet",
    outs = ["wordcount.json"],
    deps = [":workflow"],
)

jsonnet_to_json(
    name = "intersection",
    src = "intersection.jsonnet",
    outs = ["intersection.json"],
    deps = [":workflow"],
)
```

### Example: Multiple output files

To use Jsonnet's [multiple output files][multiple-output-files], suppose you
add a file `shell-workflows.jsonnet` that imports `wordcount.jsonnet` and
`intersection.jsonnet`:

`workflows/shell-workflows.jsonnet`:

```
local wordcount = import "workflows/wordcount.jsonnet";
local intersection = import "workflows/intersection.jsonnet";

{
  "wordcount-workflow.json": wordcount,
  "intersection-workflow.json": intersection,
}
```

To compile `shell-workflows.jsonnet` into the two JSON files,
`wordcount-workflow.json` and `intersection-workflow.json`, first create a
`jsonnet_library` target containing the two files that
`shell-workflows.jsonnet` depends on:

```python
jsonnet_library(
    name = "shell-workflows-lib",
    srcs = [
        "wordcount.jsonnet",
        "intersection.jsonnet",
    ],
    deps = [":workflow"],
)
```

Then, create a `jsonnet_to_json` target and set `outs` to the list of output
files to indicate that multiple output JSON files are generated:

```python
jsonnet_to_json(
    name = "shell-workflows",
    src = "shell-workflows.jsonnet",
    deps = [":shell-workflows-lib"],
    outs = [
        "wordcount-workflow.jsonnet",
        "intersection-workflow.jsonnet",
    ],
)
```

[multiple-output-files]: http://google.github.io/jsonnet/doc/commandline.html
