#!/bin/bash
#
# Copyright (C) 2010-2016 CERN. All rights not expressly granted are reserved.
# 
# This file is part of the CERN Control and Monitoring Platform 'C2MON'.
# C2MON is free software: you can redistribute it and/or modify it under the
# terms of the GNU Lesser General Public License as published by the Free
# Software Foundation, either version 3 of the license.
# 
# C2MON is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for
# more details.
# 
# You should have received a copy of the GNU Lesser General Public License
# along with C2MON. If not, see <http://www.gnu.org/licenses/>.
##


# Script for starting and stopping a C2MON server on the local machine.


###############################
# Initialise helper functions #
###############################

BOOTUP=color
# column to start "[  OK  ]" label in 
RES_COL=60
# terminal sequence to move to that column. You could change this
# to something like "tput hpa ${RES_COL}" if your terminal supports it
MOVE_TO_COL="echo -en \\033[${RES_COL}G"
# terminal sequence to set color to a 'success' color (currently: green)
SETCOLOR_SUCCESS="echo -en \\033[0;32m"
# terminal sequence to set color to a 'failure' color (currently: red)
SETCOLOR_FAILURE="echo -en \\033[0;31m"
# terminal sequence to set color to a 'warning' color (currently: yellow)
SETCOLOR_WARNING="echo -en \\033[0;33m"
# terminal sequence to reset to the default color.
SETCOLOR_NORMAL="echo -en \\033[0;39m"

echo_success() {
  [ "$BOOTUP" = "color" ] && $MOVE_TO_COL
  echo -n "["
  [ "$BOOTUP" = "color" ] && $SETCOLOR_SUCCESS
  echo -n $"  OK  "
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  echo -n "]"
  echo -ne "\r"
  return 0
}

echo_failure() {
  [ "$BOOTUP" = "color" ] && $MOVE_TO_COL
  echo -n "["
  [ "$BOOTUP" = "color" ] && $SETCOLOR_FAILURE
  echo -n $"FAILED"
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  echo -n "]"
  echo -ne "\r"
  return 1
}

echo_warning() {
  [ "$BOOTUP" = "color" ] && $MOVE_TO_COL
  echo -n "["
  [ "$BOOTUP" = "color" ] && $SETCOLOR_WARNING
  echo -n $"WARNING"
  [ "$BOOTUP" = "color" ] && $SETCOLOR_NORMAL
  echo -n "]"
  echo -ne "\r"
  return 1
}


########################
# DEPLOYMENT VARIABLES #
########################

# get the current location 
SCRIPT=`readlink -f $(which $0)`
SCRIPTPATH=`dirname $SCRIPT`
export C2MON_HOME="$(dirname "$SCRIPTPATH")"


####################
# MORE SETTINGS... #
####################

#log directory
LOG_DIR=$C2MON_HOME/log

#set correct host
HOST_TMP_DIR=$C2MON_HOME/tmp/

#set correct PID file
C2MON_PIDFILE=$HOST_TMP_DIR/c2mon.pid

# Parse options (could be handled better with getopt)
if [ "$1" == "run" ]  ; then
  export RUN_FOREGROUND="TRUE"
fi

if [ "$2" == "-r" ] || [ "$2" == "--recover" ] || [ "$3" == "-r" ] || [ "$3" == "--recover" ] ; then
  export C2MON_RECOVERY_ENABLED="TRUE"
fi

if [ "$2" == "-d" ] || [ "$2" == "--debug" ] || [ "$3" == "-d" ] || [ "$3" == "--debug" ] ; then
  export REMOTE_DEBUG_ENABLED="TRUE"
fi

# Checking for start script
C2MON_START_CMD=`ls $C2MON_HOME/bin/*.jvm | head -1`
if [ ! -f $C2MON_START_CMD ] ; then
  printf "[ERROR] No jvm start script found in $C2MON_HOME/bin/\n"
  exit 1
fi

RETVAL=0

# Call the runs function to find out if a process with
# a certain pid runs. The "runs" function will return
# 0 if the process is running, 1 if it is not.
# #Example: runs 23049
runs() {
  pid=${1##*/}
  tmp=`ps -p $pid -o pid=`
  if [ -z "$tmp" ] ; then
    return 1 
  else
    return 0 
  fi
}

# The start function checks whether the Oc4j service is already
# running and, if it think it is it, calls the really_start 
# function to launch it.
# Otherwise it exits with a warning message

start() {
  printf "Starting a C2MON server: "
  # Check if the PID file exists
  # If it exists, check whether the process is really running
  # If it is already running, print an error message and exit
  # If it is not running, clean up the PID and LOCK files and start OC4J

  if [ -f $C2MON_PIDFILE ] ; then
    # Check if process with PID in PID files is running
    pid=`cat $C2MON_PIDFILE`
    runs $pid
    if [ $? -eq 1 ] ; then
      # It is not running --> remove PID file and LOCK file
      rm $C2MON_PIDFILE

      really_start
    else
      echo_warning
      echo
      echo "This C2MON server seems to be already running"
    fi
  else
    #clean
    really_start
  fi
}  

# The really_start function tries to start the C2MON server
# and then checks if it running.

really_start() {
  cd $C2MON_HOME

  if [ -n "${RUN_FOREGROUND+set}" ] ; then
    echo "Running in foreground mode (no log files will be written)"
    $C2MON_START_CMD
  else
    # Calls the script that was generated by the deployment procedure
    $C2MON_START_CMD > $LOG_DIR/out.log 2> $LOG_DIR/err.log &

    pid=$!
    sleep 1
    runs $pid
    if [ $? -eq 0 ] ; then
      echo $pid > $C2MON_PIDFILE
      echo_success
    else
      echo_failure
    fi
    echo
    return $chk
  fi
}

stop() {
  printf $"Shutting down this C2MON server..."
  if [ -f $C2MON_PIDFILE ] ; then
    pid=`cat $C2MON_PIDFILE`

    # First check if C2MON is running
    runs $pid

    if [ $? -eq 1 ] ; then
      # C2MON is not running --> just remove the PID file
      rm -f $C2MON_PIDFILE
      echo_warning
      echo
      echo "C2MON server is not running on this host."
    else
      # C2MON is running --> try a gentle shutdown
      kill $pid >/dev/null 2>&1
      PROC_RUNS=$?
      PROC_WAIT=0;
      while [ $PROC_RUNS -eq 0 ]; do
        printf .
        sleep 1 
        if [ $PROC_WAIT -lt 20 ]; then
          let PROC_WAIT=$PROC_WAIT+1
          runs $pid
          PROC_RUNS=$?
        else
          PROC_RUNS=1
        fi
      done  
      runs $pid
      if [ $? -eq 0 ] ; then
        echo_warning
        echo
        echo "Gentle shutdown failed. Killing this C2MON server."
        echo "To ensure all data was processed, one of the following actions should be taken: "
        echo "  - either restart a server using the *recover* option"
        echo "  - or use the JConsole to run the *RecoveryManager -> recover task* (on other running server for instance)"
        kill -9 $pid >/dev/null 2>&1
        sleep 1
        runs $pid
        if [ $? -eq 1 ] ; then
          rm -f $C2MON_PIDFILE
          echo_success
          echo
          RETVAL=0
        else
          echo_failure
          echo
          echo "Unable to shut down C2MON server (supposed pid=$pid)."
          RETVAL=1
        fi
      else
        rm -f $C2MON_PIDFILE
        echo_success
        echo
        RETVAL=0
      fi
      
    fi
  else
    echo_failure
    echo
    echo "No pid file ($C2MON_PIDFILE) found. If this C2MON server is running, kill it manually"
    RETVAL=1
  fi
  return $RETVAL
}  

status() {
  pid=
  if [ -f $C2MON_PIDFILE ]; then
    pid=`cat $C2MON_PIDFILE`
    runs $pid
    if [ $? -eq 0 ] ; then
       echo "This C2MON server is running (pid=$pid)."
       RETVAL=0
    else
       echo "A pid file exists ($C2MON_PIDFILE) but the C2MON process is not running."
       RETVAL=1
    fi
  else
    echo "This C2MON server does not seem to be running."
    RETVAL=2
  fi
  return $RETVAL
}

#make tmp dir on correct machine
if [ ! -d "$HOST_TMP_DIR" ]; then
  mkdir $HOST_TMP_DIR
fi

case "$1" in
'start')
  start
;;

'stop')
  stop
;;

'restart')
  stop && start
;;

'status')
  status
;;

'run')
  start
;;

*)
  echo
  echo $"Usage: $0 {start|stop|restart|status|run} [-d|--debug] [-r|--recover]"
  echo
  echo $"start   - Starts the C2MON server on this host, if it is not running."
  echo $"stop    - Stops the C2MON server on this host, if it is running. If a gentle shutdown fails, the process is killed after 20 seconds."
  echo $"restart - Restarts the C2MON server."
  echo $"status  - Checks the status (running/stopped) of the C2MON server."
  echo $"run     - Starts the C2MON server in the foreground without logging to a file."
  echo
  echo $"-d, --debug"
  echo $"         Allows attaching a remote debugger to the C2MON Java process."
  echo $"-r, --recover"
  echo $"         Recover after a server crash."
  exit 1
esac

exit $?
