package runner

import (
	"context"
	"encoding/json"
	"fmt"
	"os"
	"path/filepath"
	"regexp"
	"runtime"
	"strings"

	"github.com/nektos/act/pkg/container"

	"github.com/nektos/act/pkg/common"
	"github.com/nektos/act/pkg/model"
	"github.com/sirupsen/logrus"
	log "github.com/sirupsen/logrus"
)

// RunContext contains info about current job
type RunContext struct {
	Config       *Config
	Matrix       map[string]interface{}
	Run          *model.Run
	EventJSON    string
	Env          map[string]string
	ExtraPath    []string
	CurrentStep  string
	StepResults  map[string]*stepResult
	ExprEval     ExpressionEvaluator
	JobContainer container.Container
}

type stepResult struct {
	Success bool              `json:"success"`
	Outputs map[string]string `json:"outputs"`
}

// GetEnv returns the env for the context
func (rc *RunContext) GetEnv() map[string]string {
	if rc.Env == nil {
		rc.Env = mergeMaps(rc.Run.Workflow.Env, rc.Run.Job().Env)
	}
	return rc.Env
}

func (rc *RunContext) jobContainerName() string {
	return createContainerName("act", rc.Run.String())
}

func (rc *RunContext) startJobContainer() common.Executor {
	job := rc.Run.Job()

	var image string
	if job.Container != nil {
		image = job.Container.Image
	} else {
		platformName := rc.ExprEval.Interpolate(job.RunsOn)
		image = rc.Config.Platforms[strings.ToLower(platformName)]
	}

	return func(ctx context.Context) error {
		rawLogger := common.Logger(ctx).WithField("raw_output", true)
		logWriter := common.NewLineWriter(rc.commandHandler(ctx), func(s string) bool {
			if rc.Config.LogOutput {
				rawLogger.Infof(s)
			} else {
				rawLogger.Debugf(s)
			}
			return true
		})

		common.Logger(ctx).Infof("\U0001f680  Start image=%s", image)
		name := rc.jobContainerName()

		envList := make([]string, 0)
		bindModifiers := ""
		if runtime.GOOS == "darwin" {
			bindModifiers = ":delegated"
		}

		envList = append(envList, fmt.Sprintf("%s=%s", "RUNNER_TOOL_CACHE", "/toolcache"))

		binds := []string{
			fmt.Sprintf("%s:%s", "/var/run/docker.sock", "/var/run/docker.sock"),
		}
		if rc.Config.BindWorkdir {
			binds = append(binds, fmt.Sprintf("%s:%s%s", rc.Config.Workdir, "/github/workspace", bindModifiers))
		}

		rc.JobContainer = container.NewContainer(&container.NewContainerInput{
			Cmd:        nil,
			Entrypoint: []string{"/usr/bin/tail", "-f", "/dev/null"},
			WorkingDir: "/github/workspace",
			Image:      image,
			Name:       name,
			Env:        envList,
			Mounts: map[string]string{
				name:            "/github",
				"act-toolcache": "/toolcache",
				"act-actions":   "/actions",
			},

			Binds:  binds,
			Stdout: logWriter,
			Stderr: logWriter,
		})

		return common.NewPipelineExecutor(
			rc.JobContainer.Pull(rc.Config.ForcePull),
			rc.JobContainer.Remove().IfBool(!rc.Config.ReuseContainers),
			rc.JobContainer.Create(),
			rc.JobContainer.Start(false),
			rc.JobContainer.CopyDir("/github/workspace", rc.Config.Workdir+"/.").IfBool(!rc.Config.BindWorkdir),
			rc.JobContainer.Copy("/github/", &container.FileEntry{
				Name: "workflow/event.json",
				Mode: 644,
				Body: rc.EventJSON,
			}, &container.FileEntry{
				Name: "home/.act",
				Mode: 644,
				Body: "",
			}),
		)(ctx)
	}
}
func (rc *RunContext) execJobContainer(cmd []string, env map[string]string) common.Executor {
	return func(ctx context.Context) error {
		return rc.JobContainer.Exec(cmd, env)(ctx)
	}
}
func (rc *RunContext) stopJobContainer() common.Executor {
	return func(ctx context.Context) error {
		if rc.JobContainer != nil && !rc.Config.ReuseContainers {
			return rc.JobContainer.Remove().
				Then(container.NewDockerVolumeRemoveExecutor(rc.jobContainerName(), false))(ctx)
		}
		return nil
	}
}

// ActionCacheDir is for rc
func (rc *RunContext) ActionCacheDir() string {
	var xdgCache string
	var ok bool
	if xdgCache, ok = os.LookupEnv("XDG_CACHE_HOME"); !ok {
		if home, ok := os.LookupEnv("HOME"); ok {
			xdgCache = fmt.Sprintf("%s/.cache", home)
		}
	}
	return filepath.Join(xdgCache, "act")
}

// Executor returns a pipeline executor for all the steps in the job
func (rc *RunContext) Executor() common.Executor {
	steps := make([]common.Executor, 0)
	steps = append(steps, rc.startJobContainer())

	for i, step := range rc.Run.Job().Steps {
		if step.ID == "" {
			step.ID = fmt.Sprintf("%d", i)
		}
		steps = append(steps, rc.newStepExecutor(step))
	}
	steps = append(steps, rc.stopJobContainer())

	return common.NewPipelineExecutor(steps...).If(rc.isEnabled)
}

func (rc *RunContext) newStepExecutor(step *model.Step) common.Executor {
	sc := &StepContext{
		RunContext: rc,
		Step:       step,
	}
	return func(ctx context.Context) error {
		rc.CurrentStep = sc.Step.ID
		rc.StepResults[rc.CurrentStep] = &stepResult{
			Success: true,
			Outputs: make(map[string]string),
		}
		rc.ExprEval = sc.NewExpressionEvaluator()

		if !rc.EvalBool(sc.Step.If) {
			log.Debugf("Skipping step '%s' due to '%s'", sc.Step.String(), sc.Step.If)
			return nil
		}

		common.Logger(ctx).Infof("\u2B50  Run %s", sc.Step)
		err := sc.Executor()(ctx)
		if err == nil {
			common.Logger(ctx).Infof("  \u2705  Success - %s", sc.Step)
		} else {
			common.Logger(ctx).Errorf("  \u274C  Failure - %s", sc.Step)
			rc.StepResults[rc.CurrentStep].Success = false
		}
		return err
	}
}

func (rc *RunContext) isEnabled(ctx context.Context) bool {
	job := rc.Run.Job()
	log := common.Logger(ctx)
	if !rc.EvalBool(job.If) {
		log.Debugf("Skipping job '%s' due to '%s'", job.Name, job.If)
		return false
	}

	platformName := rc.ExprEval.Interpolate(rc.Run.Job().RunsOn)
	if img, ok := rc.Config.Platforms[strings.ToLower(platformName)]; !ok || img == "" {
		log.Infof("  \U0001F6A7  Skipping unsupported platform '%s'", platformName)
		return false
	}
	return true
}

// EvalBool evaluates an expression against current run context
func (rc *RunContext) EvalBool(expr string) bool {
	if expr != "" {
		v, err := rc.ExprEval.Evaluate(expr)
		if err != nil {
			log.Errorf("Error evaluating expression '%s' - %v", expr, err)
			return false
		}
		return v == "true"
	}
	return true
}

func mergeMaps(maps ...map[string]string) map[string]string {
	rtnMap := make(map[string]string)
	for _, m := range maps {
		for k, v := range m {
			rtnMap[k] = v
		}
	}
	return rtnMap
}

func createContainerName(parts ...string) string {
	name := make([]string, 0)
	pattern := regexp.MustCompile("[^a-zA-Z0-9]")
	partLen := (30 / len(parts)) - 1
	for i, part := range parts {
		if i == len(parts)-1 {
			name = append(name, pattern.ReplaceAllString(part, "-"))
		} else {
			name = append(name, trimToLen(pattern.ReplaceAllString(part, "-"), partLen))
		}
	}
	return trimToLen(strings.Trim(strings.Join(name, "-"), "-"), 30)
}

func trimToLen(s string, l int) string {
	if l < 0 {
		l = 0
	}
	if len(s) > l {
		return s[:l]
	}
	return s
}

type jobContext struct {
	Status    string `json:"status"`
	Container struct {
		ID      string `json:"id"`
		Network string `json:"network"`
	} `json:"container"`
	Services map[string]struct {
		ID string `json:"id"`
	} `json:"services"`
}

func (rc *RunContext) getJobContext() *jobContext {
	jobStatus := "success"
	for _, stepStatus := range rc.StepResults {
		if !stepStatus.Success {
			jobStatus = "failure"
			break
		}
	}
	return &jobContext{
		Status: jobStatus,
	}
}

func (rc *RunContext) getStepsContext() map[string]*stepResult {
	return rc.StepResults
}

type githubContext struct {
	Event      map[string]interface{} `json:"event"`
	EventPath  string                 `json:"event_path"`
	Workflow   string                 `json:"workflow"`
	RunID      string                 `json:"run_id"`
	RunNumber  string                 `json:"run_number"`
	Actor      string                 `json:"actor"`
	Repository string                 `json:"repository"`
	EventName  string                 `json:"event_name"`
	Sha        string                 `json:"sha"`
	Ref        string                 `json:"ref"`
	HeadRef    string                 `json:"head_ref"`
	BaseRef    string                 `json:"base_ref"`
	Token      string                 `json:"token"`
	Workspace  string                 `json:"workspace"`
	Action     string                 `json:"action"`
}

func (rc *RunContext) getGithubContext() *githubContext {
	ghc := &githubContext{
		Event:     make(map[string]interface{}),
		EventPath: "/github/workflow/event.json",
		Workflow:  rc.Run.Workflow.Name,
		RunID:     "1",
		RunNumber: "1",
		Actor:     "nektos/act",

		EventName: rc.Config.EventName,
		Token:     os.Getenv("GITHUB_TOKEN"),
		Workspace: "/github/workspace",
		Action:    rc.CurrentStep,
	}

	repoPath := rc.Config.Workdir
	repo, err := common.FindGithubRepo(repoPath)
	if err != nil {
		log.Warningf("unable to get git repo: %v", err)
	} else {
		ghc.Repository = repo
	}

	_, sha, err := common.FindGitRevision(repoPath)
	if err != nil {
		log.Warningf("unable to get git revision: %v", err)
	} else {
		ghc.Sha = sha
	}

	ref, err := common.FindGitRef(repoPath)
	if err != nil {
		log.Warningf("unable to get git ref: %v", err)
	} else {
		log.Debugf("using github ref: %s", ref)
		ghc.Ref = ref
	}
	err = json.Unmarshal([]byte(rc.EventJSON), &ghc.Event)
	if err != nil {
		logrus.Error(err)
	}
	return ghc
}

func (rc *RunContext) withGithubEnv(env map[string]string) map[string]string {
	github := rc.getGithubContext()
	env["HOME"] = "/github/home"
	env["GITHUB_WORKFLOW"] = github.Workflow
	env["GITHUB_RUN_ID"] = github.RunID
	env["GITHUB_RUN_NUMBER"] = github.RunNumber
	env["GITHUB_ACTION"] = github.Action
	env["GITHUB_ACTIONS"] = "true"
	env["GITHUB_ACTOR"] = github.Actor
	env["GITHUB_REPOSITORY"] = github.Repository
	env["GITHUB_EVENT_NAME"] = github.EventName
	env["GITHUB_EVENT_PATH"] = github.EventPath
	env["GITHUB_WORKSPACE"] = github.Workspace
	env["GITHUB_SHA"] = github.Sha
	env["GITHUB_REF"] = github.Ref
	return env
}
