/**
 * This module contains the implementation of the C++ header generation available through
 * the command line switch -Hc.
 *
 * Copyright:   Copyright (C) 1999-2021 by The D Language Foundation, All Rights Reserved
 * Authors:     $(LINK2 http://www.digitalmars.com, Walter Bright)
 * License:     $(LINK2 http://www.boost.org/LICENSE_1_0.txt, Boost License 1.0)
 * Source:      $(LINK2 https://github.com/dlang/dmd/blob/master/src/dmd/dtohd, _dtoh.d)
 * Documentation:  https://dlang.org/phobos/dmd_dtoh.html
 * Coverage:    https://codecov.io/gh/dlang/dmd/src/master/src/dmd/dtoh.d
 */
module dmd.dtoh;

import core.stdc.stdio;
import core.stdc.string;
import core.stdc.ctype;

import dmd.astcodegen;
import dmd.arraytypes;
import dmd.dsymbol;
import dmd.errors;
import dmd.globals;
import dmd.identifier;
import dmd.root.filename;
import dmd.visitor;
import dmd.tokens;

import dmd.root.outbuffer;
import dmd.utils;

//debug = Debug_DtoH;

private struct DMDType
{
    __gshared Identifier c_long;
    __gshared Identifier c_ulong;
    __gshared Identifier c_longlong;
    __gshared Identifier c_ulonglong;
    __gshared Identifier c_long_double;
    __gshared Identifier c_wchar_t;
    __gshared Identifier c_complex_float;
    __gshared Identifier c_complex_double;
    __gshared Identifier c_complex_real;

    static void _init()
    {
        c_long          = Identifier.idPool("__c_long");
        c_ulong         = Identifier.idPool("__c_ulong");
        c_longlong      = Identifier.idPool("__c_longlong");
        c_ulonglong     = Identifier.idPool("__c_ulonglong");
        c_long_double   = Identifier.idPool("__c_long_double");
        c_wchar_t       = Identifier.idPool("__c_wchar_t");
        c_complex_float  = Identifier.idPool("__c_complex_float");
        c_complex_double = Identifier.idPool("__c_complex_double");
        c_complex_real = Identifier.idPool("__c_complex_real");
    }
}

private void initialize()
{
    __gshared bool initialized;

    if (!initialized)
    {
        initialized = true;

        DMDType._init();
    }
}

void hashIf(ref OutBuffer buf, string content)
{
    buf.writestring("#if ");
    buf.writestringln(content);
}

void hashElIf(ref OutBuffer buf, string content)
{
    buf.writestring("#elif ");
    buf.writestringln(content);
}

void hashEndIf(ref OutBuffer buf)
{
    buf.writestringln("#endif");
}

void hashDefine(ref OutBuffer buf, string content)
{
    buf.writestring("# define ");
    buf.writestringln(content);
}

void hashInclude(ref OutBuffer buf, string content)
{
    buf.writestring("#include ");
    buf.writestringln(content);
}



extern(C++) void genCppHdrFiles(ref Modules ms)
{
    initialize();

    OutBuffer fwd;
    OutBuffer check;
    OutBuffer done;
    OutBuffer decl;

    // enable indent by spaces on buffers
    fwd.doindent = true;
    fwd.spaces = true;
    decl.doindent = true;
    decl.spaces = true;
    check.doindent = true;
    check.spaces = true;

    scope v = new ToCppBuffer(&check, &fwd, &done, &decl);

    OutBuffer buf;
    buf.doindent = true;
    buf.spaces = true;

    foreach (m; ms)
        m.accept(v);

    if (global.params.doCxxHdrGeneration == CxxHeaderMode.verbose)
        buf.printf("// Automatically generated by %s Compiler v%d", global.vendor.ptr, global.versionNumber());
    else
        buf.printf("// Automatically generated by %s Compiler", global.vendor.ptr);

    buf.writenl();
    buf.writenl();
    buf.writestringln("#pragma once");
    buf.writenl();
    hashInclude(buf, "<assert.h>");
    hashInclude(buf, "<stddef.h>");
    hashInclude(buf, "<stdint.h>");
    hashInclude(buf, "<math.h>");
//    buf.writestring(buf, "#include <stdio.h>\n");
//    buf.writestring("#include <string.h>\n");

    // Emit array compatibility because extern(C++) types may have slices
    // as members (as opposed to function parameters)
    buf.writestring(`
#ifdef CUSTOM_D_ARRAY_TYPE
#define _d_dynamicArray CUSTOM_D_ARRAY_TYPE
#else
/// Represents a D [] array
template<typename T>
struct _d_dynamicArray
{
    size_t length;
    T *ptr;

    _d_dynamicArray() : length(0), ptr(NULL) { }

    _d_dynamicArray(size_t length_in, T *ptr_in)
        : length(length_in), ptr(ptr_in) { }

    T& operator[](const size_t idx) {
        assert(idx < length);
        return ptr[idx];
    }

    const T& operator[](const size_t idx) const {
        assert(idx < length);
        return ptr[idx];
    }
};
#endif
`);

    if (v.hasReal)
    {
        hashIf(buf, "!defined(_d_real)");
        {
            hashDefine(buf, "_d_real long double");
        }
        hashEndIf(buf);
    }
    buf.writenl();
    // buf.writestringln("// fwd:");
    buf.write(&fwd);
    if (fwd.length > 0)
        buf.writenl();

    // buf.writestringln("// done:");
    buf.write(&done);

    // buf.writestringln("// decl:");
    buf.write(&decl);

    debug (Debug_DtoH)
    {
        // buf.writestringln("// check:");
        buf.writestring(`
#if OFFSETS
    template <class T>
    size_t getSlotNumber(int dummy, ...)
    {
        T c;
        va_list ap;
        va_start(ap, dummy);

        void *f = va_arg(ap, void*);
        for (size_t i = 0; ; i++)
        {
            if ( (*(void***)&c)[i] == f)
            return i;
        }
        va_end(ap);
    }

    void testOffsets()
    {
`);
        buf.write(&check);
        buf.writestring(`
    }
#endif
`);
    }

    if (global.params.cxxhdrname is null)
    {
        // Write to stdout; assume it succeeds
        size_t n = fwrite(buf[].ptr, 1, buf.length, stdout);
        assert(n == buf.length); // keep gcc happy about return values
    }
    else
    {
        const(char)[] name = FileName.combine(global.params.cxxhdrdir, global.params.cxxhdrname);
        writeFile(Loc.initial, name, buf[]);
    }
}

/****************************************************
 */
extern(C++) final class ToCppBuffer : Visitor
{
    alias visit = Visitor.visit;
public:
    enum EnumKind
    {
        Int,
        Numeric,
        String,
        Enum,
        Other
    }

    alias AST = ASTCodegen;

    bool[void*] visited;
    bool[void*] forwarded;
    OutBuffer* fwdbuf;
    OutBuffer* checkbuf;
    OutBuffer* donebuf;
    OutBuffer* buf;
    AST.AggregateDeclaration adparent;
    AST.TemplateDeclaration tdparent;
    Identifier ident;
    LINK linkage = LINK.d;
    bool forwardedAA;
    AST.Type* origType;
    /// Last written visibility level
    AST.Visibility.Kind currentVisibility;
    AST.STC storageClass; /// Currently applicable storage classes

    int ignoredCounter; /// How many symbols were ignored
    bool hasReal;
    const bool printIgnored;

    this(OutBuffer* checkbuf, OutBuffer* fwdbuf, OutBuffer* donebuf, OutBuffer* buf)
    {
        this.checkbuf = checkbuf;
        this.fwdbuf = fwdbuf;
        this.donebuf = donebuf;
        this.buf = buf;
        this.printIgnored = global.params.doCxxHdrGeneration == CxxHeaderMode.verbose;
    }

    /// Visit `dsym` with `buf` while temporarily clearing **parent fields
    private void visitAsRoot(AST.Dsymbol dsym, OutBuffer* buf)
    {
        const stcStash = this.storageClass;
        auto adStash = this.adparent;
        auto tdStash = this.tdparent;
        auto bufStash = this.buf;
        auto countStash = this.ignoredCounter;

        this.storageClass = AST.STC.undefined_;
        this.adparent = null;
        this.tdparent = null;
        this.buf = buf;

        dsym.accept(this);

        this.storageClass = stcStash;
        this.adparent = adStash;
        this.tdparent = tdStash;
        this.buf = bufStash;
        this.ignoredCounter = countStash;
    }

    private EnumKind getEnumKind(AST.Type type)
    {
        if (type) switch (type.ty)
        {
            case AST.Tint32:
                return EnumKind.Int;
            case AST.Tbool,
                AST.Tchar, AST.Twchar, AST.Tdchar,
                AST.Tint8, AST.Tuns8,
                AST.Tint16, AST.Tuns16,
                AST.Tuns32,
                AST.Tint64, AST.Tuns64:
                return EnumKind.Numeric;
            case AST.Tarray:
                if (type.isString())
                    return EnumKind.String;
                break;
            case AST.Tenum:
                return EnumKind.Enum;
            default:
                break;
        }
        return EnumKind.Other;
    }

    private AST.Type determineEnumType(AST.Type type)
    {
        if (auto arr = type.isTypeDArray())
        {
            switch (arr.next.ty)
            {
                case AST.Tchar:  return AST.Type.tchar.constOf.pointerTo;
                case AST.Twchar: return AST.Type.twchar.constOf.pointerTo;
                case AST.Tdchar: return AST.Type.tdchar.constOf.pointerTo;
                default: break;
            }
        }
        return type;
    }

    void writeDeclEnd()
    {
        buf.writestringln(";");

        if (!adparent)
            buf.writenl();
    }

    /// Writes the corresponding access specifier if necessary
    private void writeProtection(const AST.Visibility.Kind kind)
    {
        // Don't write visibility for global declarations
        if (!adparent)
            return;

        string token;

        switch(kind) with(AST.Visibility.Kind)
        {
            case none, private_:
                if (this.currentVisibility == AST.Visibility.Kind.private_)
                    return;
                this.currentVisibility = AST.Visibility.Kind.private_;
                token = "private:";
                break;

            case package_, protected_:
                if (this.currentVisibility == AST.Visibility.Kind.protected_)
                    return;
                this.currentVisibility = AST.Visibility.Kind.protected_;
                token = "protected:";
                break;

            case undefined, public_, export_:
                if (this.currentVisibility == AST.Visibility.Kind.public_)
                    return;
                this.currentVisibility = AST.Visibility.Kind.public_;
                token = "public:";
                break;

            default:
                printf("Unexpected visibility: %d!\n", kind);
                assert(0);
        }

        buf.level--;
        buf.writestringln(token);
        buf.level++;
    }

    /**
     * Writes an identifier into `buf` and checks for reserved identifiers. The
     * parameter `canFix` determines how this function handles C++ keywords:
     *
     * `false` => Raise a warning and print the identifier as-is
     * `true`  => Append an underscore to the identifier
     *
     * Params:
     *   s        = the symbol denoting the identifier
     *   canFixup = whether the identifier may be changed without affecting
     *              binary compatibility
     */
    private void writeIdentifier(const AST.Dsymbol s, const bool canFix = false)
    {
        writeIdentifier(s.ident, s.loc, s.kind(), canFix);
    }

    /** Overload of `writeIdentifier` used for all AST nodes not descending from Dsymbol **/
    private void writeIdentifier(const Identifier ident, const Loc loc, const char* kind, const bool canFix = false)
    {
        bool needsFix;

        void warnCxxCompat(const(char)* reason)
        {
            if (canFix)
            {
                needsFix = true;
                return;
            }

            static bool warned = false;
            warning(loc, "%s `%s` is a %s", kind, ident.toChars(), reason);

            if (!warned)
            {
                warningSupplemental(loc, "The generated C++ header will contain " ~
                                    "identifiers that are keywords in C++");
                warned = true;
            }
        }

        if (global.params.warnings != DiagnosticReporting.off || canFix)
        {
            // Warn about identifiers that are keywords in C++.
            switch (ident.toString())
            {
                // C++ operators
                case "and":
                case "and_eq":
                case "bitand":
                case "bitor":
                case "compl":
                case "not":
                case "not_eq":
                case "or":
                case "or_eq":
                case "xor":
                case "xor_eq":
                    warnCxxCompat("special operator in C++");
                    break;

                // C++ keywords
                case "const_cast":
                case "delete":
                case "dynamic_cast":
                case "explicit":
                case "friend":
                case "inline":
                case "mutable":
                case "namespace":
                case "operator":
                case "register":
                case "reinterpret_cast":
                case "signed":
                case "static_cast":
                case "typedef":
                case "typename":
                case "unsigned":
                case "using":
                case "virtual":
                case "volatile":
                    warnCxxCompat("keyword in C++");
                    break;

                // C++11 keywords
                case "alignas":
                case "alignof":
                case "char16_t":
                case "char32_t":
                case "constexpr":
                case "decltype":
                case "noexcept":
                case "nullptr":
                case "static_assert":
                case "thread_local":
                    if (global.params.cplusplus >= CppStdRevision.cpp11)
                        warnCxxCompat("keyword in C++11");
                    break;

                // C++20 keywords
                case "char8_t":
                case "consteval":
                case "constinit":
                // Concepts-related keywords
                case "concept":
                case "requires":
                // Coroutines-related keywords
                case "co_await":
                case "co_yield":
                case "co_return":
                    if (global.params.cplusplus >= CppStdRevision.cpp20)
                        warnCxxCompat("keyword in C++20");
                    break;

                default:
                    break;
            }
        }
        buf.writestring(ident.toString());
        if (needsFix)
            buf.writeByte('_');
    }

    override void visit(AST.Dsymbol s)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.Dsymbol enter] %s\n", s.toChars());
            import dmd.asttypename;
            printf("[AST.Dsymbol enter] %s\n", s.astTypeName().ptr);
            scope(exit) printf("[AST.Dsymbol exit] %s\n", s.toChars());
        }
    }

    override void visit(AST.Import i)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.Import enter] %s\n", i.toChars());
            scope(exit) printf("[AST.Import exit] %s\n", i.toChars());
        }
    }

    override void visit(AST.AttribDeclaration pd)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.AttribDeclaration enter] %s\n", pd.toChars());
            scope(exit) printf("[AST.AttribDeclaration exit] %s\n", pd.toChars());
        }
        Dsymbols* decl = pd.include(null);
        if (!decl)
            return;

        foreach (s; *decl)
        {
            if (adparent || s.visible().kind >= AST.Visibility.Kind.public_)
                s.accept(this);
        }
    }

    override void visit(AST.StorageClassDeclaration scd)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.StorageClassDeclaration enter] %s\n", scd.toChars());
            scope(exit) printf("[AST.StorageClassDeclaration exit] %s\n", scd.toChars());
        }
        const stcStash = this.storageClass;
        this.storageClass |= scd.stc;
        visit(cast(AST.AttribDeclaration) scd);
        this.storageClass = stcStash;
    }

    override void visit(AST.LinkDeclaration ld)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.LinkDeclaration enter] %s\n", ld.toChars());
            scope(exit) printf("[AST.LinkDeclaration exit] %s\n", ld.toChars());
        }
        auto save = linkage;
        linkage = ld.linkage;
        visit(cast(AST.AttribDeclaration)ld);
        linkage = save;
    }

    override void visit(AST.CPPMangleDeclaration md)
    {
        const oldLinkage = this.linkage;
        this.linkage = LINK.cpp;
        visit(cast(AST.AttribDeclaration) md);
        this.linkage = oldLinkage;
    }

    override void visit(AST.Module m)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.Module enter] %s\n", m.toChars());
            scope(exit) printf("[AST.Module exit] %s\n", m.toChars());
        }
        foreach (s; *m.members)
        {
            if (s.visible().kind < AST.Visibility.Kind.public_)
                continue;
            s.accept(this);
        }
    }

    override void visit(AST.FuncDeclaration fd)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.FuncDeclaration enter] %s\n", fd.toChars());
            scope(exit) printf("[AST.FuncDeclaration exit] %s\n", fd.toChars());
        }
        if (cast(void*)fd in visited)
            return;
        // printf("FuncDeclaration %s %s\n", fd.toPrettyChars(), fd.type.toChars());
        visited[cast(void*)fd] = true;

        // Note that tf might be null for templated (member) functions
        auto tf = cast(AST.TypeFunction)fd.type;
        if ((tf && tf.linkage != LINK.c && tf.linkage != LINK.cpp) || (!tf && fd.isPostBlitDeclaration()))
        {
            ignored("function %s because of linkage", fd.toPrettyChars());

            // Virtual extern(D) functions require a dummy declaration to ensure proper
            // vtable layout - but omit redundant declarations - the slot was already
            // reserved  in the base class
            if (fd.isVirtual() && fd.introducing)
            {
                // Hide placeholders because they are not ABI compatible
                writeProtection(AST.Visibility.Kind.private_);

                __gshared int counter; // Ensure unique names in all cases
                buf.printf("virtual void __vtable_slot_%u();", counter++);
                buf.writenl();
            }
            return;
        }
        if (!adparent && !fd.fbody)
        {
            ignored("function %s because it is extern", fd.toPrettyChars());
            return;
        }
        if (fd.visibility.kind == AST.Visibility.Kind.none || fd.visibility.kind == AST.Visibility.Kind.private_)
        {
            ignored("function %s because it is private", fd.toPrettyChars());
            return;
        }

        writeProtection(fd.visibility.kind);

        if (tf && tf.linkage == LINK.c)
            buf.writestring("extern \"C\" ");
        else if (!adparent)
            buf.writestring("extern ");
        if (adparent && fd.isStatic())
            buf.writestring("static ");
        else if (adparent && (
            // Virtual functions in non-templated classes
            (fd.vtblIndex != -1 && !fd.isOverride()) ||

            // Virtual functions in templated classes (fd.vtblIndex still -1)
            (tdparent && adparent.isClassDeclaration() && !(this.storageClass & AST.STC.final_ || fd.isFinal))))
                buf.writestring("virtual ");

        if (adparent && !tdparent)
        {
            auto s = adparent.search(Loc.initial, fd.ident);
            auto cd = adparent.isClassDeclaration();

            if (!(adparent.storage_class & AST.STC.abstract_) &&
                !(cd && cd.isAbstract()) &&
                s is fd && !fd.overnext)
            {
                const cn = adparent.ident.toChars();
                const fn = fd.ident.toChars();
                const vi = fd.vtblIndex;

                checkbuf.printf("assert(getSlotNumber <%s>(0, &%s::%s) == %d);",
                                                       cn,     cn, fn,    vi);
                checkbuf.writenl();
           }
        }

        if (adparent && fd.isDisabled && global.params.cplusplus < CppStdRevision.cpp11)
            writeProtection(AST.Visibility.Kind.private_);
        funcToBuffer(tf, fd);
        // FIXME: How to determine if fd is const without tf?
        if (adparent && tf && (tf.isConst() || tf.isImmutable()))
        {
            bool fdOverridesAreConst = true;
            foreach (fdv; fd.foverrides)
            {
                auto tfv = cast(AST.TypeFunction)fdv.type;
                if (!tfv.isConst() && !tfv.isImmutable())
                {
                    fdOverridesAreConst = false;
                    break;
                }
            }

            buf.writestring(fdOverridesAreConst ? " const" : " /* const */");
        }
        if (adparent && fd.isAbstract())
            buf.writestring(" = 0");
        if (adparent && fd.isDisabled && global.params.cplusplus >= CppStdRevision.cpp11)
            buf.writestring(" = delete");
        buf.writestringln(";");
        if (adparent && fd.isDisabled && global.params.cplusplus < CppStdRevision.cpp11)
            writeProtection(AST.Visibility.Kind.public_);

        if (!adparent)
            buf.writenl();

    }

    override void visit(AST.UnitTestDeclaration utd)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.UnitTestDeclaration enter] %s\n", utd.toChars());
            scope(exit) printf("[AST.UnitTestDeclaration exit] %s\n", utd.toChars());
        }
    }

    override void visit(AST.VarDeclaration vd)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.VarDeclaration enter] %s\n", vd.toChars());
            scope(exit) printf("[AST.VarDeclaration exit] %s\n", vd.toChars());
        }
        if (cast(void*)vd in visited)
            return;
        visited[cast(void*)vd] = true;

        // Tuple field are expanded into multiple VarDeclarations
        // (we'll visit them later)
        if (vd.type && vd.type.isTypeTuple())
            return;

        if (vd.type == AST.Type.tsize_t)
            origType = &vd.originalType;
        scope(exit) origType = null;

        if (vd.alignment != STRUCTALIGN_DEFAULT)
        {
            buf.printf("// Ignoring var %s alignment %u", vd.toChars(), vd.alignment);
            buf.writenl();
        }

        if (vd.storage_class & AST.STC.manifest &&
            vd._init && vd._init.isExpInitializer() && vd.type !is null)
        {
            if (linkage != LINK.c && linkage != LINK.cpp)
            {
                ignored("variable %s because of linkage", vd.toPrettyChars());
                return;
            }
            AST.Type type = vd.type;
            EnumKind kind = getEnumKind(type);

            if (vd.visibility.kind == AST.Visibility.Kind.none || vd.visibility.kind == AST.Visibility.Kind.private_) {
                ignored("enum `%s` because it is `%s`.", vd.toPrettyChars(), AST.visibilityToChars(vd.visibility.kind));
                return;
            }

            writeProtection(vd.visibility.kind);

            final switch (kind)
            {
                case EnumKind.Int, EnumKind.Numeric:
                    // 'enum : type' is only available from C++-11 onwards.
                    if (global.params.cplusplus < CppStdRevision.cpp11)
                        goto case;
                    buf.writestring("enum : ");
                    determineEnumType(type).accept(this);
                    buf.writestring(" { ");
                    writeIdentifier(vd, true);
                    buf.writestring(" = ");
                    auto ie = AST.initializerToExpression(vd._init).isIntegerExp();
                    visitInteger(ie.toInteger(), type);
                    buf.writestring(" };");
                    break;

                case EnumKind.String, EnumKind.Enum:
                    buf.writestring("static ");
                    auto target = determineEnumType(type);
                    target.accept(this);
                    buf.writestring(" const ");
                    writeIdentifier(vd, true);
                    buf.writestring(" = ");
                    auto e = AST.initializerToExpression(vd._init);
                    printExpressionFor(target, e);
                    buf.writestring(";");
                    break;

                case EnumKind.Other:
                    ignored("enum `%s` because type `%s` is currently not supported for enum constants.", vd.toPrettyChars(), type.toChars());
                    return;
            }
            buf.writenl();
            buf.writenl();
            return;
        }

        if (tdparent && vd.type && !vd.type.deco)
        {
            if (linkage != LINK.c && linkage != LINK.cpp)
            {
                ignored("variable %s because of linkage", vd.toPrettyChars());
                return;
            }
            writeProtection(vd.visibility.kind);
            typeToBuffer(vd.type, vd, adparent && !(vd.storage_class & (AST.STC.static_ | AST.STC.gshared)));
            buf.writestringln(";");
            return;
        }

        if (vd.storage_class & (AST.STC.static_ | AST.STC.extern_ | AST.STC.tls | AST.STC.gshared) ||
        vd.parent && vd.parent.isModule())
        {
            if (vd.linkage != LINK.c && vd.linkage != LINK.cpp)
            {
                ignored("variable %s because of linkage", vd.toPrettyChars());
                return;
            }
            if (vd.storage_class & AST.STC.tls)
            {
                ignored("variable %s because of thread-local storage", vd.toPrettyChars());
                return;
            }
            writeProtection(vd.visibility.kind);
            if (vd.linkage == LINK.c)
                buf.writestring("extern \"C\" ");
            else if (!adparent)
                buf.writestring("extern ");
            if (adparent)
                buf.writestring("static ");
            typeToBuffer(vd.type, vd);
            writeDeclEnd();
            return;
        }

        if (adparent && vd.type && vd.type.deco)
        {
            writeProtection(vd.visibility.kind);
            typeToBuffer(vd.type, vd, true);
            buf.writestringln(";");

            if (auto t = vd.type.isTypeStruct())
                includeSymbol(t.sym);

            checkbuf.level++;
            const pn = adparent.ident.toChars();
            const vn = vd.ident.toChars();
            const vo = vd.offset;
            checkbuf.printf("assert(offsetof(%s, %s) == %d);",
                                             pn, vn,    vo);
            checkbuf.writenl();
            checkbuf.level--;
            return;
        }

        visit(cast(AST.Dsymbol)vd);
    }

    override void visit(AST.TypeInfoDeclaration tid)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeInfoDeclaration enter] %s\n", tid.toChars());
            scope(exit) printf("[AST.TypeInfoDeclaration exit] %s\n", tid.toChars());
        }
    }

    override void visit(AST.AliasDeclaration ad)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.AliasDeclaration enter] %s\n", ad.toChars());
            scope(exit) printf("[AST.AliasDeclaration exit] %s\n", ad.toChars());
        }
        writeProtection(ad.visibility.kind);

        if (auto t = ad.type)
        {
            if (t.ty == AST.Tdelegate)
            {
                visit(cast(AST.Dsymbol)ad);
                return;
            }

            // for function pointers we need to original type
            if (ad.type.ty == AST.Tpointer &&
                (cast(AST.TypePointer)t).nextOf.ty == AST.Tfunction)
            {
                origType = &ad.originalType;
            }
            scope(exit) origType = null;

            buf.writestring("typedef ");
            typeToBuffer(origType ? *origType : t, ad);
            writeDeclEnd();
            return;
        }
        if (!ad.aliassym)
        {
            assert(0);
        }
        if (auto ti = ad.aliassym.isTemplateInstance())
        {
            visitTi(ti);
            return;
        }
        if (auto sd = ad.aliassym.isStructDeclaration())
        {
            buf.writestring("typedef ");
            sd.type.accept(this);
            buf.writestring(" ");
            writeIdentifier(ad);
            writeDeclEnd();
            return;
        }
        else if (auto td = ad.aliassym.isTemplateDeclaration())
        {
            if (global.params.cplusplus < CppStdRevision.cpp11)
            {
                ignored("%s because `using` declarations require C++ 11", ad.toPrettyChars());
                return;
            }

            printTemplateParams(td);
            buf.writestring("using ");
            writeIdentifier(ad);
            buf.printf(" = %s<", td.ident.toChars());

            foreach (const idx, const p; *td.parameters)
            {
                if (idx)
                    buf.writestring(", ");
                writeIdentifier(p.ident, p.loc, "parameter", true);
            }
            buf.writestringln(">;");
            return;
        }

        if (ad.aliassym.isDtorDeclaration())
        {
            // Ignore. It's taken care of while visiting FuncDeclaration
            return;
        }

        ignored("%s %s", ad.aliassym.kind(), ad.aliassym.toPrettyChars());
    }

    override void visit(AST.Nspace ns)
    {
        handleNspace(ns, ns.members);
    }

    override void visit(AST.CPPNamespaceDeclaration ns)
    {
        handleNspace(ns, ns.decl);
    }

    void handleNspace(AST.Dsymbol namespace, Dsymbols* members)
    {
        buf.writestring("namespace ");
        writeIdentifier(namespace);
        buf.writenl();
        buf.writestring("{");
        buf.writenl();
        buf.level++;
        foreach(decl;(*members))
        {
            decl.accept(this);
        }
        buf.level--;
        buf.writestring("}");
        buf.writenl();
    }

    override void visit(AST.AnonDeclaration ad)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.AnonDeclaration enter] %s\n", ad.toChars());
            scope(exit) printf("[AST.AnonDeclaration exit] %s\n", ad.toChars());
        }

        buf.writestringln(ad.isunion ? "union" : "struct");
        buf.writestringln("{");
        buf.level++;
        foreach (s; *ad.decl)
        {
            s.accept(this);
        }
        buf.level--;
        buf.writestringln("};");
    }

    private bool memberField(AST.VarDeclaration vd)
    {
        if (!vd.type || !vd.type.deco || !vd.ident)
            return false;
        if (!vd.isField())
            return false;
        if (vd.type.ty == AST.Tfunction)
            return false;
        if (vd.type.ty == AST.Tsarray)
            return false;
        return true;
    }

    override void visit(AST.StructDeclaration sd)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.StructDeclaration enter] %s\n", sd.toChars());
            scope(exit) printf("[AST.StructDeclaration exit] %s\n", sd.toChars());
        }

        if (isSkippableTemplateInstance(sd))
            return;

        if (cast(void*)sd in visited)
            return;

        visited[cast(void*)sd] = true;
        if (linkage != LINK.c && linkage != LINK.cpp)
        {
            ignored("non-cpp struct %s because of linkage", sd.toChars());
            return;
        }

        const ignoredStash = this.ignoredCounter;
        scope (exit) this.ignoredCounter = ignoredStash;

        pushAlignToBuffer(sd.alignment);

        writeProtection(sd.visibility.kind);

        const structAsClass = sd.cppmangle == CPPMANGLE.asClass;
        if (sd.isUnionDeclaration())
            buf.writestring("union ");
        else
            buf.writestring(structAsClass ? "class " : "struct ");

        writeIdentifier(sd);
        if (!sd.members)
        {
            buf.writestringln(";");
            buf.writenl();
            return;
        }

        buf.writenl();
        buf.writestring("{");

        const protStash = this.currentVisibility;
        this.currentVisibility = structAsClass ? AST.Visibility.Kind.private_ : AST.Visibility.Kind.public_;
        scope (exit) this.currentVisibility = protStash;

        buf.level++;
        buf.writenl();
        auto save = adparent;
        adparent = sd;

        foreach (m; *sd.members)
        {
            m.accept(this);
        }
        // Generate default ctor
        if (!sd.noDefaultCtor && !sd.isUnionDeclaration())
        {
            writeProtection(AST.Visibility.Kind.public_);
            buf.printf("%s()", sd.ident.toChars());
            size_t varCount;
            bool first = true;
            buf.level++;
            foreach (m; *sd.members)
            {
                if (auto vd = m.isVarDeclaration())
                {
                    if (!memberField(vd))
                        continue;
                    varCount++;

                    if (!vd._init && !vd.type.isTypeBasic() && !vd.type.isTypePointer && !vd.type.isTypeStruct &&
                        !vd.type.isTypeClass && !vd.type.isTypeDArray && !vd.type.isTypeSArray)
                    {
                        continue;
                    }
                    if (vd._init && vd._init.isVoidInitializer())
                        continue;

                    if (first)
                    {
                        buf.writestringln(" :");
                        first = false;
                    }
                    else
                    {
                        buf.writestringln(",");
                    }
                    writeIdentifier(vd, true);
                    buf.writeByte('(');

                    if (vd._init)
                    {
                        auto e = AST.initializerToExpression(vd._init);
                        printExpressionFor(vd.type, e, true);
                    }
                    buf.printf(")");
                }
            }
            buf.level--;
            buf.writenl();
            buf.writestringln("{");
            buf.writestringln("}");
            auto ctor = sd.ctor ? sd.ctor.isFuncDeclaration() : null;
            if (varCount && (!ctor || ctor.storage_class & AST.STC.disable))
            {
                buf.printf("%s(", sd.ident.toChars());
                first = true;
                foreach (m; *sd.members)
                {
                    if (auto vd = m.isVarDeclaration())
                    {
                        if (!memberField(vd))
                            continue;
                        if (!first)
                            buf.writestring(", ");
                        assert(vd.type);
                        assert(vd.ident);
                        typeToBuffer(vd.type, vd, true);
                        // Don't print default value for first parameter to not clash
                        // with the default ctor defined above
                        if (!first)
                        {
                            buf.writestring(" = ");
                            if (vd._init && !vd._init.isVoidInitializer())
                                printExpressionFor(vd.type, AST.initializerToExpression(vd._init));
                            else
                                printExpressionFor(vd.type, vd.type.defaultInitLiteral(Loc.initial));
                        }
                        first = false;
                    }
                }
                buf.writestring(") :");
                buf.level++;
                buf.writenl();

                first = true;
                foreach (m; *sd.members)
                {
                    if (auto vd = m.isVarDeclaration())
                    {
                        if (!memberField(vd))
                            continue;

                        if (first)
                            first = false;
                        else
                            buf.writestringln(",");

                        writeIdentifier(vd, true);
                        buf.writeByte('(');
                        writeIdentifier(vd, true);
                        buf.writeByte(')');
                    }
                }
                buf.writenl();
                buf.writestringln("{}");
                buf.level--;
            }
        }

        buf.level--;
        adparent = save;
        buf.writestringln("};");

        popAlignToBuffer(sd.alignment);
        buf.writenl();

        // Workaround because size triggers a forward-reference error
        // for struct templates (the size is undetermined even if the
        // size doesn't depend on the parameters)
        if (!tdparent)
        {
            checkbuf.level++;
            const sn = sd.ident.toChars();
            const sz = sd.size(Loc.initial);
            checkbuf.printf("assert(sizeof(%s) == %llu);", sn, sz);
            checkbuf.writenl();
            checkbuf.level--;
        }
    }

    private void pushAlignToBuffer(uint alignment)
    {
        // DMD ensures alignment is a power of two
        //assert(alignment > 0 && ((alignment & (alignment - 1)) == 0),
        //       "Invalid alignment size");

        // When no alignment is specified, `uint.max` is the default
        // FIXME: alignment is 0 for structs templated members
        if (alignment == STRUCTALIGN_DEFAULT || (tdparent && alignment == 0))
        {
            return;
        }

        buf.printf("#pragma pack(push, %d)", alignment);
        buf.writenl();
    }

    private void popAlignToBuffer(uint alignment)
    {
        if (alignment == STRUCTALIGN_DEFAULT || (tdparent && alignment == 0))
            return;

        buf.writestringln("#pragma pack(pop)");
    }

    private void includeSymbol(AST.Dsymbol ds)
    {
        debug (Debug_DtoH)
        {
            printf("[includeSymbol(AST.Dsymbol) enter] %s\n", ds.toChars());
            scope(exit) printf("[includeSymbol(AST.Dsymbol) exit] %s\n", ds.toChars());
        }
        if (cast(void*) ds in visited)
            return;

        OutBuffer decl;
        decl.doindent = true;
        decl.spaces = true;
        visitAsRoot(ds, &decl);
        donebuf.writestring(decl.peekChars());
    }

    override void visit(AST.ClassDeclaration cd)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.ClassDeclaration enter] %s\n", cd.toChars());
            scope(exit) printf("[AST.ClassDeclaration exit] %s\n", cd.toChars());
        }

        if (isSkippableTemplateInstance(cd))
            return;

        if (cast(void*)cd in visited)
            return;
        visited[cast(void*)cd] = true;
        if (!cd.isCPPclass() && !(tdparent && linkage == LINK.cpp)) // FIXME: ClassKind not set for templated classes?
        {
            ignored("non-cpp class %s", cd.toChars());
            return;
        }

        writeProtection(cd.visibility.kind);

        const classAsStruct = cd.cppmangle == CPPMANGLE.asStruct;
        buf.writestring(classAsStruct ? "struct " : "class ");
        writeIdentifier(cd);

        if (cd.storage_class & AST.STC.final_ || (tdparent && this.storageClass & AST.STC.final_))
            buf.writestring(" final");

        assert(cd.baseclasses);

        foreach (i, base; *cd.baseclasses)
        {
            buf.writestring(i == 0 ? " : public " : ", public ");

            // Base classes/interfaces might depend on template parameters,
            // e.g. class A(T) : B!T { ... }
            if (base.sym is null)
            {
                base.type.accept(this);
            }
            // base.type references onemember for template instances
            // and hence skips the TypeInstance...
            else if (auto ti = base.sym.isInstantiated())
            {
                visitTi(ti);
            }
            else
            {
                buf.writestring(base.sym.toChars());
                includeSymbol(base.sym);
            }
        }

        if (!cd.members)
        {
            buf.writestring(";");
            buf.writenl();
            buf.writenl();
            return;
        }

        buf.writenl();
        buf.writestringln("{");

        const protStash = this.currentVisibility;
        this.currentVisibility = classAsStruct ? AST.Visibility.Kind.public_ : AST.Visibility.Kind.private_;
        scope (exit) this.currentVisibility = protStash;

        auto save = adparent;
        adparent = cd;
        buf.level++;
        foreach (m; *cd.members)
        {
            m.accept(this);
        }
        buf.level--;
        adparent = save;

        buf.writestringln("};");
        buf.writenl();
    }

    override void visit(AST.EnumDeclaration ed)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.EnumDeclaration enter] %s\n", ed.toChars());
            scope(exit) printf("[AST.EnumDeclaration exit] %s\n", ed.toChars());
        }
        if (cast(void*)ed in visited)
            return;

        visited[cast(void*)ed] = true;

        //if (linkage != LINK.c && linkage != LINK.cpp)
        //{
            //ignored("non-cpp enum %s because of linkage\n", ed.toChars());
            //return;
        //}

        if (ed.isSpecial())
        {
            //ignored("%s because it is a special C++ type", ed.toPrettyChars());
            return;
        }

        // we need to know a bunch of stuff about the enum...
        bool isAnonymous = ed.ident is null;
        const isOpaque = !ed.members;
        AST.Type type = ed.memtype;
        if (!type && !isOpaque)
        {
            // check all keys have matching type
            foreach (_m; *ed.members)
            {
                auto m = _m.isEnumMember();
                if (!type)
                    type = m.type;
                else if (m.type !is type)
                {
                    type = null;
                    break;
                }
            }
        }
        EnumKind kind = getEnumKind(type);

        if (isOpaque)
        {
            // Opaque enums were introduced in C++ 11 (workaround?)
            if (global.params.cplusplus < CppStdRevision.cpp11)
            {
                ignored("%s because opaque enums require C++ 11", ed.toPrettyChars());
                return;
            }
            // Opaque enum defaults to int but the type might not be set
            else if (!type)
            {
                kind = EnumKind.Int;
            }
            // Cannot apply namespace workaround for non-integral types
            else if (kind != EnumKind.Int && kind != EnumKind.Numeric)
            {
                ignored("enum %s because of its base type", ed.toPrettyChars());
                return;
            }
        }

        // determine if this is an enum, or just a group of manifest constants
        bool manifestConstants = !isOpaque && (!type || (isAnonymous && kind == EnumKind.Other));
        assert(!manifestConstants || isAnonymous);

        writeProtection(ed.visibility.kind);

        // write the enum header
        if (!manifestConstants)
        {
            if (kind == EnumKind.Int || kind == EnumKind.Numeric)
            {
                buf.writestring("enum");
                // D enums are strong enums, but there exists only a direct mapping
                // with 'enum class' from C++-11 onwards.
                if (global.params.cplusplus >= CppStdRevision.cpp11)
                {
                    if (!isAnonymous)
                    {
                        buf.writestring(" class ");
                        writeIdentifier(ed);
                    }
                    if (kind == EnumKind.Numeric)
                    {
                        buf.writestring(" : ");
                        determineEnumType(type).accept(this);
                    }
                }
                else if (!isAnonymous)
                {
                    buf.writeByte(' ');
                    writeIdentifier(ed);
                }
            }
            else
            {
                buf.writestring("namespace");
                if(!isAnonymous)
                {
                    buf.writeByte(' ');
                    writeIdentifier(ed);
                }
            }
            // Opaque enums have no members, hence skip the body
            if (isOpaque)
            {
                buf.writestringln(";");
                return;
            }
            else
            {
                buf.writenl();
                buf.writestringln("{");
            }
        }

        // emit constant for each member
        if (!manifestConstants)
            buf.level++;

        foreach (_m; *ed.members)
        {
            auto m = _m.isEnumMember();
            AST.Type memberType = type ? type : m.type;
            const EnumKind memberKind = type ? kind : getEnumKind(memberType);

            if (!manifestConstants && (kind == EnumKind.Int || kind == EnumKind.Numeric))
            {
                // C++-98 compatible enums must use the typename as a prefix to avoid
                // collisions with other identifiers in scope.  For consistency with D,
                // the enum member `Type.member` is emitted as `Type_member` in C++-98.
                if (!isAnonymous && global.params.cplusplus < CppStdRevision.cpp11)
                {
                    writeIdentifier(ed);
                    buf.writeByte('_');
                }
                writeIdentifier(m, true);
                buf.writestring(" = ");

                auto ie = cast(AST.IntegerExp)m.value;
                visitInteger(ie.toInteger(), memberType);
                buf.writestring(",");
            }
            else if (global.params.cplusplus >= CppStdRevision.cpp11 &&
                     manifestConstants && (memberKind == EnumKind.Int || memberKind == EnumKind.Numeric))
            {
                buf.writestring("enum : ");
                determineEnumType(memberType).accept(this);
                buf.writestring(" { ");
                writeIdentifier(m, true);
                buf.writestring(" = ");

                auto ie = cast(AST.IntegerExp)m.value;
                visitInteger(ie.toInteger(), memberType);
                buf.writestring(" };");
            }
            else
            {
                buf.writestring("static ");
                auto target = determineEnumType(memberType);
                target.accept(this);
                buf.writestring(" const ");
                writeIdentifier(m, true);
                buf.writestring(" = ");
                printExpressionFor(target, m.origValue);
                buf.writestring(";");
            }
            buf.writenl();
        }

        if (!manifestConstants)
            buf.level--;
        // write the enum tail
        if (!manifestConstants)
            buf.writestring("};");
        buf.writenl();
        buf.writenl();
    }

    override void visit(AST.EnumMember em)
    {
        assert(false, "This node type should be handled in the EnumDeclaration");
    }

    /**
     * Prints a member/parameter/variable declaration into `buf`.
     *
     * Params:
     *   t        = the type (used if `this.origType` is null)
     *   s        = the symbol denoting the identifier
     *   canFixup = whether the identifier may be changed without affecting
     *              binary compatibility (forwarded to `writeIdentifier`)
     */
    private void typeToBuffer(AST.Type t, AST.Dsymbol s, const bool canFixup = false)
    {
        debug (Debug_DtoH)
        {
            printf("[typeToBuffer(AST.Type, AST.Dsymbol) enter] %s sym %s\n", t.toChars(), s.toChars());
            scope(exit) printf("[typeToBuffer(AST.Type, AST.Dsymbol) exit] %s sym %s\n", t.toChars(), s.toChars());
        }

        this.ident = s.ident;
        origType ? origType.accept(this) : t.accept(this);
        if (this.ident)
        {
            buf.writeByte(' ');
            writeIdentifier(s, canFixup);
        }
        this.ident = null;
        if (auto tsa = t.isTypeSArray())
        {
            buf.writeByte('[');
            tsa.dim.accept(this);
            buf.writeByte(']');
        }
    }

    override void visit(AST.Type t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.Type enter] %s\n", t.toChars());
            scope(exit) printf("[AST.Type exit] %s\n", t.toChars());
        }
        printf("Invalid type: %s\n", t.toPrettyChars());
        assert(0);
    }

    override void visit(AST.TypeIdentifier t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeIdentifier enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeIdentifier exit] %s\n", t.toChars());
        }
        if (t.idents.length)
            buf.writestring("typename ");

        writeIdentifier(t.ident, t.loc, "type", tdparent !is null);

        foreach (arg; t.idents)
        {
            buf.writestring("::");

            import dmd.root.rootobject;
            // Is this even possible?
            if (arg.dyncast != DYNCAST.identifier)
            {
                printf("arg.dyncast() = %d\n", arg.dyncast());
                assert(false);
            }
            buf.writestring((cast(Identifier) arg).toChars());
        }
    }

    override void visit(AST.TypeNull t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeNull enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeNull exit] %s\n", t.toChars());
        }
        if (global.params.cplusplus >= CppStdRevision.cpp11)
            buf.writestring("nullptr_t");
        else
            buf.writestring("void*");

    }

    override void visit(AST.TypeTypeof t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeInstance enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeInstance exit] %s\n", t.toChars());
        }
        assert(t.exp);

        if (t.exp.type)
        {
            t.exp.type.accept(this);
        }
        else if (t.exp.isThisExp())
        {
            // Short circuit typeof(this) => <Aggregate name>
            assert(adparent);
            buf.writestring(adparent.ident.toChars());
        }
        else
        {
            // Relying on C++'s typeof might produce wrong results
            // but it's the best we've got here.
            buf.writestring("typeof(");
            t.exp.accept(this);
            buf.writeByte(')');
        }
    }

    override void visit(AST.TypeBasic t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeBasic enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeBasic exit] %s\n", t.toChars());
        }
        if (t.isConst() || t.isImmutable())
            buf.writestring("const ");
        string typeName;
        switch (t.ty)
        {
            case AST.Tvoid:     typeName = "void";      break;
            case AST.Tbool:     typeName = "bool";      break;
            case AST.Tchar:     typeName = "char";      break;
            case AST.Twchar:    typeName = "char16_t";  break;
            case AST.Tdchar:    typeName = "char32_t";  break;
            case AST.Tint8:     typeName = "int8_t";    break;
            case AST.Tuns8:     typeName = "uint8_t";   break;
            case AST.Tint16:    typeName = "int16_t";   break;
            case AST.Tuns16:    typeName = "uint16_t";  break;
            case AST.Tint32:    typeName = "int32_t";   break;
            case AST.Tuns32:    typeName = "uint32_t";  break;
            case AST.Tint64:    typeName = "int64_t";   break;
            case AST.Tuns64:    typeName = "uint64_t";  break;
            case AST.Tfloat32:  typeName = "float";     break;
            case AST.Tfloat64:  typeName = "double";    break;
            case AST.Tfloat80:
                typeName = "_d_real";
                hasReal = true;
                break;
            default:
                //t.print();
                assert(0);
        }
        buf.writestring(typeName);
    }

    override void visit(AST.TypePointer t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypePointer enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypePointer exit] %s\n", t.toChars());
        }
        auto ts = t.next.isTypeStruct();
        if (ts && !strcmp(ts.sym.ident.toChars(), "__va_list_tag"))
        {
            buf.writestring("va_list");
            return;
        }
        t.next.accept(this);
        if (t.next.ty != AST.Tfunction)
            buf.writeByte('*');
        if (t.isConst() || t.isImmutable())
            buf.writestring(" const");
    }

    override void visit(AST.TypeSArray t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeSArray enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeSArray exit] %s\n", t.toChars());
        }
        t.next.accept(this);
    }

    override void visit(AST.TypeAArray t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeAArray enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeAArray exit] %s\n", t.toChars());
        }
        AST.Type.tvoidptr.accept(this);
    }

    override void visit(AST.TypeFunction tf)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeFunction enter] %s\n", tf.toChars());
            scope(exit) printf("[AST.TypeFunction exit] %s\n", tf.toChars());
        }
        tf.next.accept(this);
        buf.writeByte('(');
        buf.writeByte('*');
        if (ident)
            buf.writestring(ident.toChars());
        ident = null;
        buf.writeByte(')');
        buf.writeByte('(');
        foreach (i, fparam; tf.parameterList)
        {
            if (i)
                buf.writestring(", ");
            fparam.accept(this);
        }
        if (tf.parameterList.varargs)
        {
            if (tf.parameterList.parameters.dim && tf.parameterList.varargs == 1)
                buf.writestring(", ");
            buf.writestring("...");
        }
        buf.writeByte(')');
    }

    private void enumToBuffer(AST.EnumDeclaration ed)
    {
        debug (Debug_DtoH)
        {
            printf("[enumToBuffer(AST.EnumDeclaration) enter] %s\n", ed.toChars());
            scope(exit) printf("[enumToBuffer(AST.EnumDeclaration) exit] %s\n", ed.toChars());
        }
        if (ed.isSpecial())
        {
            if (ed.ident == DMDType.c_long)
                buf.writestring("long");
            else if (ed.ident == DMDType.c_ulong)
                buf.writestring("unsigned long");
            else if (ed.ident == DMDType.c_longlong)
                buf.writestring("long long");
            else if (ed.ident == DMDType.c_ulonglong)
                buf.writestring("unsigned long long");
            else if (ed.ident == DMDType.c_long_double)
                buf.writestring("long double");
            else if (ed.ident == DMDType.c_wchar_t)
                buf.writestring("wchar_t");
            else if (ed.ident == DMDType.c_complex_float)
                buf.writestring("_Complex float");
            else if (ed.ident == DMDType.c_complex_double)
                buf.writestring("_Complex double");
            else if (ed.ident == DMDType.c_complex_real)
                buf.writestring("_Complex long double");
            else
            {
                //ed.print();
                assert(0);
            }
            return;
        }

        const kind = getEnumKind(ed.memtype);

        // Check if the enum was emitted as a real enum
        if (kind == EnumKind.Int || kind == EnumKind.Numeric)
            buf.writestring(ed.toChars());
        else
        {
            // Use the base type if the enum was emitted as a namespace
            buf.printf("/* %s */ ", ed.ident.toChars());
            ed.memtype.accept(this);
        }
    }

    override void visit(AST.TypeEnum t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeEnum enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeEnum exit] %s\n", t.toChars());
        }
        if (cast(void*)t.sym !in forwarded)
        {
            forwarded[cast(void*)t.sym] = true;
            //printf("Visiting enum %s from module %s %s\n", t.sym.toPrettyChars(), t.toChars(), t.sym.loc.toChars());
            visitAsRoot(t.sym, fwdbuf);
        }
        if (t.isConst() || t.isImmutable())
            buf.writestring("const ");
        enumToBuffer(t.sym);
    }

    override void visit(AST.TypeStruct t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeStruct enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeStruct exit] %s\n", t.toChars());
        }
        if (cast(void*)t.sym !in forwarded && !t.sym.parent.isTemplateInstance())
        {
            forwarded[cast(void*)t.sym] = true;
            fwdbuf.writestring(t.sym.isUnionDeclaration() ? "union " : "struct ");
            fwdbuf.writestring(t.sym.toChars());
            fwdbuf.writestringln(";");
        }

        if (t.isConst() || t.isImmutable())
            buf.writestring("const ");
        if (auto ti = t.sym.parent.isTemplateInstance())
        {
            visitTi(ti);
            return;
        }
        buf.writestring(t.sym.toChars());
    }

    override void visit(AST.TypeDArray t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeDArray enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeDArray exit] %s\n", t.toChars());
        }
        if (t.isConst() || t.isImmutable())
            buf.writestring("const ");
        buf.writestring("_d_dynamicArray< ");
        t.next.accept(this);
        buf.writestring(" >");
    }

    override void visit(AST.TypeInstance t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeInstance enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeInstance exit] %s\n", t.toChars());
        }
        visitTi(t.tempinst);
    }

    private void visitTi(AST.TemplateInstance ti)
    {
        debug (Debug_DtoH)
        {
            printf("[visitTi(AST.TemplateInstance) enter] %s\n", ti.toChars());
            scope(exit) printf("[visitTi(AST.TemplateInstance) exit] %s\n", ti.toChars());
        }


        foreach (o; *ti.tiargs)
        {
            if (!AST.isType(o))
                return;
        }
        buf.writestring(ti.name.toChars());
        buf.writeByte('<');
        foreach (i, o; *ti.tiargs)
        {
            if (i)
                buf.writestring(", ");
            if (auto tt = AST.isType(o))
            {
                tt.accept(this);
            }
            else
            {
                //ti.print();
                //o.print();
                assert(0);
            }
        }
        buf.writestring(" >");
    }

    override void visit(AST.TemplateDeclaration td)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TemplateDeclaration enter] %s\n", td.toChars());
            scope(exit) printf("[AST.TemplateDeclaration exit] %s\n", td.toChars());
        }
        if (cast(void*)td in visited)
            return;
        visited[cast(void*)td] = true;

        if (!td.parameters || !td.onemember || (!td.onemember.isStructDeclaration && !td.onemember.isClassDeclaration && !td.onemember.isFuncDeclaration))
        {
            visit(cast(AST.Dsymbol)td);
            return;
        }

        // Explicitly disallow templates with non-type parameters or specialization.
        foreach (p; *td.parameters)
        {
            if (!p.isTemplateTypeParameter() || p.specialization())
            {
                visit(cast(AST.Dsymbol)td);
                return;
            }
        }

        if (linkage != LINK.c && linkage != LINK.cpp)
        {
            ignored("template %s because of linkage", td.toPrettyChars());
            return;
        }

        auto save = tdparent;
        tdparent = td;
        const bookmark = buf.length;
        printTemplateParams(td);

        const oldIgnored = this.ignoredCounter;
        td.onemember.accept(this);

        // Remove "template<...>" if the symbol could not be emitted
        if (oldIgnored != this.ignoredCounter)
            buf.setsize(bookmark);

        tdparent = save;
    }

    private void printTemplateParams(const AST.TemplateDeclaration td)
    {
        buf.writestring("template <");
        bool first = true;
        foreach (p; *td.parameters)
        {
            if (first)
                first = false;
            else
                buf.writestring(", ");
            buf.writestring("typename ");
            writeIdentifier(p.ident, p.loc, "template parameter", true);
        }
        buf.writestringln(">");
    }

    override void visit(AST.TypeClass t)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.TypeClass enter] %s\n", t.toChars());
            scope(exit) printf("[AST.TypeClass exit] %s\n", t.toChars());
        }
        if (cast(void*)t.sym !in forwarded)
        {
            forwarded[cast(void*)t.sym] = true;
            fwdbuf.writestring("class ");
            fwdbuf.writestring(t.sym.toChars());
            fwdbuf.writestringln(";");
        }

        if (t.isConst() || t.isImmutable())
            buf.writestring("const ");
        buf.writestring(t.sym.toChars());
        buf.writeByte('*');
        if (t.isConst() || t.isImmutable())
            buf.writestring(" const");
    }

    /**
     * Writes the function signature to `buf`.
     *
     * Params:
     *   fd     = the function to print
     *   tf     = fd's type
     */
    private void funcToBuffer(AST.TypeFunction tf, AST.FuncDeclaration fd)
    {
        debug (Debug_DtoH)
        {
            printf("[funcToBuffer(AST.TypeFunction) enter] %s\n", fd.toChars());
            scope(exit) printf("[funcToBuffer(AST.TypeFunction) exit] %s\n", fd.toChars());
        }

        auto originalType = cast(AST.TypeFunction)fd.originalType;

        if (fd.isCtorDeclaration() || fd.isDtorDeclaration())
        {
            if (fd.isDtorDeclaration())
            {
                buf.writeByte('~');
            }
            buf.writestring(adparent.toChars());
            if (!tf)
            {
                assert(fd.isDtorDeclaration());
                buf.writestring("()");
                return;
            }
        }
        else
        {
            import dmd.root.string : toDString;
            assert(tf.next, fd.loc.toChars().toDString());

            tf.next == AST.Type.tsize_t ? originalType.next.accept(this) : tf.next.accept(this);
            if (tf.isref)
                buf.writeByte('&');
            buf.writeByte(' ');
            writeIdentifier(fd);
        }

        buf.writeByte('(');
        foreach (i, fparam; tf.parameterList)
        {
            if (i)
                buf.writestring(", ");
            if (fparam.type == AST.Type.tsize_t && originalType)
            {
                fparam = originalType.parameterList[i];
            }
            fparam.accept(this);
        }
        if (tf.parameterList.varargs)
        {
            if (tf.parameterList.parameters.dim && tf.parameterList.varargs == 1)
                buf.writestring(", ");
            buf.writestring("...");
        }
        buf.writeByte(')');
    }

    override void visit(AST.Parameter p)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.Parameter enter] %s\n", p.toChars());
            scope(exit) printf("[AST.Parameter exit] %s\n", p.toChars());
        }
        ident = p.ident;
        p.type.accept(this);
        if (p.storageClass & AST.STC.ref_)
            buf.writeByte('&');
        buf.writeByte(' ');
        if (ident)
            // FIXME: Parameter is missing a Loc
            writeIdentifier(ident, Loc.initial, "parameter", true);
        ident = null;

        if (p.defaultArg)
        {
            //printf("%s %d\n", p.defaultArg.toChars, p.defaultArg.op);
            buf.writestring(" = ");
            printExpressionFor(p.type, p.defaultArg);
        }
    }

    /**
     * Prints `exp` as an expression of type `target` while inserting
     * appropriate code when implicit conversion does not translate
     * directly to C++, e.g. from an enum to its base type.
     *
     * Params:
     *   target = the type `exp` is converted to
     *   exp    = the expression to print
     *   isCtor = if `exp` is a ctor argument
     */
    private void printExpressionFor(AST.Type target, AST.Expression exp, const bool isCtor = false)
    {
        /// Determines if a static_cast is required
        static bool needsCast(AST.Type target, AST.Expression exp)
        {
            // import std.stdio;
            // writefln("%s:%s: target = %s, type = %s (%s)", exp.loc.linnum, exp.loc.charnum, target, exp.type, exp.op);

            auto source = exp.type;

            // DotVarExp resolve conversions, e.g from an enum to its base type
            if (auto dve = exp.isDotVarExp())
                source = dve.var.type;

            if (!source)
                // Defensively assume that the cast is required
                return true;

            // Conversions from enum class to base type require static_cast
            if (global.params.cplusplus >= CppStdRevision.cpp11 &&
                source.isTypeEnum && !target.isTypeEnum)
                return true;

            return false;
        }

        // Slices are emitted as a special struct, hence we need to fix up
        // any expression initialising a slice variable/member
        if (auto ta = target.isTypeDArray())
        {
            if (exp.isNullExp())
            {
                if (isCtor)
                {
                    // Don't emit, use default ctor
                }
                else if (global.params.cplusplus >= CppStdRevision.cpp11)
                {
                    // Prefer initializer list
                    buf.writestring("{}");
                }
                else
                {
                    // Write __d_dynamic_array<TYPE>()
                    visit(ta);
                    buf.writestring("()");
                }
                return;
            }

            if (auto se = exp.isStringExp())
            {
                // Rewrite as <length> + <literal> pair optionally
                // wrapped in a initializer list/ctor call

                const initList = global.params.cplusplus >= CppStdRevision.cpp11;
                if (!isCtor)
                {
                    if (initList)
                        buf.writestring("{ ");
                    else
                    {
                        visit(ta);
                        buf.writestring("( ");
                    }
                }

                buf.printf("%zu, ", se.len);
                visit(se);

                if (!isCtor)
                    buf.writestring(initList ? " }" : " )");

                return;
            }
        }
        else if (needsCast(target, exp))
        {
            buf.writestring("static_cast<");
            target.accept(this);
            buf.writestring(">(");
            exp.accept(this);
            buf.writeByte(')');
        }
        else
        {
            exp.accept(this);
        }
    }

    override void visit(AST.Expression e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.Expression enter] %s\n", e.toChars());
            scope(exit) printf("[AST.Expression exit] %s\n", e.toChars());
        }
        // Valid in most cases, others should be overriden below
        // to use the appropriate operators  (:: and ->)
        buf.writestring(e.toString());
    }

    override void visit(AST.VarExp e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.VarExp enter] %s\n", e.toChars());
            scope(exit) printf("[AST.VarExp exit] %s\n", e.toChars());
        }

        /// Partially prints the FQN including parent aggregates
        void printPrefix(AST.Dsymbol var)
        {
            if (!var || !var.isAggregateDeclaration())
                return;
            printPrefix(var.parent);
            includeSymbol(var);

            buf.writestring(var.toString());
            buf.writestring("::");
        }

        // Static members are not represented as DotVarExp, hence
        // manually print the full name here
        if (e.var.storage_class & AST.STC.static_)
            printPrefix(e.var.parent);

        includeSymbol(e.var);
        writeIdentifier(e.var, !!e.var.isThis());
    }

    override void visit(AST.CallExp e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.CallExp enter] %s\n", e.toChars());
            scope(exit) printf("[AST.CallExp exit] %s\n", e.toChars());
        }

        // Dereferencing function pointers requires additional braces: (*f)(args)
        const isFp = e.e1.isPtrExp();
        if (isFp)
            buf.writeByte('(');
        else if (e.f)
            includeSymbol(e.f);

        e.e1.accept(this);

        if (isFp) buf.writeByte(')');

        assert(e.arguments);
        buf.writeByte('(');
        foreach (i, arg; *e.arguments)
        {
            if (i)
                buf.writestring(", ");
            arg.accept(this);
        }
        buf.writeByte(')');
    }

    override void visit(AST.DotVarExp e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.DotVarExp enter] %s\n", e.toChars());
            scope(exit) printf("[AST.DotVarExp exit] %s\n", e.toChars());
        }

        // Accessing members through a pointer?
        if (auto pe = e.e1.isPtrExp)
        {
            pe.e1.accept(this);
            buf.writestring("->");
        }
        else
        {
            e.e1.accept(this);
            buf.writeByte('.');
        }

        // Should only be used to access non-static members
        assert(e.var.isThis());

        writeIdentifier(e.var, true);
    }

    override void visit(AST.DotIdExp e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.DotIdExp enter] %s\n", e.toChars());
            scope(exit) printf("[AST.DotIdExp exit] %s\n", e.toChars());
        }

        e.e1.accept(this);
        buf.writestring("::");
        buf.writestring(e.ident.toChars());
    }

    override void visit(AST.NullExp e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.NullExp enter] %s\n", e.toChars());
            scope(exit) printf("[AST.NullExp exit] %s\n", e.toChars());
        }
        if (global.params.cplusplus >= CppStdRevision.cpp11)
            buf.writestring("nullptr");
        else
            buf.writestring("NULL");
    }

    override void visit(AST.ArrayLiteralExp e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.ArrayLiteralExp enter] %s\n", e.toChars());
            scope(exit) printf("[AST.ArrayLiteralExp exit] %s\n", e.toChars());
        }
        buf.writestring("arrayliteral");
    }

    override void visit(AST.StringExp e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.StringExp enter] %s\n", e.toChars());
            scope(exit) printf("[AST.StringExp exit] %s\n", e.toChars());
        }

        if (e.sz == 2)
            buf.writeByte('u');
        else if (e.sz == 4)
            buf.writeByte('U');
        buf.writeByte('"');

        for (size_t i = 0; i < e.len; i++)
        {
            uint c = e.charAt(i);
            switch (c)
            {
                case '"':
                case '\\':
                    buf.writeByte('\\');
                    goto default;
                default:
                    if (c <= 0xFF)
                    {
                        if (c >= 0x20 && c < 0x80)
                            buf.writeByte(c);
                        else
                            buf.printf("\\x%02x", c);
                    }
                    else if (c <= 0xFFFF)
                        buf.printf("\\u%04x", c);
                    else
                        buf.printf("\\U%08x", c);
                    break;
            }
        }
        buf.writeByte('"');
    }

    override void visit(AST.RealExp e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.RealExp enter] %s\n", e.toChars());
            scope(exit) printf("[AST.RealExp exit] %s\n", e.toChars());
        }

        import dmd.root.ctfloat : CTFloat;

        // Special case NaN and Infinity because floatToBuffer
        // uses D literals (`nan` and `infinity`)
        if (CTFloat.isNaN(e.value))
        {
            buf.writestring("NAN");
        }
        else if (CTFloat.isInfinity(e.value))
        {
            if (e.value < 0)
                buf.writeByte('-');
            buf.writestring("INFINITY");
        }
        else
        {
            import dmd.hdrgen;
            // Hex floating point literals were introduced in C++ 17
            const allowHex = global.params.cplusplus >= CppStdRevision.cpp17;
            floatToBuffer(e.type, e.value, buf, allowHex);
        }
    }

    override void visit(AST.IntegerExp e)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.IntegerExp enter] %s\n", e.toChars());
            scope(exit) printf("[AST.IntegerExp exit] %s\n", e.toChars());
        }
        visitInteger(e.toInteger, e.type);
    }

    private void visitInteger(dinteger_t v, AST.Type t)
    {
        debug (Debug_DtoH)
        {
            printf("[visitInteger(AST.Type) enter] %s\n", t.toChars());
            scope(exit) printf("[visitInteger(AST.Type) exit] %s\n", t.toChars());
        }
        switch (t.ty)
        {
            case AST.Tenum:
                auto te = cast(AST.TypeEnum)t;
                buf.writestring("(");
                enumToBuffer(te.sym);
                buf.writestring(")");
                visitInteger(v, te.sym.memtype);
                break;
            case AST.Tbool:
                buf.writestring(v ? "true" : "false");
                break;
            case AST.Tint8:
                buf.printf("%d", cast(byte)v);
                break;
            case AST.Tuns8:
                buf.printf("%uu", cast(ubyte)v);
                break;
            case AST.Tint16:
                buf.printf("%d", cast(short)v);
                break;
            case AST.Tuns16:
            case AST.Twchar:
                buf.printf("%uu", cast(ushort)v);
                break;
            case AST.Tint32:
            case AST.Tdchar:
                buf.printf("%d", cast(int)v);
                break;
            case AST.Tuns32:
                buf.printf("%uu", cast(uint)v);
                break;
            case AST.Tint64:
                buf.printf("%lldLL", v);
                break;
            case AST.Tuns64:
                buf.printf("%lluLLU", v);
                break;
            case AST.Tchar:
                if (v > 0x20 && v < 0x80)
                    buf.printf("'%c'", cast(int)v);
                else
                    buf.printf("%uu", cast(ubyte)v);
                break;
            default:
                //t.print();
                assert(0);
        }
    }

    override void visit(AST.StructLiteralExp sle)
    {
        debug (Debug_DtoH)
        {
            printf("[AST.StructLiteralExp enter] %s\n", sle.toChars());
            scope(exit) printf("[AST.StructLiteralExp exit] %s\n", sle.toChars());
        }
        sle.sd.type.accept(this);
        buf.writeByte('(');
        foreach(i, e; *sle.elements)
        {
            if (i)
                buf.writestring(", ");
            printExpressionFor(sle.sd.fields[i].type, e);
        }
        buf.writeByte(')');
    }

    static if (__VERSION__ < 2092)
    {
        private void ignored(const char* format, ...) nothrow
        {
            this.ignoredCounter++;

            import core.stdc.stdarg;
            if (!printIgnored)
                return;

            va_list ap;
            va_start(ap, format);
            buf.writestring("// Ignored ");
            buf.vprintf(format, ap);
            buf.writenl();
            va_end(ap);
        }
    }
    else
    {
        // Writes a formatted message into `buf` if `printIgnored` is true.
        pragma(printf)
        private void ignored(const char* format, ...) nothrow
        {
            this.ignoredCounter++;

            import core.stdc.stdarg;
            if (!printIgnored)
                return;

            va_list ap;
            va_start(ap, format);
            buf.writestring("// Ignored ");
            buf.vprintf(format, ap);
            buf.writenl();
            va_end(ap);
        }
    }

    /** Checks whether s is a template instance that should not be emitted **/
    private bool isSkippableTemplateInstance(AST.Dsymbol s)
    {
        auto td = s.isInstantiated();
        return td && td !is tdparent;
    }
}
