/*
 * Copyright (C) 1995-2008 University of Karlsruhe.  All right reserved.
 *
 * This file is part of libFirm.
 *
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file LICENSE.GPL included in the
 * packaging of this file.
 *
 * Licensees holding valid libFirm Professional Edition licenses may use
 * this file in accordance with the libFirm Commercial License.
 * Agreement provided with the Software.
 *
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */

/**
 * @file
 * @brief   Extended basis block support.
 * @author  Michael Beck
 * @date    5.2005
 */
#ifndef FIRM_ANA_IREXTBB_H
#define FIRM_ANA_IREXTBB_H

#include "firm_types.h"
#include "execfreq.h"
#include "begin.h"

/* type of callback function for ir_graph walk */
#ifndef _EXTBB_WALK_FUNC_TYPEDEF_
#define _EXTBB_WALK_FUNC_TYPEDEF_
/**
 * The type of a walk function.  Does not use the link field.
 *
 * @param blk  - the extended basic block that is just visited
 * @param env  - an environment pointer passed by the walk functions
 */
typedef void extbb_walk_func(ir_extblk *blk, void *env);
#endif

/**
 * Checks whether a pointer points to a extended basic block.
 * Intern version for libFirm.
 */
FIRM_API int is_ir_extbb(const void *thing);

/**
 * Compute the extended basic blocks for a graph.
 */
FIRM_API void compute_extbb(ir_graph *irg);

/**
 * Compute the extended basic blocks for a graph based on execution frequencies.
 */
FIRM_API void compute_extbb_execfreqs(ir_graph *irg, ir_exec_freq *execfreqs);

/**
 * free all extended block info.
 */
FIRM_API void free_extbb(ir_graph *irg);

/**
 * Returns the extended block of a node.
 *
 * @param node  the node
 */
FIRM_API ir_extblk *get_nodes_extbb(const ir_node *node);

/**
 * Returns the visited counter of an extended block.
 *
 * @param blk  the extended basic block
 */
FIRM_API ir_visited_t get_extbb_visited(const ir_extblk *blk);

/**
 * Sets the visited counter of an extended block.
 *
 * @param blk      the extended basic block
 * @param visited  new value for the visited counter
 */
FIRM_API void set_extbb_visited(ir_extblk *blk, ir_visited_t visited);

/**
 * Mark an extended block as visited in a graph.
 * Uses the block visit flag.
 *
 * @param blk  the extended basic block
 */
FIRM_API void mark_extbb_visited(ir_extblk *blk);

/**
 * Returns non-zero if an extended was visited.
 * Uses the block visit flag.
 *
 * @param blk  the extended basic block
 */
FIRM_API int extbb_visited(const ir_extblk *blk);

/**
 * Returns non-zero if an extended block was NOT visited.
 * Uses the block visit flag.
 *
 * @param blk  the extended basic block
 */
FIRM_API int extbb_not_visited(const ir_extblk *blk);

/**
 * Returns the link field of an extended block.
 *
 * @param blk  the extended basic block
 */
FIRM_API void *get_extbb_link(const ir_extblk *blk);

/**
 * Sets the link field of an extended block.
 *
 * @param blk  the extended basic block
 * @param link the new link value
 */
FIRM_API void set_extbb_link(ir_extblk *blk, void *link);

/**
 * Returns the number of basic blocks of an extended block.
 *
 * @param blk  the extended basic block
 */
FIRM_API int get_extbb_n_blocks(const ir_extblk *blk);

/**
 * Returns the i'th basic block of an extended block.
 *
 * @param blk  the extended basic block
 * @param pos  the position
 */
FIRM_API ir_node *get_extbb_block(const ir_extblk *blk, int pos);

/**
 * Returns the leader basic block of an extended block.
 *
 * @param blk  the extended basic block
 */
FIRM_API ir_node *get_extbb_leader(const ir_extblk *blk);

/**
 * Returns the node number of an extended block.
 * Its the block number of the leader block
 *
 * @param blk  the extended basic block
 */
FIRM_API long get_extbb_node_nr(const ir_extblk *blk);

/**
 * Walks only over Extended Basic Block nodes in the graph.
 *
 * @param blk  - the start extended block node
 * @param pre  - walker function, executed before the predecessor of a node are
 *               visited
 * @param post - walker function, executed after the predecessor of a node are
 *               visited
 * @param env  - environment, passed to pre and post
 *
 * This function Walks only over Block nodes in the graph. Has its own visited
 * flag, so that it can be interleaved with the other walker.
 * If a none block is passed, starts at the block this node belongs to.
 * If end is passed also visits kept alive blocks. Does not use the link field.
 */
FIRM_API void irg_extblock_walk(ir_extblk *blk, extbb_walk_func *pre,
                                extbb_walk_func *post, void *env);

/**
 * Walks only over reachable Extended Basic Block nodes in the graph.
 * Ensures, that the extended block containing the End node is visited last
 * and the block containing Start visited first (in post order).
 *
 * @param irg  - the irg graph
 * @param pre  - walker function, executed before the predecessor of a block
 *               are visited
 * @param post - walker function, executed after the predecessor of a block
 *               are visited
 * @param env  - environment, passed to pre and post
 */
FIRM_API void irg_extblock_walk_graph(ir_graph *irg, extbb_walk_func *pre,
                                      extbb_walk_func *post, void *env);

#include "end.h"

#endif
