package main

import (
	"flag"
	"fmt"
	"os"
	"path/filepath"
	"strings"
	"text/template"

	"github.com/hashicorp/packer/fix"
)

var deprecatedOptsTemplate = template.Must(template.New("deprecatedOptsTemplate").
	Parse(`//<!-- Code generated by generate-fixer-deprecations; DO NOT EDIT MANUALLY -->

package config

var DeprecatedOptions = []string{
{{- range .DeprecatedOpts}}
	"{{.}}",
{{- end}}
}
`))

type executeOpts struct {
	DeprecatedOpts []string
}

func main() {
	// Figure out location in directory structure
	args := flag.Args()
	if len(args) == 0 {
		// Default: process the file
		args = []string{os.Getenv("GOFILE")}
	}
	fname := args[0]

	absFilePath, err := filepath.Abs(fname)
	if err != nil {
		panic(err)
	}
	paths := strings.SplitAfter(absFilePath, "packer"+string(os.PathSeparator))
	packerDir := paths[0]

	// Load all deprecated options from all active fixers
	allDeprecatedOpts := []string{}
	for _, name := range fix.FixerOrder {
		fixer, ok := fix.Fixers[name]
		if !ok {
			panic("fixer not found: " + name)
		}

		deprecated := fixer.DeprecatedOptions()
		allDeprecatedOpts = append(allDeprecatedOpts, deprecated...)
	}

	deprecated_path := filepath.Join(packerDir, "helper", "config",
		"deprecated_options.go")

	outputFile, err := os.Create(deprecated_path)
	if err != nil {
		panic(err)
	}
	defer outputFile.Close()

	deprecated := &executeOpts{DeprecatedOpts: allDeprecatedOpts}
	err = deprecatedOptsTemplate.Execute(outputFile, deprecated)
	if err != nil {
		fmt.Printf("%v", err)
		os.Exit(1)
	}
}
