#!/bin/bash
#
# Copyright 2016 The Bazel Authors. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# discard_graph_edges_test.sh: basic tests for the --discard_graph_edges flag.

# Load the test setup defined in the parent directory
CURRENT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "${CURRENT_DIR}/../integration_test_setup.sh" \
  || { echo "integration_test_setup.sh not found!" >&2; exit 1; }

#### SETUP #############################################################

set -e

function set_up() {
  mkdir -p testing || fail "Couldn't create directory"
  echo "cc_test(name='mytest', srcs=['mytest.cc'], malloc=':system_malloc')" > testing/BUILD || fail
  echo "cc_library(name='system_malloc', srcs=[])"                           >> testing/BUILD || fail
  echo "int main() {return 0;}"         > testing/mytest.cc || fail
}

STARTUP_FLAGS="--batch"
BUILD_FLAGS="--keep_going --discard_analysis_cache"

#### TESTS #############################################################

function test_build() {
  bazel $STARTUP_FLAGS build $BUILD_FLAGS //testing:mytest >& $TEST_log \
    || fail "Expected success"
}

function test_test() {
  bazel $STARTUP_FLAGS test $BUILD_FLAGS //testing:mytest >& $TEST_log \
    || fail "Expected success"
}

# bazel info inherits from bazel build, but it doesn't have much in common with it.
function test_info() {
  bazel $STARTUP_FLAGS info $BUILD_FLAGS >& $TEST_log || fail "Expected success"
}

function test_empty_build() {
  bazel $STARTUP_FLAGS build $BUILD_FLAGS >& $TEST_log || fail "Expected success"
}

function test_query() {
  bazel $STARTUP_FLAGS query 'somepath(//testing:mytest,//testing:system_malloc)' >& $TEST_log \
    || fail "Expected success"
  expect_log "//testing:mytest"
  expect_log "//testing:system_malloc"
}

# Action conflicts can cause deletion of nodes, and deletion is tricky with no edges.
function test_action_conflict() {
  mkdir -p conflict || fail "Couldn't create directory"
  cat > conflict/BUILD <<EOF || fail "Couldn't create BUILD file"
cc_library(name='x', srcs=['foo.cc'])
cc_binary(name='_objs/x/conflict/foo.pic.o', srcs=['bar.cc'])
cc_binary(name='foo', deps=['x'], data=['_objs/x/conflict/foo.pic.o'])
EOF
  touch conflict/foo.cc || fail
  touch conflict/bar.cc || fail

  # --nocache_test_results to make log-grepping easier.
  bazel $STARTUP_FLAGS test $BUILD_FLAGS --nocache_test_results //conflict:foo //testing:mytest >& $TEST_log \
    && fail "Expected failure"
  exit_code=$?
  [ $exit_code -eq 1 ] || fail "Wrong exit code: $exit_code"
  expect_log "is generated by these conflicting actions"
  expect_not_log "Graph edges not stored"
  expect_log "mytest *PASSED"
}

# The following tests are not expected to exercise codepath -- make sure nothing bad happens.

function test_no_batch() {
  bazel $STARTUP_FLAGS --nobatch test $BUILD_FLAGS //testing:mytest >& $TEST_log \
    || fail "Expected success"
}

function test_no_keep_going() {
  bazel $STARTUP_FLAGS test $BUILD_FLAGS --nokeep_going //testing:mytest >& $TEST_log \
    || fail "Expected success"
}

function test_no_discard_analysis_cache() {
  bazel $STARTUP_FLAGS test $BUILD_FLAGS --nodiscard_analysis_cache //testing:mytest >& $TEST_log \
    || fail "Expected success"
}

run_suite "test for --discard_graph_edges"
