---
author: Christopher Chedeau (@vjeux)
authorURL: https://twitter.com/vjeux
title: "Prettier 1.5: GraphQL, CSS-in-JS & JSON"
---

This release adds GraphQL formatting support, CSS-in-JS (including styled-components), and JSON support to Prettier.

<!--truncate-->

This is the release I've been waiting for a very long time: one that has only minimal changes to JavaScript!

For the past 6 months, we kept doing changes to various aspects of printing JavaScript, with the hope that one day we would get to a stable place. No automated tool is going to print perfect code for all the possible edge cases. The goal is to find a good place where when people report code that is printed in a funny way, we can't make it better without making other pieces of code look worse, introduce behavior that's very hard to understand for humans and doesn't introduce some disproportionate complexity to the codebase.

We're not 100% there yet, but we're closer than ever!

Now that JavaScript needs for support is trending down, it's an opportunity to support other languages that front-end developers are working on and want formatted. We've introduced TypeScript and CSS in the last release and are doing a batch of fixes for them in this release. We're also adding support for new languages: GraphQL queries, embedding CSS-in-JS and JSON are now available in prettier!

#### Blog Post: [Adding a new layout strategy to Prettier](https://medium.com/geckoboard-under-the-hood/adding-a-new-layout-strategy-to-prettier-8d33084c0f99) by [@karl](https://github.com/karl)

Prettier is not only a useful tool but it's also a really cool piece of technology. [@karl](https://github.com/karl) spent a bunch of time improving JSX support and in the process implemented a new primitive to prettier: `fill`. He wrote a very interesting blog post [Adding a new layout strategy to Prettier](https://medium.com/geckoboard-under-the-hood/adding-a-new-layout-strategy-to-prettier-8d33084c0f99) that I highly recommend reading if you're interested in how prettier is working behind the scenes.

### GraphQL

Thanks to [@stubailo](https://github.com/stubailo), [@jnwng](https://github.com/jnwng), [@tgriesser](https://github.com/tgriesser) and [@azz](https://github.com/azz), prettier now supports printing GraphQL queries!

It works for `.graphql` files and within JavaScript templates that start with `graphql`, `graphql.experimental` and `gql` in order to work with [Relay](https://facebook.github.io/relay/) and [Apollo](https://www.apollodata.com/).

<!-- prettier-ignore -->
```jsx
ReactDOM.render(
  <QueryRenderer
    query={graphql`
      query appQuery {
        viewer {
          ...TodoApp_viewer
        }
      }
    `}
    // ...
  />,
  mountNode
);
```

Note that it only supports the open source syntax of GraphQL, therefore doesn't work with Relay Classic, it only works with Relay Modern.

### CSS-in-JS

If you are using [styled-components](https://github.com/styled-components/styled-components) or [styled-jsx](https://github.com/zeit/styled-jsx), prettier will now reformat the CSS inside of your template expressions. Thanks to [@pomber](https://github.com/pomber) for the awesome work!

<!-- prettier-ignore -->
```js
const EqualDivider = styled.div`
  margin: 0.5rem;
  padding: 1rem;
  background: papayawhip;
  > * {
    flex: 1;
    &:not(:first-child) {
      ${props => (props.vertical ? "margin-top" : "margin-left")}: 1rem;
    }
  }
`;
```

### JSON

This was pretty straightforward to implement but nonetheless very useful. Thanks to [@josephfrazier](https://github.com/josephfrazier) for doing it :)

<!-- prettier-ignore -->
```json
{
  "name": "prettier",
  "version": "1.5.0",
  "description": "Prettier is an opinionated JavaScript formatter",
  "bin": {
    "prettier": "./bin/prettier.js"
  }
}
```

### CSS

I'm really excited because we only put a few days to build the initial CSS support and it has worked surprisingly well. This release brings a handful of important improvements to CSS but nothing that required big changes.

#### CSS: Every selector is now printed in its own line ([#2047](https://github.com/prettier/prettier/pull/2047)) by [@yuchi](https://github.com/yuchi)

The biggest unknown when printing CSS was how to deal with multiple selectors. The initial approach we took was to use the 80 columns rule where we would only split if it was bigger than that. Many people reported that they were using another strategy for this: always break after a `,`. It turns out that many popular codebases are using this approach and it feels good as you can see the structure of the selectors when layed out on-top of each others.

<!-- prettier-ignore -->
```css
// Before
.clusterPlannerDialog input[type="text"], .clusterPlannerDialog .uiTypeahead {
  color: #333;
}

// After
.clusterPlannerDialog input[type="text"],
.clusterPlannerDialog .uiTypeahead {
  color: #333;
}
```

#### CSS: lowercase hex colors ([#2203](https://github.com/prettier/prettier/pull/2203)) by [@azz](https://github.com/azz)

The concept of code formatting has blurry boundaries. The core aspect of it is around whitespaces but some things like single vs double quotes and semi-colons are usually bundled with it. With prettier on JavaScript, we also lightly reformat strings by removing extra `\` and normalize numbers. For CSS, we need to do a similar interpretation of where the boundary ends. For colors, we decided to turn all the letters into lowercase and stop there. Turning rgb() into hex or 6 hex into 3 hex is out of scope.

<!-- prettier-ignore -->
```css
// Before
.foo {
  color: #AAA;
  -o-color: #fabcd3;
  -ms-color: #AABBCC;
}

// After
.foo {
  color: #aa;
  -o-color: #fabcd3;
  -ms-color: #aabbcc;
}
```

#### CSS: Use fill for CSS values ([#2224](https://github.com/prettier/prettier/pull/2224))

The new fill primitive turned out to be very useful for CSS. For long values, instead of breaking and putting a `\n` before every element, we can instead only put a `\n` when it goes over the limit. It leads to much better looking code.

<!-- prettier-ignore -->
```scss
// Before
foo {
  border-left:
    1px
    solid
    mix($warningBackgroundColors, $warningBorderColors, 50%);
}

// After
foo {
  border-left: 1px solid
    mix($warningBackgroundColors, $warningBorderColors, 50%);
}
```

#### CSS: Allow long media rules to break ([#2219](https://github.com/prettier/prettier/pull/2219))

This is another small fix in the journey of properly supporting a new language. We now encode the ability to break on long `@media` rules.

<!-- prettier-ignore -->
```css
// Before
@media all and (-webkit-min-device-pixel-ratio: 1.5), all and (-o-min-device-pixel-ratio: 3/2), all and (min--moz-device-pixel-ratio: 1.5), all and (min-device-pixel-ratio: 1.5) {
}

// After
@media all and (-webkit-min-device-pixel-ratio: 1.5),
  all and (-o-min-device-pixel-ratio: 3/2),
  all and (min--moz-device-pixel-ratio: 1.5),
  all and (min-device-pixel-ratio: 1.5) {
}
```

#### CSS: Print @else on same line as } ([#2088](https://github.com/prettier/prettier/pull/2088)) by [@azz](https://github.com/azz)

Less and SCSS are turning into real programming languages :) Step by step, we're starting to print all their constructs in the same way as JavaScript. This time, it's the `else` placement.

<!-- prettier-ignore -->
```scss
// Before
@if $media == phonePortrait {
  $k: .15625;
}
@else if $media == tabletPortrait {
  $k: .065106;
}

// After
@if $media == phonePortrait {
  $k: .15625;
} @else if $media == tabletPortrait {
  $k: .065106;
}
```

#### CSS: implement prettier-ignore ([#2089](https://github.com/prettier/prettier/pull/2089)) by [@azz](https://github.com/azz)

While we want prettier to format the entire codebase, there are times where we "know better" and want an escape hatch. This is where the `prettier-ignore` comment comes in. It wasn't working for CSS but that was an oversight, now it is implemented :)

<!-- prettier-ignore -->
```css
// Before
foo {
  /* prettier-ignore */
  thing: foo;
  -ms-thing: foo;
}

// After
foo {
  /* prettier-ignore */
  thing:     foo;
  -ms-thing: foo;
}
```

#### CSS: Fix css-modules composes breaking with long line width ([#2190](https://github.com/prettier/prettier/pull/2190)) by [@tdeekens](https://github.com/tdeekens)

In order to be fast, many "packagers" do not parse files in order to extract dependencies but instead use a crude regex. This is a reason why we don't break long `require()` calls and it happens to also affect CSS Modules. If you add new lines in the `composes` field, it doesn't recognize it anymore. So we're no longer breaking it there, even if it goes over 80 columns.

<!-- prettier-ignore -->
```css
// Before
.reference {
  composes:
    selector
    from
    "a/long/file/path/exceeding/the/maximum/length/forcing/a/line-wrap/file.css";
}

// After
.reference {
  composes: selector from "a/long/file/path/exceeding/the/maximum/length/forcing/a/line-wrap/file.css";
}
```

#### CSS: First try scss when there's an [@import](https://github.com/import) with comma ([#2225](https://github.com/prettier/prettier/pull/2225))

We made a decision to have only a single high level "parser" for CSS, SCSS and Less even though we are using `postcss-less` and `postcss-scss` under the hood. We use a regex to figure out which parser to try first and fallback to the other one if a syntax error is thrown. Unfortunately, for certain features, the first (incorrect) parser doesn't throw and instead skips some elements. So, we need to beef up the regex to make sure we are right for the early detection.

Thankfully, this hack is working well in practice. If we find a lot more edge cases, we'll likely want to do the right thing(tm) and split them into two parsers.

<!-- prettier-ignore -->
```scss
// Before
@import "text-shadow";

// After
@import "rounded-corners", "text-shadow";
```

### TypeScript

TypeScript support is now solid, all the changes for this release are small edge cases.

#### TypeScript: print arrow function type params on same line as params ([#2101](https://github.com/prettier/prettier/pull/2101)) by [@azz](https://github.com/azz)

The core algorithm of prettier is to expand a group if all the elements do not fit. It works really well in practice for most of JavaScript but there's one case it doesn't handle very well is when there are two groups side by side, in this case: `<Generics>(Arguments)`. We have to carefully create groups such that arguments expand first as this is generally what people expect.

<!-- prettier-ignore -->
```ts
// Before
export const forwardS = R.curry(<
  V,
  T
>(prop: string, reducer: ReducerFunction<V, T>, value: V, state: {[name: string]: T}) =>
  R.assoc(prop, reducer(value, state[prop]), state)
);

// After
export const forwardS = R.curry(
  <V, T>(
    prop: string,
    reducer: ReducerFunction<V, T>,
    value: V,
    state: { [name: string]: T }
  ) => R.assoc(prop, reducer(value, state[prop]), state)
);
```

#### TypeScript: keep parens around with yield/await non-null assertion ([#2149](https://github.com/prettier/prettier/pull/2149)) by [@azz](https://github.com/azz)

For better or worse, we decided to manually handle adding parenthesis. So when a new operator is introduced, we need to make sure that we add correct parenthesis when nested with any other combination of operators. In this case, we missed await inside of TypeScript `!`.

<!-- prettier-ignore -->
```ts
// Before
const bar = await foo(false)!;

// After
const bar = (await foo(false))!;
```

#### TypeScript: Print {} in import if it's in the source ([#2150](https://github.com/prettier/prettier/pull/2150)) by [@azz](https://github.com/azz)

We use typescript-eslint-parser project that translates TypeScript AST into estree AST in order for prettier to print it. From time to time we're going to find edge cases that it doesn't handle yet. In this case, it didn't give a way to tell that there's an empty `{}`, which apparently is important for TypeScript. Thankfully, the team is very responsive and they fixed it after we put a workaround inside of prettier.

<!-- prettier-ignore -->
```ts
// Before
import from "@types/googlemaps";

// After
import {} from "@types/googlemaps";
```

#### TypeScript: Always break interfaces onto multiple lines ([#2161](https://github.com/prettier/prettier/pull/2161)) by [@azz](https://github.com/azz)

The code that implements `interface` is shared with the code that prints `object`s, which contains a rule to keep them expanded if there's a `\n` inside. But, this is not the intended behavior for interfaces. We always want to expand, like we do for classes, even if it fits 80 columns.

<!-- prettier-ignore -->
```ts
// Before
interface FooBar { [id: string]: number; }

// After
interface FooBar {
  [id: string]: number;
}
```

#### TypeScript: Fix extra semicolon in ambient typescript declaration emit ([#2167](https://github.com/prettier/prettier/pull/2167)) by [@azz](https://github.com/azz)

`no-semi` and `semi` are often requested but on the prettier team we're one step ahead and implemented `two-semi` for you! Just kidding, it was a bug and is now fixed ;)

<!-- prettier-ignore -->
```ts
// Before
declare module "classnames" {
  export default function classnames(
    ...inputs: (string | number | false | object | undefined)[]
  ): string;;
}

// After
declare module "classnames" {
  export default function classnames(
    ...inputs: (string | number | false | object | undefined)[]
  ): string;
}
```

#### TypeScript: group function params in call/construct signatures ([#2169](https://github.com/prettier/prettier/pull/2169)) by [@azz](https://github.com/azz)

Adding a comment before a method used to take into account the comment length and would often expand the method when it wasn't expected. Thankfully, it was a simple fix, just wrap the output in a `group`.

<!-- prettier-ignore -->
```ts
// Before
interface TimerConstructor {
  // Line-splitting comment
  new (
    interval: number,
    callback: (handler: Timer) => void
  ): Timer;
}

interface TimerConstructor {
  // Line-splitting comment
  new (interval: number, callback: (handler: Timer) => void): Timer;
}
```

#### TypeScript: Upgrade tsep ([#2183](https://github.com/prettier/prettier/pull/2183)) by [@azz](https://github.com/azz)

This bug was very annoying if you ran into it: anytime you formatted the code, it would add one more `_` to the object key!

<!-- prettier-ignore -->
```ts
// Before
obj = {
  __: 42
  ___: 42
};

// After
obj = {
  _: 42
  __: 42
};
```

#### TypeScript: break on multiple interface extends ([#2085](https://github.com/prettier/prettier/pull/2085)) by [@azz](https://github.com/azz)

Unlike in JavaScript, TypeScript lets you extend multiple classes at once. It turns out that people use this feature and prettier now does a better job at printing it.

<!-- prettier-ignore -->
```ts
// Before
export interface ThirdVeryLongAndBoringInterfaceName extends AVeryLongAndBoringInterfaceName, AnotherVeryLongAndBoringInterfaceName, AThirdVeryLongAndBoringInterfaceName {}

// After
export interface ThirdVeryLongAndBoringInterfaceName
  extends AVeryLongAndBoringInterfaceName,
    AnotherVeryLongAndBoringInterfaceName,
    AThirdVeryLongAndBoringInterfaceName {}
```

#### TypeScript: handle ObjectPattern instead of ObjectExpression inside BinaryExpression ([#2238](https://github.com/prettier/prettier/pull/2238)) by [@azz](https://github.com/azz)

This one isn't very interesting, it's an edge case that's not properly handled in the TypeScript -> estree conversion.

<!-- prettier-ignore -->
```ts
// Before
call(c => { bla: 1 }) || [];

// After
call(c => ({ bla: 1 })) || [];
```

#### Preserve lines after directives ([#2070](https://github.com/prettier/prettier/pull/2070))

By supporting TypeScript, prettier is now being used in a lot of Angular codebases which exercises edge cases that were not properly handled. In this case, we didn't preserve empty lines after directives inside of a function.

<!-- prettier-ignore -->
```ts
// Before
export default class {
  constructor($log, $uibModal) {
    "ngInject";
    Object.assign(this, { $log, $uibModal });

// After
export default class {
  constructor($log, $uibModal) {
    "ngInject";

    Object.assign(this, { $log, $uibModal });
```

### JavaScript

This release is very light in terms of JavaScript changes, which is awesome. We're starting to see the light at the end of the tunnel and get towards a great pretty printer. We're never going to get to a 100% perfect automatic pretty printer. The goal is that for every issue we get, there are no clear ways to improve the way it is printed without regressing other pieces.

#### Allow JSX lines to be recombined ([#1831](https://github.com/prettier/prettier/pull/1831)) by [@karl](https://github.com/karl)

The goal of prettier is to have a consistent way to format your code: given an AST, we always print the same way. In two places we had to compromise and read the original format: JSX and Objects. With this change, we're no longer relying on the original input for JSX with text inside. This lets us reflow text inside of JSX.

<!-- prettier-ignore -->
```jsx
// Before
const Abc = () => {
  return (
    <div>
      Please state your
      {" "}
      <b>name</b>
      {" "}
      and
      {" "}
      <b>occupation</b>
      {" "}
      for the board of directors.
    </div>
  );
};

// After
const Abc = () => {
  return (
    <div>
      Please state your <b>name</b> and <b>occupation</b> for the board of
      directors.
    </div>
  );
}
```

#### Break on non-literal computed member expression ([#2087](https://github.com/prettier/prettier/pull/2087)) by [@azz](https://github.com/azz)

Printing member chains is the most complicated piece of prettier and we keep finding small tweaks we can do to make it a better experience.

<!-- prettier-ignore -->
```js
// Before
nock(/test/)
  .matchHeader("Accept", "application/json")[httpMethodNock(method)]("/foo")
  .reply(200, {
    foo: "bar",
  });

// After
nock(/test/)
  .matchHeader("Accept", "application/json")
  [httpMethodNock(method)]("/foo")
  .reply(200, {
    foo: "bar",
  });
```

#### Indent first variable in one-var scenario ([#2095](https://github.com/prettier/prettier/pull/2095)) by [@azz](https://github.com/azz)

Up until recently we haven't done much to support printing multiple variables in a single declaration as the most common practice is to do one variable declaration per variable. For single declarations, we don't want to indent it, but it turns out that we do when there are other ones afterwards, otherwise it looks weird.

<!-- prettier-ignore -->
```js
// Before
var templateTagsMapping = {
 '%{itemIndex}': 'index',
 '%{itemContentMetaTextViews}': 'views'
},
  separator = '<span class="item__content__meta__separator">•</span>';

// After
var templateTagsMapping = {
   '%{itemIndex}': 'index',
   '%{itemContentMetaTextViews}': 'views'
  },
  separator = '<span class="item__content__meta__separator">•</span>';
```

#### Allow break with both default named import ([#2096](https://github.com/prettier/prettier/pull/2096)) by [@azz](https://github.com/azz)

This one is an unfortunate regression from 1.4 where we inlined import that only contained a single element. Turns out the definition of a single element allowed a single type and a single element. This is now corrected!

<!-- prettier-ignore -->
```js
// Before
import transformRouterContext, { type TransformedContextRouter } from '../../helpers/transformRouterContext';

// After
import transformRouterContext, {
  type TransformedContextRouter
} from '../../helpers/transformRouterContext';
```

#### Turn allowImportExportEverywhere on ([#2207](https://github.com/prettier/prettier/pull/2207)) by [@zimme](https://github.com/zimme)

The goal of prettier is to format code people write in practice, so we enable loose/experimental modes for all the parsers we support. Babylon allows you to write import within a function, which is not part of the standard, but it doesn't cost us much to allow it.

<!-- prettier-ignore -->
```js
// Before
ParseError

// After
function f() {
  import x from 'x';
}
```

#### Support inline template for new calls ([#2222](https://github.com/prettier/prettier/pull/2222))

We keep adding features for function calls and have to backport them to new calls as they have a different AST node type but in practice we want to treat them the same. This fix refactored the two so that they are going through the same call site, so hopefully should prevent more from sneaking in.

<!-- prettier-ignore -->
```js
// Before
new Error(
  formatErrorMessage`
    This is a really bad error.
    Which has more than one line.
  `
);

// After
new Error(formatErrorMessage`
  This is a really bad error.
  Which has more than one line.
`);
```

#### Don't indent + in object value ([#2227](https://github.com/prettier/prettier/pull/2227))

When we switched to using the same heuristic for assignment (`a = b`) for objects (`{a: b}`), we forgot to fix the indentation. Now it's fixed.

<!-- prettier-ignore -->
```js
// Before
var abc = {
  thing:
    "asdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdf" +
      "asdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdf" +
      "asdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdf",
}

// After
var abc = {
  thing:
    "asdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdf" +
    "asdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdf" +
    "asdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdfasdf",
}
```

#### Handle conditions inside of a ternary ([#2228](https://github.com/prettier/prettier/pull/2228))

Prettier already had a special case when the expression was a conditional but it didn't apply when the conditional was the left part of a ternary. Now it does.

<!-- prettier-ignore -->
```js
// Before
room = room.map((row, rowIndex) =>
  row.map(
    (col, colIndex) =>
      rowIndex === 0 ||
        colIndex === 0 ||
        rowIndex === height ||
        colIndex === width
        ? 1
        : 0
  )
);

// After
room = room.map((row, rowIndex) =>
  row.map(
    (col, colIndex) =>
      rowIndex === 0 ||
      colIndex === 0 ||
      rowIndex === height ||
      colIndex === width
        ? 1
        : 0
  )
);
```

#### Add caching for printing ([#2259](https://github.com/prettier/prettier/pull/2259))

With the 1.0 release, we fixed a bug in the printing that introduced an exponential behavior. We've been able to mitigate the biggest issue such that reasonable code didn't time out, but it wasn't completely fixed it. By adding a caching layer at the right spot, we should now be in the clear.

This should make printing the IR of prettier using prettier in debug mode no longer time out.

<!-- prettier-ignore -->
```js
// Before
...times out...

// After
someObject.someFunction().then(function () {
    return someObject.someFunction().then(function () {
        return someObject.someFunction().then(function () {
            return someObject.someFunction().then(function () {
                return someObject.someFunction().then(function () {
                    return someObject.someFunction().then(function () {
                        return someObject.someFunction().then(function () {
                            return someObject.someFunction().then(function () {
                                return someObject.someFunction().then(function () {
                                    anotherFunction();
                                });
                            });
                        });
                    });
                });
            });
        });
    });
});
```

#### Fix variance location ([#2261](https://github.com/prettier/prettier/pull/2261))

We refactored the code that prints modifiers when we introduced TypeScript support and accidentally moved around the variance (`+`) part before `static` which is not valid in Flow. This is now fixed.

<!-- prettier-ignore -->
```ts
// Before
class Route {
  +static param: T;
}

// After
class Route {
  static +param: T;
}
```

### Miscellaneous

#### Various fixes for range and cursor tracking ([#2266](https://github.com/prettier/prettier/pull/2266), [#2248](https://github.com/prettier/prettier/pull/2248), [#2250](https://github.com/prettier/prettier/pull/2250), [#2136](https://github.com/prettier/prettier/pull/2136)) by [@CiGit](https://github.com/CiGit) and [@josephfrazier](https://github.com/josephfrazier)

Both those features were introduced in the last release and we discovered a bunch of issues when actually using them in production. A bunch of them got fixed, if you see more, please report them!
