package cjmx.cli

import scala.collection.immutable.Seq

import javax.management.{ ObjectName, Attribute, MBeanInfo }

import cjmx.util.jmx.JMX._

object TextMessageFormatter extends MessageFormatter {

  override def formatNames(names: Seq[ObjectName]) = {
    names.map { _.toString }
  }

  override def formatAttributes(attrsByName: Seq[(ObjectName, Seq[Attribute])]) = {
    val out = new OutputBuilder
    attrsByName foreach { case (name, attrs) =>
      out <+ name.toString
      out <+ ("-" * name.toString.size)
      out indented {
        attrs foreach { attr => out <+ attr.toString }
      }
      out <+ ""
    }
    out.lines
  }

  override def formatInfo(info: Seq[(ObjectName, MBeanInfo)], detailed: Boolean) = {
    val out = new OutputBuilder
    for ((name, inf) <- info) {
      val nameLine = "Object name: %s".format(name)
      out <+ nameLine
      out <+ ("-" * nameLine.size)
      out <+ "Description: %s".format(inf.getDescription)
      out <+ ""

      val attributes = inf.getAttributes
      if (attributes.nonEmpty) {
        out <+ "Attributes:"
        out indented {
          attributes.foreach { attr =>
            out <+ "%s: %s".format(attr.getName, JType(attr.getType).toString)
            if (detailed) out.indented {
              out <+ "Description: %s".format(attr.getDescription)
            }
          }
        }
        out <+ ""
      }

      val operations = inf.getOperations
      if (operations.nonEmpty) {
        out <+ "Operations:"
        out indented {
          operations foreach { op =>
            out <+ "%s(%s): %s".format(
              op.getName,
              op.getSignature.map { pi => "%s: %s".format(pi.getName, JType(pi.getType).toString) }.mkString(", "),
              JType(op.getReturnType).toString)
            if (detailed) out.indented {
              out <+ "Description: %s".format(op.getDescription)
            }
          }
        }
        out <+ ""
      }

      val notifications = inf.getNotifications
      if (notifications.nonEmpty) {
        out <+ "Notifications:"
        out indented {
          notifications foreach { nt =>
            out <+ nt.getName
            if (detailed) out.indented {
              out <+ "Description: %S".format(nt.getDescription)
              out <+ "Notification types:"
              out indented {
                nt.getNotifTypes foreach { out <+ _ }
              }
            }
          }
        }
        out <+ ""
      }
    }
    out.lines
  }

  override def formatInvocationResults(namesAndResults: Seq[(ObjectName, InvocationResult)]) = {
    namesAndResults map { case (name, result) => "%s: %s".format(name, result) }
  }
}

