---
title: Apache Mesos - C++ Style Guide
layout: documentation
---

# Mesos C++ Style Guide

The Mesos codebase follows the [Google C++ Style Guide](https://google.github.io/styleguide/cppguide.html) with some notable differences, as described below. Note that the [clang-format](clang-format.md) tool can be helpful to ensure that some of the mechanical style rules are obeyed.

## Scoping

### Namespaces
* We avoid `using namespace foo` statements as it is not explicit about which symbols are pulled in, and it can often pull in a lot of symbols, which sometimes lead to conflicts.
* It is OK to use namespace aliases to help pull in sub-namespaces, such as `namespace http = process::http;`. These should only be present at the top of the .cpp file.

## Naming

### Variable Names
* We use [lowerCamelCase](http://en.wikipedia.org/wiki/CamelCase#Variations_and_synonyms) for variable names (Google uses snake_case, and their class member variables have trailing underscores).
* We prepend constructor and function arguments with a leading underscore to avoid ambiguity and / or shadowing:

~~~{.cpp}
Try(State _state, T* _t = nullptr, const std::string& _message = "")
  : state(_state), t(_t), message(_message) {}
~~~

* Prefer trailing underscores for use as member fields (but not required). Some trailing underscores are used to distinguish between similar variables in the same scope (think prime symbols), *but this should be avoided as much as possible, including removing existing instances in the code base.*

* If you find yourself creating a copy of an argument passed by const reference, consider passing it by value instead (if you don't want to use a leading underscore and copy in the body of the function):

~~~{.cpp}
// You can pass-by-value in ProtobufProcess::install() handlers.
void Slave::statusUpdate(StatusUpdate update, const UPID& pid)
{
  ...
  update.mutable_status()->set_source(
      pid == UPID() ? TaskStatus::SOURCE_SLAVE : TaskStatus::SOURCE_EXECUTOR);
  ...
}
~~~

### Constant Names
* We use [SCREAMING_SNAKE_CASE](http://en.wikipedia.org/wiki/Letter_case#Special_case_styles) for constant names (Google uses a `k` followed by mixed case, e.g. `kDaysInAWeek`).

### Function Names
* We use [lowerCamelCase](http://en.wikipedia.org/wiki/CamelCase#Variations_and_synonyms) for function names (Google uses mixed case for regular functions; and their accessors and mutators match the name of the variable).

## Strings
* Strings used in log and error messages should end without a period.

## Comments
* End each sentence within a comment with a punctuation mark (please note that we generally prefer periods); this applies to incomplete sentences as well.
* For trailing comments, leave one space.
* Use backticks when quoting code excerpts or object/variable/function names. For example:

~~~{.cpp}
// Use `SchedulerDriver::acceptOffers()` to send several offer
// operations. This makes use of the `RESERVE()` and `UNRESERVE()`
// helpers, which take a `Resources` object as input and produce
// appropriate offer operations. Note that we are unreserving the
// resources contained in `dynamicallyReserved1`.
driver.acceptOffers({offer.id()},
    {UNRESERVE(dynamicallyReserved1),
     RESERVE(dynamicallyReserved2),
     RESERVE(dynamicallyReserved3)},
    filters);
~~~

## Breaks
* Break before braces on enum, function, and record (i.e. struct, class, union) definitions.

## Indentation

### Class Format
* Access modifiers are not indented (Google uses one space indentation).
* Constructor initializers are indented by two spaces (Google indents by four).

### Templates
* Leave one space after the `template` keyword, e.g. `template <typename T>` rather than `template<typename T>`.

### Function Definition/Invocation
* Newline when calling or defining a function: indent with four spaces.
* We do not follow Google's style of wrapping on the open parenthesis, the general goal is to reduce visual "jaggedness" in the code. Prefer (1), (4), (5), sometimes (3), never (2):

~~~{.cpp}
// 1: OK.
allocator->resourcesRecovered(frameworkId, agentId, resources, filters);

// 2: Don't use.
allocator->resourcesRecovered(frameworkId, agentId,
                              resources, filters);

// 3: Don't use in this case due to "jaggedness".
allocator->resourcesRecovered(frameworkId,
                              agentId,
                              resources,
                              filters);

// 3: In this case, 3 is OK.
foobar(someArgument,
       someOtherArgument,
       theLastArgument);

// 4: OK.
allocator->resourcesRecovered(
    frameworkId,
    agentId,
    resources,
    filters);

// 5: OK.
allocator->resourcesRecovered(
    frameworkId, agentId, resources, filters);
~~~

### Continuation
* Newline for an assignment statement: indent with two spaces.

~~~{.cpp}
Try<Duration> failoverTimeout =
  Duration::create(FrameworkInfo().failover_timeout());
~~~

## Empty Lines
* One empty line at the end of the file.
* Inside a code block, every multi-line statement should be followed by one empty line.

~~~{.cpp}
Try<very_very_long_type> long_name =
  ::protobuf::parse<very_very_long_type>(
      request);

for (int i = 0; i < very_very_long_expression();
     i++) {
  // No empty line here for control constructs.
}
~~~

* Elements outside classes (classes, structs, global functions, etc.) should be spaced apart by two empty lines.
* Elements inside classes (member variables and functions) should not be spaced apart by more than one empty line.

## Capture by Reference

We disallow capturing **temporaries** by reference. See [MESOS-2629](https://issues.apache.org/jira/browse/MESOS-2629) for the rationale.

~~~{.cpp}
Future<Nothing> f() { return Nothing(); }
Future<bool> g() { return false; }

struct T
{
  T(const char* data) : data(data) {}
  const T& member() const { return *this; }
  const char* data;
};

// 1: Don't use.
const Future<Nothing>& future = f();

// 1: Instead use.
const Future<Nothing> future = f();

// 2: Don't use.
const Future<Nothing>& future = Future<Nothing>(Nothing());

// 2: Instead use.
const Future<Nothing> future = Future<Nothing>(Nothing());

// 3: Don't use.
const Future<bool>& future = f().then(lambda::bind(g));

// 3: Instead use.
const Future<bool> future = f().then(lambda::bind(g));

// 4: Don't use (since the T that got constructed is a temporary!).
const T& t = T("Hello").member();

// 4: Preferred alias pattern (see below).
const T t("Hello");
const T& t_ = t.member();

// 4: Can also use.
const T t = T("Hello").member();
~~~

We allow capturing non-temporaries by *constant reference* when the intent is to **alias**.

The goal is to make code more concise and improve readability. Use this if an expression referencing a field by `const` is:

* Used repeatedly.
* Would benefit from a concise name to provide context for readability.
* Will **not** be invalidated during the lifetime of the alias. Otherwise document this explicitly.

~~~{.cpp}
hashmap<string, hashset<int>> index;

// 1: Ok.
const hashset<int>& values = index[2];

// 2: Ok.
for (auto iterator = index.begin(); iterator != index.end(); ++iterator) {
  const hashset<int>& values = iterator->second;
}

// 3: Ok.
foreachpair (const string& key, const hashset<int>& values, index) {}
foreachvalue (const hashset<int>& values, index) {}
foreachkey (const string& key, index) {}

// 4: Avoid aliases in most circumstances as they can be dangerous.
//    This is an example of a dangling alias!
vector<string> strings{"hello"};

string& s = strings[0];

strings.erase(strings.begin());

s += "world"; // THIS IS A DANGLING REFERENCE!
~~~

## File Headers

* Mesos source files must contain the "ASF" header:

         // Licensed to the Apache Software Foundation (ASF) under one
         // or more contributor license agreements.  See the NOTICE file
         // distributed with this work for additional information
         // regarding copyright ownership.  The ASF licenses this file
         // to you under the Apache License, Version 2.0 (the
         // "License"); you may not use this file except in compliance
         // with the License.  You may obtain a copy of the License at
         //
         //     http://www.apache.org/licenses/LICENSE-2.0
         //
         // Unless required by applicable law or agreed to in writing, software
         // distributed under the License is distributed on an "AS IS" BASIS,
         // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
         // See the License for the specific language governing permissions and
         // limitations under the License.


* Stout and libprocess source files must contain the "Apache License Version 2.0" header:

         // Licensed under the Apache License, Version 2.0 (the "License");
         // you may not use this file except in compliance with the License.
         // You may obtain a copy of the License at
         //
         //     http://www.apache.org/licenses/LICENSE-2.0
         //
         // Unless required by applicable law or agreed to in writing, software
         // distributed under the License is distributed on an "AS IS" BASIS,
         // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
         // See the License for the specific language governing permissions and
         // limitations under the License

## Order of includes

In addition to the ordering rules from the Google style guide, Mesos related headers are separated into sections. Newline to separate each section.
Mesos related headers in `include` directories are partitioned by their subfolders, sorted alphabetically, and included using brackets.
Header in `src` directories are included afterwards, using the same rules but with quotes instead of brackets.

Example for `src/common/foo.cpp`:

~~~{.cpp}
#include "common/foo.hpp"

#include <stdint.h>

#include <string>
#include <vector>

#include <boost/circular_buffer.hpp>

#include <mesos/mesos.hpp>
#include <mesos/type_utils.hpp>

#include <mesos/module/authenticator.hpp>

#include <mesos/scheduler/scheduler.hpp>

#include <process/http.hpp>
#include <process/protobuf.hpp>

#include <stout/foreach.hpp>
#include <stout/hashmap.hpp>

#include "common/build.hpp"
#include "common/protobuf_utils.hpp"

#include "master/flags.hpp"
~~~

## C++11

We support C++11 and require GCC 4.8+ or Clang 3.5+ compilers. The whitelist of supported C++11 features is:

* `nullptr`
* Static assertions.
* Multiple right angle brackets.
* Type inference (`auto` and `decltype`). The main goal is to increase code readability. This is safely the case if the exact same type omitted on the left is already fully stated on the right. Here are several examples:

~~~{.cpp}
// 1: OK.
const auto i = values.find(keys.front());
// Compare with
const typename map::iterator i = values.find(keys.front());

// 2: OK.
auto names = shared_ptr<list<string>>(new list<string>());
// Compare with
shared_ptr<list<string>> names = shared_ptr<list<string>>(new list<string>());

// 3: Don't use.
auto authorizer = LocalAuthorizer::create(acls);
// Compare with
Try<Owned<LocalAuthorizer>> authorizer = LocalAuthorizer::create();
~~~

* Rvalue references.
* Explicitly-defaulted functions.
* Variadic templates.
* Delegating constructors.
* Mutexes.
  * `std::mutex`
  * `std::lock_guard<std::mutex>`
  * `std::unique_lock<std::mutex>`
* Atomics (`std::atomic`)
  * The standard defines a number of predefined typedefs for atomic types (e.g., `std::atomic_int`), in addition to `std::atomic<T>`. When a typedef is available, it should be preferred over explicit template specialization of `std::atomic<T>`.
  * When reading from and writing to atomic values, the `load` and `store` member functions should be used instead of the overloads of `operator T()` and `operator=`. Being explicit helps to draw the reader's attention to the fact that atomic values are being manipulated.
* Shared from this.
  * `class T : public std::enable_shared_from_this<T>`
  * `shared_from_this()`
* Lambdas!
  * Don't put a space between the capture list and the parameter list:

~~~{.cpp}
// 1: OK.
[]() { ...; };

// 2: Don't use.
[] () { ...; };
~~~

  * Prefer default capture by value, explicit capture by value, then capture by reference. To avoid dangling-pointer bugs, *never* use default capture by reference:

~~~{.cpp}
// 1: OK.
[=]() { ... }; // Default capture by value.
[n]() { ... }; // Explicit capture by value.
[&n]() { ... }; // Explicit capture by reference.
[=, &n]() { ... }; // Default capture by value, explicit capture by reference.

// 2: Don't use.
[&]() { ... }; // Default capture by reference.
~~~

  * Use `mutable` only when absolutely necessary.

~~~{.cpp}
// 1: OK.
[]() mutable { ...; };
~~~

  * Feel free to ignore the return type by default, adding it as necessary to appease the compiler or be more explicit for the reader.

~~~{.cpp}
// 1: OK.
[]() { return true; };
[]() -> bool { return ambiguous(); };
~~~

  * Feel free to use `auto` when naming a lambda expression:

~~~{.cpp}
// 1: OK.
auto lambda = []() { ...; };
~~~

  * Format lambdas similar to how we format functions and methods. Feel free to let lambdas be one-liners:

~~~{.cpp}
// 1: OK.
auto lambda = []() {
  ...;
};

// 2: OK.
auto lambda = []() { ...; };
~~~

  * Feel free to inline lambdas within function arguments:

~~~{.cpp}
instance.method([]() {
  ...;
});
~~~

  * Chain function calls on a newline after the closing brace of the lambda and the closing parenthesis of function call:

~~~{.cpp}
// 1: OK.
instance
  .method([]() {
    ...;
  })
  .then([]() { ...; })
  .then([]() {
    ...;
  });

// 2: OK (when no chaining, compare to 1).
instance.method([]() {
  ...;
});

// 3: OK (if no 'instance.method').
function([]() {
  ...;
})
.then([]() { ...; })
.then([]() {
  ...;
});

// 3: OK (but prefer 1).
instance.method([]() {
  ...;
})
.then([]() { ...; })
.then([]() {
  ...;
});
~~~

  * Wrap capture lists independently of parameters, *use the same formatting as if the capture list were template parameters*:

~~~{.cpp}
// 1: OK.
function([&capture1, &capture2, &capture3](
    const T1& p1, const T2& p2, const T3& p3) {
  ...;
});

function(
    [&capture1, &capture2, &capture3](
        const T1& p1, const T2& p2, const T3& p3) {
  ...;
});

auto lambda = [&capture1, &capture2, &capture3](
    const T1& p1, const T2& p2, const T3& p3) {
  ...;
};

auto lambda =
  [&capture1, &capture2, &capture3](
      const T1& p1, const T2& p2, const T3& p3) {
  ...;
};

// 2: OK (when capture list is longer than 80 characters).
function([
    &capture1,
    &capture2,
    &capture3,
    &capture4](
        const T1& p1, const T2& p2) {
  ...;
});

auto lambda = [
    &capture1,
    &capture2,
    &capture3,
    &capture4](
        const T1& p1, const T2& p2) {
  ...;
};

// 3: OK (but prefer 2).
function([
    &capture1,
    &capture2,
    &capture3,
    &capture4](const T1& p1, const T2& t2) {
  ...;
});

auto lambda = [
    &capture1,
    &capture2,
    &capture3,
    &capture4](const T1& p1, const T2& p2) {
  ...;
};

// 3: Don't use.
function([&capture1,
          &capture2,
          &capture3,
          &capture4](const T1& p1, const T2& p2) {
  ...;
});

auto lambda = [&capture1,
               &capture2,
               &capture3,
               &capture4](const T1& p1, const T2& p2) {
  ...;
  };

// 4: Don't use.
function([&capture1,
           &capture2,
           &capture3,
           &capture4](
    const T1& p1, const T2& p2, const T3& p3) {
  ...;
});

auto lambda = [&capture1,
               &capture2,
               &capture3,
               &capture4](
    const T1& p1, const T2& p2, const T3& p3) {
  ...;
};

// 5: Don't use.
function([&capture1,
          &capture2,
          &capture3,
          &capture4](
    const T1& p1,
    const T2& p2,
    const T3& p3) {
  ...;
  });

auto lambda = [&capture1,
               &capture2,
               &capture3,
               &capture4](
    const T1& p1,
    const T2& p2,
    const T3& p3) {
  ...;
};

// 6: OK (parameter list longer than 80 characters).
function([&capture1, &capture2, &capture3](
    const T1& p1,
    const T2& p2,
    const T3& p3,
    const T4& p4) {
  ...;
});

auto lambda = [&capture1, &capture2, &capture3](
    const T1& p1,
    const T2& p2,
    const T3& p3,
    const T4& p4) {
  ...;
};

// 7: OK (capture and parameter lists longer than 80 characters).
function([
    &capture1,
    &capture2,
    &capture3,
    &capture4](
        const T1& p1,
        const T2& p2,
        const T3& p3,
        const T4& p4) {
  ...;
});

auto lambda = [
    &capture1,
    &capture2,
    &capture3,
    &capture4](
        const T1& p1,
        const T2& p2,
        const T3& p3,
        const T4& p4) {
  ...;
};
~~~

* Unrestricted Union.

  Like the pre-existing `union`, we can overlap storage allocation for objects that never exist simultaneously. However, with C++11 we are no longer *restricted to having only non-POD types in unions*. Adding non-POD types to unions complicates things, however, because we need to make sure to properly call constructors and destructors. Therefore, only use unrestricted unions (i.e., unions with non-POD types) when the union has only a single field. What does this buy us? Now we can avoid dynamic memory allocations for "container" like types, e.g., `Option`, `Try`, `Result`, etc. In effect, we treat the union like a dynamic allocation, calling *placement new*, `new (&t) T(...)` anyplace we would have just called `new T(...)` and the destructor `t.~T()` anyplace we would have called `delete t`.

* Constant expressions.

  Constant expressions allow the declaration of static non-POD objects while eliminating the unpredictable runtime initialization and destruction issues normally encountered, helping eliminate macros and hard-coded literals without sacrificing performance and type safety.  Changes which require converting from `constexpr` to `const` can propagate through the dependency tree requiring that dependent `constexpr` uses also be converted to `const`, hence we avoid using `constexpr` in complex functions.

  `constexpr` behaves as a combination of `inline` and `const` and hence must be defined before use in another `constexpr`.

  Prefer `constexpr` to `const` for all constant POD declarations, `constexpr` `char` arrays are preferred to `const` `string` literals.

~~~{.cpp}
  // OK
  constexpr char LITERAL[] = "value";

  // Not OK - not available at compile time for optimization and
  // definition required in a separate compilation module.
  const char LITERAL[];

  // Not OK - uncertain initialization order, cannot be used in other
  // constexpr statements.
  const string LITERAL("value");
~~~

  `constexpr` functions are evaluated at compile time if all their arguments are constant expressions. Otherwise they default to initialization at runtime. However `constexpr` functions are limited in that they cannot perform dynamic casts, memory allocation or calls to non-constexpr functions.  Prefer `constexpr` over const inline functions.

~~~{.cpp}
  constexpr size_t MIN = 200;
  constexpr size_t MAX = 1000;
  constexpr size_t SPAN() { return MAX-MIN; }
  int array[SPAN()];
~~~

Const expression constructors allow object initialization at compile time provided that all the constructor arguments are `constexpr` and the constructor body is empty, i.e. all initialization is performed in the initialization list.  Classes which provide `constexpr` constructors should normally also provide `constexpr` copy constructors to allow the class to be used in the return value from a `constexpr` function.

~~~{.cpp}
  class C
  {
  public:
    constexpr C(int _i) : i(_i) {};
    constexpr C(const C& c) : i(c.i) {}
  private:
    const int i;
  };
~~~

  C++11 does not provide `constexpr string` or `constexpr` containers in the STL and hence `constexpr` cannot be used for any class using stout's Error() class.
