# SQLC Database Layer Migration 🗄️

Station uses [sqlc](https://sqlc.dev/) for type-safe database operations. This document outlines our migration from manual SQL queries to generated, type-safe sqlc queries.

## 🎯 Why SQLC?

### **Benefits**
- ✅ **Type Safety**: Compile-time error detection for SQL queries
- ✅ **Performance**: No reflection, no runtime query parsing
- ✅ **Developer Experience**: Auto-completion and type checking
- ✅ **Maintainability**: Schema changes break compilation early
- ✅ **Code Generation**: No boilerplate repository code

### **Before SQLC (Manual Queries)**
```go
func (r *MCPToolRepo) GetByEnvironmentID(environmentID int64) ([]*models.MCPTool, error) {
    query := `SELECT t.id, t.mcp_server_id, t.name, t.description, t.input_schema, t.created_at 
              FROM mcp_tools t
              JOIN mcp_servers s ON t.mcp_server_id = s.id
              WHERE s.environment_id = ?`
    // Manual scanning, error-prone...
}
```

### **After SQLC (Generated Queries)**
```go
// Generated by sqlc
func (q *Queries) ListMCPToolsByEnvironment(ctx context.Context, environmentID int64) ([]MucTool, error) {
    rows, err := q.db.QueryContext(ctx, listMCPToolsByEnvironment, environmentID)
    // Type-safe scanning automatically generated
}
```

## 📋 Migration Strategy

### **Phase 1: Schema Synchronization** ✅
1. ✅ Update `internal/db/schema.sql` to match actual database
2. ✅ Fix column name mismatches (e.g., `server_id` → `mcp_server_id`)
3. ✅ Ensure all tables have correct foreign key relationships

### **Phase 2: Query Migration** 🔄
1. 🔄 Update all `.sql` files in `internal/db/queries/`
2. 🔄 Regenerate Go code with `sqlc generate`
3. 🔄 Update repository layer to use generated queries
4. 🔄 Remove manual SQL query methods

### **Phase 3: Testing & Validation** 📝
1. 📝 Ensure all existing functionality works
2. 📝 Add integration tests for critical queries
3. 📝 Performance benchmarking vs manual queries

## 🛠️ SQLC Configuration

### **sqlc.yaml**
```yaml
version: "2"
sql:
  - engine: "sqlite"
    queries: "internal/db/queries"
    schema: "internal/db/schema.sql"
    gen:
      go:
        package: "queries"
        out: "internal/db/queries"
        emit_json_tags: true
        emit_prepared_queries: false
        emit_interface: false
        emit_exact_table_names: false
```

### **Directory Structure**
```
internal/db/
├── schema.sql              # Master schema file
├── queries/               # SQL query definitions
│   ├── agents.sql
│   ├── environments.sql
│   ├── mcp_configs.sql
│   ├── mcp_servers.sql
│   ├── mcp_tools.sql
│   └── users.sql
├── queries/               # Generated Go code
│   ├── agents.sql.go      # Generated from agents.sql
│   ├── db.go              # Generated database interface
│   ├── models.go          # Generated model structs
│   └── ...
└── repositories/          # Repository layer (uses generated queries)
    ├── agents.go
    ├── environments.go
    └── ...
```

## 📝 Writing SQLC Queries

### **Query Definition Example**
```sql
-- name: CreateMCPTool :one
INSERT INTO mcp_tools (mcp_server_id, name, description, input_schema)
VALUES (?, ?, ?, ?)
RETURNING *;

-- name: ListMCPToolsByEnvironment :many
SELECT t.* FROM mcp_tools t
JOIN mcp_servers s ON t.mcp_server_id = s.id
WHERE s.environment_id = ?
ORDER BY s.name, t.name;

-- name: DeleteMCPTool :exec
DELETE FROM mcp_tools WHERE id = ?;
```

### **Generated Go Usage**
```go
type Repository struct {
    queries *queries.Queries
}

func (r *Repository) CreateTool(ctx context.Context, tool CreateMCPToolParams) (McpTool, error) {
    return r.queries.CreateMCPTool(ctx, tool)
}

func (r *Repository) ListToolsByEnvironment(ctx context.Context, envID int64) ([]McpTool, error) {
    return r.queries.ListMCPToolsByEnvironment(ctx, envID)
}
```

## 🔄 Migration Checklist

### **Schema Files**
- [x] `internal/db/schema.sql` - Updated to match actual database
- [ ] All table definitions validated against current database
- [ ] Foreign key relationships verified

### **Query Files**
- [ ] `agents.sql` - Agent CRUD operations
- [ ] `environments.sql` - Environment management
- [ ] `mcp_configs.sql` - MCP configuration handling
- [ ] `mcp_servers.sql` - MCP server management
- [x] `mcp_tools.sql` - MCP tool operations (partially updated)
- [ ] `users.sql` - User management
- [ ] `agent_tools.sql` - Agent-tool relationships
- [ ] `agent_runs.sql` - Agent execution tracking

### **Repository Updates**
- [ ] `agents.go` - Use generated agent queries
- [ ] `environments.go` - Use generated environment queries
- [ ] `mcp_configs.go` - Use generated config queries
- [ ] `mcp_servers.go` - Use generated server queries
- [x] `mcp_tools.go` - Use generated tool queries (in progress)
- [ ] `users.go` - Use generated user queries

### **Testing**
- [ ] Unit tests for all repository methods
- [ ] Integration tests with actual database
- [ ] Performance benchmarks
- [ ] Error handling validation

## 🚀 Development Workflow

### **Making Schema Changes**
1. Update `internal/db/schema.sql`
2. Create migration file in `internal/db/migrations/`
3. Run `sqlc generate` to update generated code
4. Update affected repository methods
5. Run tests to ensure compatibility

### **Adding New Queries**
1. Add query to appropriate `.sql` file in `internal/db/queries/`
2. Use sqlc annotations (`:one`, `:many`, `:exec`)
3. Run `sqlc generate` to generate Go code
4. Use generated method in repository
5. Add tests for new functionality

### **Common Commands**
```bash
# Generate sqlc code
sqlc generate

# Verify schema and queries
sqlc compile

# Check for issues
sqlc vet

# Development build with updated queries
make dev
```

## 🎯 Best Practices

### **Query Naming**
- Use descriptive names: `ListMCPToolsByEnvironment` not `GetTools`
- Follow pattern: `<Action><Entity><Condition>`
- Examples: `CreateAgent`, `UpdateUserByID`, `DeleteExpiredSessions`

### **Return Types**
- `:one` - Single row (returns struct or error)
- `:many` - Multiple rows (returns slice)
- `:exec` - No return data (returns error only)

### **Parameter Naming**
- Use clear parameter names matching column names
- Example: `WHERE environment_id = ?` → parameter name should be `environmentID`

### **Complex Queries**
- Use JOINs efficiently
- Consider performance implications
- Add indexes for frequently queried columns

## 🐛 Common Issues & Solutions

### **Schema Mismatch**
```bash
# Error: column "xyz" does not exist
# Solution: Update schema.sql to match actual database
sqlite3 station.db ".schema table_name"
```

### **Type Conflicts**
```bash
# Error: cannot convert string to int64
# Solution: Check column types in schema.sql match Go types
```

### **Missing Foreign Keys**
```bash
# Error: FOREIGN KEY constraint failed
# Solution: Ensure foreign key relationships are properly defined
```

## 📊 Performance Considerations

### **Query Optimization**
- Use appropriate indexes
- Limit result sets with LIMIT clauses
- Avoid N+1 query problems with JOINs

### **Connection Management**
- Use connection pooling
- Implement proper transaction handling
- Consider read replicas for heavy read workloads

## 🔮 Future Improvements

### **Planned Enhancements**
- [ ] **Prepared Statements**: Enable for better performance
- [ ] **Query Analysis**: Add sqlc vet rules for Station-specific patterns
- [ ] **Schema Validation**: Automated schema drift detection
- [ ] **Performance Monitoring**: Query performance tracking

### **Advanced Features**
- [ ] **Multi-tenant Queries**: Row-level security patterns
- [ ] **Audit Logging**: Automatic change tracking
- [ ] **Soft Deletes**: Implement soft delete patterns
- [ ] **Pagination**: Standardized pagination helpers

---

## 📞 Getting Help

- **SQLC Documentation**: https://docs.sqlc.dev/
- **Station Issues**: Report migration issues in GitHub Issues
- **Development**: Check `/docs/contributing/` for development guides

---

**Migration Status**: 🔄 **In Progress**

*Last Updated: 2025-07-31*

**Next Steps**: Complete query file updates and repository migration. See todo list for current progress.