//
//  KSCrashMonitor_CPPException.c
//
//  Copyright (c) 2012 Karl Stenerud. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall remain in place
// in this source code.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//

#include "KSCrashMonitor_CPPException.h"
#include "KSCrashMonitorContext.h"
#include "KSID.h"
#include "KSThread.h"
#include "KSMachineContext.h"
#include "KSStackCursor_SelfThread.h"
#include "KSCrash_fishhook.h"

//#define KSLogger_LocalLevel TRACE
#include "KSLogger.h"

#include <cxxabi.h>
#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <typeinfo>


#define STACKTRACE_BUFFER_LENGTH 30
#define DESCRIPTION_BUFFER_LENGTH 1000


// Compiler hints for "if" statements
#define likely_if(x) if(__builtin_expect(x,1))
#define unlikely_if(x) if(__builtin_expect(x,0))


// ============================================================================
#pragma mark - Globals -
// ============================================================================

/** True if this handler has been installed. */
static volatile bool g_isEnabled = false;

/** True if the handler should capture the next stack trace. */
static bool g_captureNextStackTrace = false;

static std::terminate_handler g_originalTerminateHandler;

static char g_eventID[37];

static KSCrash_MonitorContext g_monitorContext;

// TODO: Thread local storage is not supported < ios 9.
// Find some other way to do thread local. Maybe storage with lookup by tid?
static KSStackCursor g_stackCursor;

/** True if the cxa_throw capture the stack**/
static bool g_capturedStackCursor = false;

// ============================================================================
#pragma mark - Callbacks -
// ============================================================================

//typedef void (*cxa_throw_type)(void*, std::type_info*, void (*)(void*));
//
//extern "C"
//{
//    void __cxa_throw(void* thrown_exception, std::type_info* tinfo, void (*dest)(void*)) __attribute__ ((weak));
//
//    void __cxa_throw(void* thrown_exception, std::type_info* tinfo, void (*dest)(void*))
//    {
//        if(g_captureNextStackTrace)
//        {
//            kssc_initSelfThread(&g_stackCursor, 1);
//            g_capturedStackCursor = true;
//        }
//
//        static cxa_throw_type orig_cxa_throw = NULL;
//        unlikely_if(orig_cxa_throw == NULL)
//        {
//            orig_cxa_throw = (cxa_throw_type) dlsym(RTLD_NEXT, "__cxa_throw");
//        }
//        orig_cxa_throw(thrown_exception, tinfo, dest);
//        __builtin_unreachable();
//    }
//}

static void (*orig_cxa_throw)(void* thrown_exception, std::type_info* tinfo, void (*dest)(void*));

void my_cxa_throw(void* thrown_exception, std::type_info* tinfo, void (*dest)(void*))
{
    if(g_captureNextStackTrace)
    {
        kssc_initSelfThread(&g_stackCursor, 1);
        g_capturedStackCursor = true;
        KSLOG_DEBUG("Trapped C++ Stack");
    }
    return orig_cxa_throw(thrown_exception, tinfo, dest);
}

static void CPPExceptionTerminate(void)
{
    ksmc_suspendEnvironment();
    KSLOG_DEBUG("Trapped c++ exception");
    const char* name = NULL;
    std::type_info* tinfo = __cxxabiv1::__cxa_current_exception_type();
    if(tinfo != NULL)
    {
        name = tinfo->name();
    }
    KSLOG_DEBUG("name:%s capured:%d", name, g_capturedStackCursor);
    // 之前判断是否是cpp exception的条件，继承自NSException的NSException会被当做cpp exception处理
    // if (name == NULL || strcmp(name, "NSException") != 0
    if (g_capturedStackCursor && (name == NULL || strcmp(name, "NSException") != 0))
    {
        kscm_notifyFatalExceptionCaptured(false);
        KSCrash_MonitorContext* crashContext = &g_monitorContext;
        memset(crashContext, 0, sizeof(*crashContext));

        char descriptionBuff[DESCRIPTION_BUFFER_LENGTH];
        const char* description = descriptionBuff;
        descriptionBuff[0] = 0;

        KSLOG_DEBUG("Discovering what kind of exception was thrown.");
        g_captureNextStackTrace = false;
        try
        {
            throw;
        }
        catch(std::exception& exc)
        {
            strncpy(descriptionBuff, exc.what(), sizeof(descriptionBuff));
        }
#define CATCH_VALUE(TYPE, PRINTFTYPE) \
catch(TYPE value)\
{ \
    snprintf(descriptionBuff, sizeof(descriptionBuff), "%" #PRINTFTYPE, value); \
}
        CATCH_VALUE(char,                 d)
        CATCH_VALUE(short,                d)
        CATCH_VALUE(int,                  d)
        CATCH_VALUE(long,                ld)
        CATCH_VALUE(long long,          lld)
        CATCH_VALUE(unsigned char,        u)
        CATCH_VALUE(unsigned short,       u)
        CATCH_VALUE(unsigned int,         u)
        CATCH_VALUE(unsigned long,       lu)
        CATCH_VALUE(unsigned long long, llu)
        CATCH_VALUE(float,                f)
        CATCH_VALUE(double,               f)
        CATCH_VALUE(long double,         Lf)
        CATCH_VALUE(char*,                s)
        catch(...)
        {
            description = NULL;
        }
        g_captureNextStackTrace = g_isEnabled;

        // TODO: Should this be done here? Maybe better in the exception handler?
        KSMC_NEW_CONTEXT(machineContext);
        ksmc_getContextForThread(ksthread_self(), machineContext, true);

        KSLOG_DEBUG("Filling out context.");
        crashContext->crashType = KSCrashMonitorTypeCPPException;
        crashContext->eventID = g_eventID;
        crashContext->registersAreValid = false;
        crashContext->stackCursor = &g_stackCursor;
        crashContext->CPPException.name = name;
        crashContext->exceptionName = name;
        crashContext->crashReason = description;
        crashContext->offendingMachineContext = machineContext;

        ksTmpSingal.si_signo = SIGABRT;
        ksTmpSingal.si_pid = (int)ksmc_getThreadFromContext(machineContext);

        kscm_handleException(crashContext);
        ksmc_resumeEnvironment();
        kscm_innerHandleSignal(&ksTmpSingal);
        kscm_handleSignal(&ksTmpSingal);
    }
    else
    {
        KSLOG_DEBUG("Detected NSException. Letting the current NSException handler deal with it.");
        ksmc_resumeEnvironment();
    }


    KSLOG_DEBUG("Calling original terminate handler.");
    g_originalTerminateHandler();
}


// ============================================================================
#pragma mark - Public API -
// ============================================================================

static void initialize()
{
    static bool isInitialized = false;
    if(!isInitialized)
    {
        isInitialized = true;
        kssc_initCursor(&g_stackCursor, NULL, NULL);
    }
}

static void setEnabled(bool isEnabled)
{
    if(isEnabled != g_isEnabled)
    {
        g_isEnabled = isEnabled;
        if(isEnabled)
        {
            initialize();

            ksid_generate(g_eventID);
            g_originalTerminateHandler = std::set_terminate(CPPExceptionTerminate);
            
            ks_rebinding item;
            item.name = "__cxa_throw";
            item.replacement = (void *)my_cxa_throw;
            item.replaced = (void **)&orig_cxa_throw;
            ks_rebind_symbols(&item, 1);
//            int ret_val = ks_rebind_symbols(&item, 1);
//            KSLOG_DEBUG("rebind ret: %d", ret_val);
        }
        else
        {
            std::set_terminate(g_originalTerminateHandler);
        }
        g_captureNextStackTrace = isEnabled;
    }
}

static bool isEnabled()
{
    return g_isEnabled;
}

extern "C" KSCrashMonitorAPI* kscm_cppexception_getAPI()
{
    static KSCrashMonitorAPI api =
    {
        .setEnabled = setEnabled,
        .isEnabled = isEnabled
    };
    return &api;
}
