package org.jtester.module.database.environment;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Map;

import javax.sql.DataSource;

import org.jtester.module.database.environment.normalise.NameNormaliser;
import org.jtester.module.dbfit.db.model.DbParameterAccessor;

@SuppressWarnings({ "rawtypes" })
public interface DBEnvironment {
	/**
	 * 默认的数据源名称
	 */
	public final static String DEFAULT_DATASOURCE_NAME = "DEFAULT";

	/**
	 * jTester配置文件
	 */
	public final static String DEFAULT_DATASOURCE_FROM = "JTESTER-PROPERTIES";

	/**
	 * 自定义数据源
	 */
	public final static String CUSTOMIZED_DATASOURCE_NAME = "CUSTOMIZED";

	/**
	 * Meta-data retrieval method that provides a list of parameters for a given
	 * stored procedure or function name. The name may contain a schema
	 * qualifier.
	 * 
	 * While implementing, use {@link NameNormaliser} to make sure parameters
	 * are mapped properly.
	 * 
	 * Parameters that map to return values should have an empty string for the
	 * name.
	 */
	Map<String, DbParameterAccessor> getAllProcedureParameters(String procName) throws SQLException;

	/**
	 * Meta-data retrieval method that provides a list of columns a given stored
	 * table or view. The name may contain a schema qualifier.
	 */
	Map<String, DbParameterAccessor> getAllColumns(String tableOrViewName) throws SQLException;

	/**
	 * This method creates an insert command that will be used to populate new
	 * rows in a table.
	 */
	String buildInsertCommand(String tableName, DbParameterAccessor[] accessors) throws SQLException;

	/**
	 * 构造delete语句
	 * 
	 * @param tableName
	 * @param accessors
	 * @return
	 */
	String buildDeleteCommand(String tableName, DbParameterAccessor[] accessors) throws SQLException;

	/**
	 * A flag to signal whether the database supports output parameters on
	 * insert commands.
	 * 
	 * If this method returns true, then output parameters will be used to
	 * retrieve possible outputs after an insert. if not, then we can only try
	 * to fetch new IDs using the JDBC support for autogenerated ID retrieval.
	 */
	boolean supportsOuputOnInsert();

	/**
	 * Create a {@link PreparedStatement} object and binds fixture symbols to
	 * SQL statement parameters with matching names.
	 */
	PreparedStatement createStatementWithBoundFixtureSymbols(String commandText) throws SQLException;

	/**
	 * 关闭当前数据源<br>
	 * <br>
	 * Closes the current connection and rolls back or commit any active
	 * transactions.
	 */
	void close();

	/**
	 * 设置数据库连接属性
	 * 
	 * @param driver
	 *            驱动类
	 * @param url
	 *            连接url
	 * @param username
	 *            数据库用户
	 * @param password
	 *            用户密码
	 */
	void setDataSource(String driver, String url, String schemas, String username, String password);

	/**
	 * 获取当前的DataSource
	 * 
	 * @param withTransactionManager
	 *            是否包装事务
	 * @return
	 */
	DataSource getDataSource(boolean withTransactionManager);

	/**
	 * 连接数据源
	 * 
	 * @throws SQLException
	 */
	Connection connect() throws SQLException;

	/**
	 * 连接数据源，如果先前没有建立过连接的话
	 * 
	 * @param environment
	 * @return 返回数据源连接
	 * @throws SQLException
	 */
	Connection connectIfNeeded();

	/**
	 * Commit current transaction.
	 */
	void commit() throws SQLException;

	/**
	 * Rollback current transaction.
	 */
	void rollback() throws SQLException;

	/**
	 * Retrieve an exception code from a database exception. This method should
	 * perform any required conversion between a JDBC exception and the real
	 * database error code.
	 */
	int getExceptionCode(SQLException ex);

	/**
	 * Retrieve current connection. Could be used by 3rd party classes to
	 * execute database commands in the same session.
	 */
	Connection getConnection();

	/**
	 * Get the Java class that should be used to store objects of a DB specific
	 * data type.
	 * 
	 * @param dataType
	 *            DB data type name
	 */
	Class getJavaClass(String dataType);

	void teardown() throws SQLException;

	/**
	 * 获得数据表的元信息
	 * 
	 * @param table
	 * @return
	 * @throws Exception
	 */
	TableMeta getTableMetaData(String table);

	/**
	 * 返回指定类型的默认值
	 * 
	 * @param javaType
	 * @return
	 */
	Object getDefaultValue(String javaType);

	/**
	 * 将字符串类型转换为java对象
	 * 
	 * @param input
	 * @param javaType
	 * @return
	 */
	Object toObjectValue(String input, String javaType);

	/**
	 * 返回数据库的字段引号符(mysql:"'"),oracle(")
	 * 
	 * @return
	 */
	String getFieldQuato();
	
	/**
	 * 过滤不需要设置默认值的字段
	 * 
	 * @param typeName		字段类型名称
	 * @return
	 */
	boolean isfilterMetaDataNeeded(String typeName);
	
	/**
	 * 重置主键
	 * 
	 */
	void resetPrimaryKey(String table);
	
	/**
	 * 准备执行脚本
	 * @param table		表名
	 * @param sql		待执行脚本
	 * @return
	 */
	public String preExecute(String table, String sql);
}
