<?php

/**
 * Class PDOPostgres
 * 	This class is used from class PDO to manage a PostgreSQL database.
 *      Look at PDO.clas.php file comments to know more about PostgreSQL connection.
 * ---------------------------------------------
 * @Author		Andrea Giammarchi
 * @Site		http://www.devpro.it/
 * @Mail		andrea [ at ] 3site [ dot ] it
 */ 
class PDOPostgres {
	
	/**
	*	__connection:Resource		Database connection
	*	__dbinfo:String			Database connection params
	*      __persistent:Boolean		Connection mode, is true on persistent, false on normal (deafult) connection
	*      __errorCode:String		Last error code
	*      __errorInfo:array		Detailed errors
	*      __result:Resource		Last query resource
	*/
	protected $__connection;
	protected $__dbinfo;
	protected $__persistent = false;
	protected $__errorCode = '';
	protected $__errorInfo = array('');
	protected $__result = null;
	protected $__throwExceptions = false;
	protected $__container_pdo;
	public $logging = false;
	
	/**
	 *	Checks connection and database selection
	 *       	new PDO_pgsql( &$host:String, &$db:String, &$user:String, &$pass:String )
	 * @Param	String		Database connection params
	 */
	function __construct(&$string_dsn) {
		if(!@$this->__connection = &pg_connect($string_dsn))
			$this->__setErrors('DBCON', true);
		else
			$this->__dbinfo = &$string_dsn;
	}

	function setContainerPDO(PDO $pdo){
		$this->__container_pdo = $pdo;
	}
	
	/**
	 * Calls pg_close function.
	 *	this->close( Void ):Boolean
	 * @Return	Boolean		True on success, false otherwise
	 */
	function close() {
		$result = is_resource($this->__connection);
		if($result)
			pg_close($this->__connection);
		return $result;
	}
	
	/**
	 * Public method:
	 *	Returns a code rappresentation of an error
	 *       	this->errorCode( void ):String
	 * @Return	String		String rappresentation of the error
	 */
	function errorCode() {
		return $this->__errorCode;
	}
	
	/**
	 *	Returns an array with error informations
	 *       	this->errorInfo( void ):array
	 * @Return	array		array with 3 keys:
	 * 				0 => error code
	 *                              1 => error number
	 *                              2 => error string
	 */
	function errorInfo() {
		return $this->__errorInfo;
	}
	
	/**
	 *	Excecutes a query and returns affected rows
	 *       	this->exec( $query:String ):Mixed
	 * @Param	String		query to execute
	 * @Return	Mixed		Number of affected rows or false on bad query.
	 */
	function exec($query) {
		$result = 0;
		$this->__uquery($query);
		if(!is_null($this->__result))
			$result = pg_affected_rows($this->__result);
		if(is_null($result))
			$result = false;
		return $result;
	}
	
	/** NOT REALLY SUPPORTED, returned value is not last inserted id
	 *	Returns pg_last_oid function
	 *       	this->lastInsertId( void ):String
	 * @Return	String		OID returned from Postgre
	 */
	function lastInsertId() {
		$result = 0;
		if(!is_null($this->__result))
			$result =  pg_last_oid($this->__result);
		return $result;
	}

	/**
	 *	Returns a new PDOStatement
	 *       	this->prepare( $query:String, $array:array ):PDOStatement
	 * @Param	String		query to prepare
	 * @Param	array		this variable is not used but respects PDO original accepted parameters
	 * @Return	PDOStatementPostgres
	 */
	function prepare($query, $array = array()) {
		return new PDOStatementPostgres($query, $this->__connection, $this->__dbinfo, $this->__container_pdo);
	}
	
	/**
	 *	Executes directly a query and returns an array with result or false on bad query
	 *       	this->query( $query:String ):Mixed
	 * @Param	String		query to execute
	 * @Return	PDOStatementPostgres
	 */
	function query($query) {
    	$statement = new PDOStatementPostgres($query, $this->__connection, $this->__dbinfo, $this->__container_pdo);
		$statement->query();
		return $statement;
	}
	
	/**
	 *	Quotes correctly a string for this database
	 *       	this->quote( $string:String ):String
	 * @Param	String		string to quote
	 * @Return	String		a correctly quoted string
	 */
	function quote($string) {
		return ("'".pg_escape_string($string)."'");
	}
	
	/**
	 *	Quotes correctly a string for this database
	 *       	this->getAttribute( $attribute:Integer ):Mixed
	 * @Param	Integer		a constant [	PDO_ATTR_SERVER_INFO,
	 * 						PDO_ATTR_SERVER_VERSION,
	 *                                              PDO_ATTR_CLIENT_VERSION,
	 *                                              PDO_ATTR_PERSISTENT	]
	 * @Return	Mixed		correct information or false
	 */
	function getAttribute($attribute) {
		$result = false;
		switch($attribute) {
			case PDO::ATTR_SERVER_INFO:
				$result = pg_parameter_status($this->__connection, 'server_encoding');
				break;
			case PDO::ATTR_SERVER_VERSION:
				$result = pg_parameter_status($this->__connection, 'server_version');
				break;
			case PDO::ATTR_CLIENT_VERSION:
				$result = pg_parameter_status($this->__connection, 'server_version');
				$result .= ' '.pg_parameter_status($this->__connection, 'client_encoding');
				break;
			case PDO::ATTR_PERSISTENT:
				$result = $this->__persistent;
				break;
		}
		return $result;
	}
	
	/**
	 *	Sets database attributes, in this version only connection mode.
	 *       	this->setAttribute( $attribute:Integer, $mixed:Mixed ):Boolean
	 * @Param	Integer		PDO_* constant, in this case only PDO_ATTR_PERSISTENT
	 * @Param	Mixed		value for PDO_* constant, in this case a Boolean value
	 * 				true for permanent connection, false for default not permament connection
	 * @Return	Boolean		true on change, false otherwise
	 */
	function setAttribute($attribute, $mixed) {
		$result = false;
		if($attribute === PDO::ATTR_ERRMODE && $mixed ===PDO::ERRMODE_EXCEPTION){
			$this->__throwExceptions = true;
		}
		elseif($attribute == PDO::ATTR_STATEMENT_CLASS && $mixed == 'LoggedPDOStatement'){
			$this->logging = true;
		}
		if($attribute === PDO::ATTR_PERSISTENT && $mixed != $this->__persistent) {
			$result = true;
			$this->__persistent = (boolean) $mixed;
			pg_close($this->__connection);
			if($this->__persistent === true)
				$this->__connection = &pg_connect($this->__dbinfo);
			else
				$this->__connection = &pg_pconnect($this->__dbinfo);
		}
		return $result;
	}
	
	function beginTransaction() {
		return false;
	}
	
	function commit() {
		return false;
	}
	
	function rollBack() {
		return false;
	}
	
	function __setErrors($er) {
		if(!is_string($this->__errorCode))
			$errno = $this->__errorCode;
		if(!is_resource($this->__connection)) {
			$errno = 1;
			$errst = pg_last_error();
		}
		else {
			$errno = 1;
			$errst = pg_last_error($this->__connection);
		}
		throw new PDOException("Database error ($errno): $errst");
		$this->__errorCode = &$er;
		$this->__errorInfo = array($this->__errorCode, $errno, $errst);
	}
	
	function __uquery(&$query) {
		if(!@$this->__result = pg_query($this->__connection, $query)) {
			$this->__setErrors('SQLER');
			$this->__result = null;
		}
		return $this->__result;
	}
}