/*
 * Copyright (C) 2010, British Broadcasting Corporation
 * All Rights Reserved.
 *
 * Author: Philip de Nier
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the British Broadcasting Corporation nor the names
 *       of its contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <bmx/mxf_op1a/OP1APCMTrack.h>
#include <bmx/mxf_op1a/OP1AFile.h>
#include <bmx/mxf_helper/ADMCHNAMXFDescriptorHelper.h>
#include <bmx/apps/AppMCALabelHelper.h>
#include <bmx/MXFUtils.h>
#include <bmx/Utils.h>
#include <bmx/BMXException.h>
#include <bmx/Logging.h>

using namespace std;
using namespace bmx;
using namespace mxfpp;


static const mxfKey BWF_ELEMENT_FW_KEY    = MXF_AES3BWF_EE_K(0x01, MXF_BWF_FRAME_WRAPPED_EE_TYPE, 0x00);
static const mxfKey BWF_ELEMENT_CW_KEY    = MXF_AES3BWF_EE_K(0x01, MXF_BWF_CLIP_WRAPPED_EE_TYPE, 0x00);
static const mxfKey AES3_ELEMENT_FW_KEY   = MXF_AES3BWF_EE_K(0x01, MXF_AES3_FRAME_WRAPPED_EE_TYPE, 0x00);
static const mxfKey AES3_ELEMENT_CW_KEY   = MXF_AES3BWF_EE_K(0x01, MXF_AES3_CLIP_WRAPPED_EE_TYPE, 0x00);
static const uint8_t AUDIO_ELEMENT_CW_LLEN  = 8;



OP1APCMTrack::OP1APCMTrack(OP1AFile *file, uint32_t track_index, uint32_t track_id, uint8_t track_type_number,
                           mxfRational frame_rate, EssenceType essence_type)
: OP1ATrack(file, track_index, track_id, track_type_number, frame_rate, essence_type)
{
    mCHNAChunk = 0;

    BMX_ASSERT(essence_type == WAVE_PCM);

    mWaveDescriptorHelper = dynamic_cast<WaveMXFDescriptorHelper*>(mDescriptorHelper);
    BMX_ASSERT(mWaveDescriptorHelper);

    mWaveDescriptorHelper->SetSamplingRate(SAMPLING_RATE_48K);
    mWaveDescriptorHelper->SetQuantizationBits(16);
    mWaveDescriptorHelper->SetChannelCount(1);

    if (mOP1AFile->IsFrameWrapped()) {
        if ((file->GetFlavour() & OP1A_ARD_ZDF_HDF_PROFILE_FLAVOUR))
            mWaveDescriptorHelper->SetSampleRate(SAMPLING_RATE_48K);
    } else {
        mEditRate = mWaveDescriptorHelper->GetSamplingRate();
        mWaveDescriptorHelper->SetSampleRate(mEditRate);
    }

    if ((file->GetFlavour() & OP1A_ARD_ZDF_HDF_PROFILE_FLAVOUR) ||
        (file->GetFlavour() & OP1A_AES_FLAVOUR))
        SetAES3Mapping(true);
    else
        SetAES3Mapping(false);

    if ((file->GetFlavour() & OP1A_IMF_FLAVOUR))
        mWaveDescriptorHelper->SetChannelAssignment(IMF_MCA_LABEL_FRAMEWORK);

    SetSampleSequence();
}

OP1APCMTrack::~OP1APCMTrack()
{
    delete mCHNAChunk;
}

void OP1APCMTrack::SetAES3Mapping(bool enable)
{
    if (enable) {
        mWaveDescriptorHelper->SetUseAES3AudioDescriptor(true);
        if (mOP1AFile->IsFrameWrapped()) {
            mTrackNumber = MXF_AES3BWF_TRACK_NUM(0x01, MXF_AES3_FRAME_WRAPPED_EE_TYPE, 0x00);
            mEssenceElementKey = AES3_ELEMENT_FW_KEY;
        } else {
            mTrackNumber = MXF_AES3BWF_TRACK_NUM(0x01, MXF_AES3_CLIP_WRAPPED_EE_TYPE, 0x00);
            mEssenceElementKey = AES3_ELEMENT_CW_KEY;
        }
    } else {
        mWaveDescriptorHelper->SetUseAES3AudioDescriptor(false);
        if (mOP1AFile->IsFrameWrapped()) {
            mTrackNumber = MXF_AES3BWF_TRACK_NUM(0x01, MXF_BWF_FRAME_WRAPPED_EE_TYPE, 0x00);
            mEssenceElementKey = BWF_ELEMENT_FW_KEY;
        } else {
            mTrackNumber = MXF_AES3BWF_TRACK_NUM(0x01, MXF_BWF_CLIP_WRAPPED_EE_TYPE, 0x00);
            mEssenceElementKey = BWF_ELEMENT_CW_KEY;
        }
    }
}

void OP1APCMTrack::SetSamplingRate(mxfRational sampling_rate)
{
    BMX_CHECK(sampling_rate == SAMPLING_RATE_48K);

    mWaveDescriptorHelper->SetSamplingRate(sampling_rate);

    if (mOP1AFile->IsClipWrapped()) {
        mEditRate = sampling_rate;
        mDescriptorHelper->SetSampleRate(sampling_rate);
    }

    SetSampleSequence();
}

void OP1APCMTrack::SetQuantizationBits(uint32_t bits)
{
    BMX_CHECK(bits > 0 && bits <= 32);

    mWaveDescriptorHelper->SetQuantizationBits(bits);
}

void OP1APCMTrack::SetChannelCount(uint32_t count)
{
    mWaveDescriptorHelper->SetChannelCount(count);
}

void OP1APCMTrack::SetLocked(bool locked)
{
    mWaveDescriptorHelper->SetLocked(locked);
}

void OP1APCMTrack::SetAudioRefLevel(int8_t level)
{
    mWaveDescriptorHelper->SetAudioRefLevel(level);
}

void OP1APCMTrack::SetDialNorm(int8_t dial_norm)
{
    mWaveDescriptorHelper->SetDialNorm(dial_norm);
}

void OP1APCMTrack::SetSequenceOffset(uint8_t offset)
{
    mWaveDescriptorHelper->SetSequenceOffset(offset);

    SetSampleSequence();
}

void OP1APCMTrack::SetChannelAssignment(UL label)
{
    mWaveDescriptorHelper->SetChannelAssignment(label);
}

void OP1APCMTrack::AddADMAudioID(const WaveCHNA::AudioID &audio_id)
{
    if (!mCHNAChunk)
        mCHNAChunk = new WaveCHNA();

    mCHNAChunk->AppendAudioID(audio_id);
}

void OP1APCMTrack::AddWaveChunkReference(uint32_t stream_id)
{
    mWaveChunkReferences.insert(stream_id);
}

AudioChannelLabelSubDescriptor* OP1APCMTrack::AddAudioChannelLabel(AudioChannelLabelSubDescriptor *copy_from)
{
    AudioChannelLabelSubDescriptor *desc;
    if (copy_from)
        desc = dynamic_cast<AudioChannelLabelSubDescriptor*>(copy_from->clone(mOP1AFile->GetHeaderMetadata()));
    else
        desc = new AudioChannelLabelSubDescriptor(mOP1AFile->GetHeaderMetadata());
    desc->setMCALinkID(generate_uuid());
    mMCALabels.push_back(desc);

    if (mOP1AFile->HavePreparedHeaderMetadata())
      mDescriptorHelper->GetFileDescriptor()->appendSubDescriptors(mMCALabels.back());

    return desc;
}

SoundfieldGroupLabelSubDescriptor* OP1APCMTrack::AddSoundfieldGroupLabel(SoundfieldGroupLabelSubDescriptor *copy_from)
{
    SoundfieldGroupLabelSubDescriptor *desc;
    if (copy_from) {
        desc = dynamic_cast<SoundfieldGroupLabelSubDescriptor*>(copy_from->clone(mOP1AFile->GetHeaderMetadata()));
    } else {
        desc = new SoundfieldGroupLabelSubDescriptor(mOP1AFile->GetHeaderMetadata());
        desc->setMCALinkID(generate_uuid());
    }
    mMCALabels.push_back(desc);

    if (mOP1AFile->HavePreparedHeaderMetadata())
      mDescriptorHelper->GetFileDescriptor()->appendSubDescriptors(mMCALabels.back());

    return desc;
}

ADMSoundfieldGroupLabelSubDescriptor* OP1APCMTrack::AddADMSoundfieldGroupLabel(ADMSoundfieldGroupLabelSubDescriptor *copy_from)
{
    ADMSoundfieldGroupLabelSubDescriptor *desc;
    if (copy_from) {
        desc = dynamic_cast<ADMSoundfieldGroupLabelSubDescriptor*>(copy_from->clone(mOP1AFile->GetHeaderMetadata()));
        BMX_CHECK_M(desc, ("Copying non-ADM SG label descriptor"));
    } else {
        desc = new ADMSoundfieldGroupLabelSubDescriptor(mOP1AFile->GetHeaderMetadata());
        desc->setMCALinkID(generate_uuid());
    }
    mMCALabels.push_back(desc);

    if (mOP1AFile->HavePreparedHeaderMetadata())
      mDescriptorHelper->GetFileDescriptor()->appendSubDescriptors(mMCALabels.back());

    return desc;
}

GroupOfSoundfieldGroupsLabelSubDescriptor* OP1APCMTrack::AddGroupOfSoundfieldGroupLabel(
        GroupOfSoundfieldGroupsLabelSubDescriptor *copy_from)
{
    GroupOfSoundfieldGroupsLabelSubDescriptor *desc;
    if (copy_from) {
        desc = dynamic_cast<GroupOfSoundfieldGroupsLabelSubDescriptor*>(copy_from->clone(mOP1AFile->GetHeaderMetadata()));
    } else {
        desc = new GroupOfSoundfieldGroupsLabelSubDescriptor(mOP1AFile->GetHeaderMetadata());
        desc->setMCALinkID(generate_uuid());
    }
    mMCALabels.push_back(desc);

    if (mOP1AFile->HavePreparedHeaderMetadata())
      mDescriptorHelper->GetFileDescriptor()->appendSubDescriptors(mMCALabels.back());

    return desc;
}

vector<uint32_t> OP1APCMTrack::GetShiftedSampleSequence() const
{
    vector<uint32_t> shifted_sample_sequence = mSampleSequence;
    offset_sample_sequence(shifted_sample_sequence, mWaveDescriptorHelper->GetSequenceOffset());

    return shifted_sample_sequence;
}

uint32_t OP1APCMTrack::GetChannelCount() const
{
    return mWaveDescriptorHelper->GetChannelCount();
}

mxfRational OP1APCMTrack::GetSamplingRate() const
{
    return mWaveDescriptorHelper->GetSamplingRate();
}

void OP1APCMTrack::AddHeaderMetadata(HeaderMetadata *header_metadata, MaterialPackage *material_package,
                                     SourcePackage *file_source_package)
{
    size_t i;
    for (i = 0; i < mMCALabels.size(); i++) {
        MCALabelSubDescriptor *desc = mMCALabels[i];
        BMX_CHECK(desc->validate(true));

        const AudioChannelLabelSubDescriptor *c_desc  = dynamic_cast<const AudioChannelLabelSubDescriptor*>(desc);
        if (c_desc) {
            if (c_desc->haveMCAChannelID()) {
                BMX_CHECK(c_desc->getMCAChannelID() > 0);
                BMX_CHECK(c_desc->getMCAChannelID() <= mWaveDescriptorHelper->GetChannelCount());
            } else {
                BMX_CHECK(mWaveDescriptorHelper->GetChannelCount() == 1);
            }
        }
    }

    set<uint32_t>::const_iterator iter;
    for (iter = mWaveChunkReferences.begin(); iter != mWaveChunkReferences.end(); iter++) {
        if (mOP1AFile->mWaveChunks.count(*iter) == 0) {
            BMX_EXCEPTION(("Wave chunk with stream ID %u has not been registered with OP1AFile", *iter));
        }
    }

    OP1ATrack::AddHeaderMetadata(header_metadata, material_package, file_source_package);

    for (i = 0; i < mMCALabels.size(); i++) {
        header_metadata->moveToEnd(mMCALabels[i]); // so that they appear after the descriptor in the file
        mDescriptorHelper->GetFileDescriptor()->appendSubDescriptors(mMCALabels[i]);
    }

    if (mCHNAChunk) {
        ADM_CHNASubDescriptor *chna_subdesc = convert_chunk_to_adm_chna_descriptor(header_metadata, mCHNAChunk);

        header_metadata->moveToEnd(chna_subdesc); // so that they appear after the descriptor in the file
        mDescriptorHelper->GetFileDescriptor()->appendSubDescriptors(chna_subdesc);
    }

    if (!mWaveChunkReferences.empty()) {
        RIFFChunkReferencesSubDescriptor *chunk_ref_subdesc = new RIFFChunkReferencesSubDescriptor(mOP1AFile->GetHeaderMetadata());
        chunk_ref_subdesc->setRIFFChunkStreamIDsArray(vector<uint32_t>(mWaveChunkReferences.begin(), mWaveChunkReferences.end()));

        header_metadata->moveToEnd(chunk_ref_subdesc); // so that they appear after the descriptor in the file
        mDescriptorHelper->GetFileDescriptor()->appendSubDescriptors(chunk_ref_subdesc);
    }
}

void OP1APCMTrack::PrepareWrite(uint8_t track_count)
{
    CompleteEssenceKeyAndTrackNum(track_count);

    if (mOP1AFile->IsFrameWrapped()) {
        mCPManager->RegisterSoundTrackElement(mTrackIndex, mEssenceElementKey,
                                              GetShiftedSampleSequence(), mWaveDescriptorHelper->GetSampleSize());
    } else {
        mCPManager->RegisterSoundTrackElement(mTrackIndex, mEssenceElementKey, AUDIO_ELEMENT_CW_LLEN);
    }

    mIndexTable->RegisterSoundTrackElement(mTrackIndex);
}

void OP1APCMTrack::SetSampleSequence()
{
    mSampleSequence.clear();
    BMX_CHECK(get_sample_sequence(mFrameRate, mWaveDescriptorHelper->GetSamplingRate(), &mSampleSequence));
}
