# Cline Rules for the Concurrent Project

This document outlines guidelines and conventions for using Cline (the AI assistant) within the 'concurrent' Go project. Following these rules helps ensure consistency and efficiency when making code changes.

## Project Structure Overview

The project follows a standard Go project layout:

-   `cmd/`: Main application entry points (api, gateway).
-   `core/`: Core abstractions, interfaces, models, and utilities shared across modules.
-   `x/`: Feature modules (e.g., `x/auth`, `x/message`, `x/timeline`). Each module typically contains:
    -   `handler.go`: HTTP handlers or similar entry points.
    -   `service.go`: Business logic implementation.
    -   `repository.go`: Data access logic.
    -   `model.go` (optional): Module-specific data structures.
    -   `*_test.go`: Unit and integration tests.
    -   `mock/`: Mocks for interfaces (often for repositories or services).
    -   `testdata/`: Test data files.
-   `internal/`: Internal packages not intended for external use.
-   `client/`: Client libraries for interacting with the service.
-   `go.mod`, `go.sum`: Go module dependency management.
-   `wire.go`, `wire_gen.go`: Dependency injection setup using Google Wire.

## Cline Usage Guidelines

### 1. File Navigation and Modification

-   **Specify full paths:** When asking Cline to read or modify files, provide the full path relative to the project root (`/home/totegamma/concrnt/concurrent`).
-   **Use `replace_in_file` for targeted edits:** For small changes (modifying functions, fixing typos, updating specific lines), prefer the `replace_in_file` tool. Ensure the `SEARCH` block matches the existing code *exactly*, including indentation and whitespace. Check the tool response for auto-formatting changes before subsequent edits.
-   **Use `write_to_file` for new files or major rewrites:** Use `write_to_file` when creating new files (handlers, services, tests, etc.) or when completely replacing the content of an existing file.
-   **Mocks (`gomock` via `go generate`):** Mocks for interfaces (repositories, services) are generated using `gomock`, triggered by `go generate`. Add a `//go:generate go run go.uber.org/mock/mockgen -source=<source_file.go> -destination=mock/<source_file.go>` directive at the top of the Go file defining the interface. Mock files are placed in the `mock/` subdirectory within the same module (e.g., `x/userkv/mock/repository.go`). **Do not run `mockgen` manually.** After adding or changing an interface method, run `go generate ./...` from the project root to update the mocks.

### 2. Adding New Features/Modules (e.g., `x/newfeature`)

1.  **Create module directory:** `x/newfeature/`
2.  **Define interfaces (if needed):** Often defined in `core/interfaces.go` or within the module itself if specific.
3.  **Create `repository.go`:** Implement data persistence logic. Define a `Repository` interface and a concrete implementation (e.g., `repo`). Add `//go:generate` directive for mock generation if needed.
4.  **Create `service.go`:** Implement business logic. Define a `Service` interface and a concrete implementation (e.g., `service`). Inject dependencies (like the repository). Add `//go:generate` directive for mock generation if needed.
5.  **Create `handler.go`:** Implement API endpoints or message handlers. Inject the service.
6.  **Add tests:** Create `repository_test.go`, `service_test.go`, `handler_test.go`. Use mocks generated in the `mock/` subdirectory.
7.  **Update Dependency Injection (`wire.go`):** Add providers for the new repository, service, and handler. Run `go generate ./...` to update `wire_gen.go` and generate mocks.
8.  **Register handlers/routes:** Update `cmd/api/main.go` or `cmd/gateway/main.go` to include the new handlers.

### 3. Testing

-   **Unit Tests:** Use standard Go `testing` package, `testify/assert` or `testify/require` for assertions, and `gomock` for mocking dependencies. Place test files (`*_test.go`) alongside the code they test.
-   **Integration Tests:** For tests requiring external services (PostgreSQL, Redis, Memcached), use the helpers in `internal/testutil/dockertest.go`. This package uses `dockertest` to spin up temporary Docker containers for these services. Tests requiring these should call functions like `testutil.CreateDB()`, `testutil.CreateRDB()`, etc., and use the returned cleanup function (`defer cleanup()`).
-   **Test Setup:** Test functions typically start by creating a `gomock.Controller` (`ctrl := gomock.NewController(t)`), setting up mocks (`mockRepo := mock_timeline.NewMockRepository(ctrl)`), defining expectations (`mockRepo.EXPECT()...`), and then instantiating the service/handler with the mocks.
-   **Tracing:** Test utilities (`internal/testutil`) also include helpers for setting up mock OpenTelemetry trace providers (`SetupMockTraceProvider`) and creating contexts with trace IDs (`SetupTraceCtx`, `CreateHttpRequest`) for testing trace propagation.
-   **Adapt Tests to Code:** When tests fail, prioritize adapting the test logic (e.g., mock expectations, assertions) to match the existing code's behavior or expected contract. **Do not modify the production code to make a test pass unless the code itself is demonstrably incorrect or the task explicitly requires changing the production code.**

### 4. Code Style and Conventions

-   **Formatting:** Use `gofmt` / `goimports`.
-   **Error Handling:** Return errors explicitly. Use `core/errors.go` for custom error types if applicable. Wrap errors with context using `fmt.Errorf("...: %w", err)`.
-   **Logging:** Use a structured logger (check existing code for patterns, likely injected).
-   **Naming:** Follow standard Go conventions. Interfaces often end with `-er` (e.g., `Repository`, `Service`). Implementations are often lowercase structs (e.g., `repo`, `service`). Test functions start with `Test...`.
-   **Documentation:** Add godoc comments (e.g., `// FunctionName describes...`) above all exported functions to explain their purpose, parameters, and return values.

### 5. Dependency Management

-   Use Go modules (`go.mod`, `go.sum`).
-   Use `go mod tidy` to add new dependencies or clean up unused ones. Avoid using `go get` directly.

### 6. Common Commands

-   **Run tests:** `go test ./...`
-   **Generate code (Mocks, Wire):** `go generate ./...` (This runs all `//go:generate` directives, including `mockgen` and `wire`).
-   **Build:** `go build ./cmd/api` or `go build ./cmd/gateway`
-   **Run locally (example):** `go run ./cmd/api/main.go` (check `main.go` or `compose.yml` for actual run commands/environment variables needed).
-   **Format code:** Run `go fmt ./...` before completing a task to ensure code is formatted correctly.

## Goal

The primary goal when using Cline in this project is to accelerate development by automating repetitive tasks, assisting with refactoring, and implementing well-defined features, while adhering to the established project structure and coding standards. Always review Cline's proposed changes carefully.
