package key

import (
	"context"
	"encoding/json"
	"time"

	"github.com/bradfitz/gomemcache/memcache"
	"github.com/pkg/errors"
	"gorm.io/gorm"

	"github.com/concrnt/concrnt/client"
	"github.com/concrnt/concrnt/core"
)

type Repository interface {
	Enact(ctx context.Context, key core.Key) (core.Key, error)
	Revoke(ctx context.Context, keyID string, payload string, signature string, signedAt time.Time) (core.Key, error)
	Get(ctx context.Context, keyID string) (core.Key, error)
	GetAll(ctx context.Context, owner string) ([]core.Key, error)
	GetRemoteKeyResolution(ctx context.Context, remote string, keyID string) ([]core.Key, error)
	Clean(ctx context.Context, ccid string) error
}

type repository struct {
	db     *gorm.DB
	mc     *memcache.Client
	client client.Client
}

func NewRepository(
	db *gorm.DB,
	mc *memcache.Client,
	client client.Client,
) Repository {
	return &repository{db, mc, client}
}

func (r *repository) GetRemoteKeyResolution(ctx context.Context, remote string, keyID string) ([]core.Key, error) {
	ctx, span := tracer.Start(ctx, "Key.Repository.GetRemoteKeyResolution")
	defer span.End()

	// check cache first
	item, err := r.mc.Get(keyID)
	if err == nil {
		var keys []core.Key
		err = json.Unmarshal(item.Value, &keys)
		if err != nil {
			span.RecordError(err)
			return nil, err
		}
		return keys, nil
	}
	// get from remote
	keys, err := r.client.GetKey(ctx, keyID, &client.Options{Resolver: remote})
	if err != nil {
		span.RecordError(err)
		return nil, err
	}

	_, err = core.ValidateKeyResolution(keys, keyID) // TODO: should have a negative cache
	if err != nil {
		span.RecordError(err)
		return nil, err
	}

	// cache
	value, err := json.Marshal(keys)
	if err != nil {
		span.RecordError(err)
		return nil, err
	}
	// TTL 30 minutes
	err = r.mc.Set(&memcache.Item{Key: keyID, Value: value, Expiration: 1800})
	if err != nil {
		span.RecordError(err)
		return nil, err
	}

	return keys, nil
}

// Get retrieves a key by its ID (CKID).
func (r *repository) Get(ctx context.Context, keyID string) (core.Key, error) {
	ctx, span := tracer.Start(ctx, "Key.Repository.Get")
	defer span.End()

	var key core.Key
	err := r.db.Where("id = ?", keyID).First(&key).Error
	if err != nil {
		if errors.Is(err, gorm.ErrRecordNotFound) {
			return core.Key{}, core.NewErrorNotFound()
		}
		return core.Key{}, err
	}

	return key, nil
}

// Enact creates a new key record in the database.
func (r *repository) Enact(ctx context.Context, key core.Key) (core.Key, error) {
	ctx, span := tracer.Start(ctx, "Key.Repository.Enact")
	defer span.End()

	err := r.db.Create(&key).Error
	if err != nil {
		if errors.Is(err, gorm.ErrDuplicatedKey) {
			return core.Key{}, core.NewErrorAlreadyExists()
		}
		return core.Key{}, err
	}

	return key, nil
}

// Revoke marks a key as revoked by setting its revoke document, signature, and valid until time.
func (r *repository) Revoke(ctx context.Context, keyID string, payload string, signature string, signedAt time.Time) (core.Key, error) {
	ctx, span := tracer.Start(ctx, "Key.Repository.Revoke")
	defer span.End()

	err := r.db.Model(&core.Key{}).Where("id = ?", keyID).Updates(
		core.Key{
			RevokeDocument:  &payload,
			RevokeSignature: &signature,
			ValidUntil:      signedAt,
		},
	).Error
	if err != nil {
		return core.Key{}, err
	}

	var key core.Key
	err = r.db.Where("id = ?", keyID).First(&key).Error
	if err != nil {
		return core.Key{}, err
	}

	return key, nil
}

// GetAll retrieves all keys associated with a specific owner (root CCID).
func (r *repository) GetAll(ctx context.Context, owner string) ([]core.Key, error) {
	ctx, span := tracer.Start(ctx, "Key.Repository.GetAll")
	defer span.End()

	var keys []core.Key
	err := r.db.Where("root = ?", owner).Find(&keys).Error
	if err != nil {
		return nil, err
	}

	return keys, nil
}

// Clean deletes all keys associated with a specific owner (root CCID).
func (r *repository) Clean(ctx context.Context, ccid string) error {
	ctx, span := tracer.Start(ctx, "Key.Repository.Clean")
	defer span.End()

	err := r.db.Where("root = ?", ccid).Delete(&core.Key{}).Error
	if err != nil {
		return err
	}

	return nil
}
