'use strict';

var SwaggerHttp = require('./http');
var _ = {
  isObject: require('lodash-compat/lang/isObject')
};

var SwaggerSpecConverter = module.exports = function () {
  this.errors = [];
  this.warnings = [];
  this.modelMap = {};
};

SwaggerSpecConverter.prototype.setDocumentationLocation = function (location) {
  this.docLocation = location;
};

/**
 * converts a resource listing OR api declaration
 **/
SwaggerSpecConverter.prototype.convert = function (obj, clientAuthorizations, callback) {
  // not a valid spec
  if(!obj || !Array.isArray(obj.apis)) {
    return this.finish(callback, null);
  }
  this.clientAuthorizations = clientAuthorizations;

  // create a new swagger object to return
  var swagger = { swagger: '2.0' };

  swagger.originalVersion = obj.swaggerVersion;

  // add the info
  this.apiInfo(obj, swagger);

  // add security definitions
  this.securityDefinitions(obj, swagger);

  // take basePath into account
  if (obj.basePath) {
    this.setDocumentationLocation(obj.basePath);
  }

  // see if this is a single-file swagger definition
  var isSingleFileSwagger = false;
  var i;
  for(i = 0; i < obj.apis.length; i++) {
    var api = obj.apis[i];
    if(Array.isArray(api.operations)) {
      isSingleFileSwagger = true;
    }
  }
  if(isSingleFileSwagger) {
    this.declaration(obj, swagger);
    this.finish(callback, swagger);
  }
  else {
    this.resourceListing(obj, swagger, callback);
  }
};

SwaggerSpecConverter.prototype.declaration = function(obj, swagger) {
  var name, i, p, pos;
  if(!obj.apis) {
    return;
  }

  if (obj.basePath.indexOf('http://') === 0) {
    p = obj.basePath.substring('http://'.length);
    pos = p.indexOf('/');
    if (pos > 0) {
      swagger.host = p.substring(0, pos);
      swagger.basePath = p.substring(pos);
    }
    else {
      swagger.host = p;
      swagger.basePath = '/';
    }
  } else if (obj.basePath.indexOf('https://') === 0) {
    p = obj.basePath.substring('https://'.length);
    pos = p.indexOf('/');
    if (pos > 0) {
      swagger.host = p.substring(0, pos);
      swagger.basePath = p.substring(pos);
    }
    else {
      swagger.host = p;
      swagger.basePath = '/';
    }
  } else {
    swagger.basePath = obj.basePath;
  }

  var resourceLevelAuth;
  if(obj.authorizations) {
    resourceLevelAuth = obj.authorizations;
  }
  if(obj.consumes) {
    swagger.consumes = obj.consumes;
  }
  if(obj.produces) {
    swagger.produces = obj.produces;
  }

  // build a mapping of id to name for 1.0 model resolutions
  if(_.isObject(obj)) {
    for(name in obj.models) {
      var existingModel = obj.models[name];
      var key = (existingModel.id || name);
      this.modelMap[key] = name;
    }
  }

  for(i = 0; i < obj.apis.length; i++) {
    var api = obj.apis[i];
    var path = api.path;
    var operations = api.operations;
    this.operations(path, obj.resourcePath, operations, resourceLevelAuth, swagger);
  }

  var models = obj.models || {};
  this.models(models, swagger);
};

SwaggerSpecConverter.prototype.models = function(obj, swagger) {
  if(!_.isObject(obj)) {
    return;
  }
  var name;

  swagger.definitions = swagger.definitions || {};
  for(name in obj) {
    var existingModel = obj[name];
    var _enum = [];
    var schema = { properties: {}};
    var propertyName;
    for(propertyName in existingModel.properties) {
      var existingProperty = existingModel.properties[propertyName];
      var property = {};
      this.dataType(existingProperty, property);
      if(existingProperty.description) {
        property.description = existingProperty.description;
      }
      if(existingProperty['enum']) {
        property['enum'] = existingProperty['enum'];
      }
      if(typeof existingProperty.required === 'boolean' && existingProperty.required === true) {
        _enum.push(propertyName);
      }
      if(typeof existingProperty.required === 'string' && existingProperty.required === 'true') {
        _enum.push(propertyName);
      }
      schema.properties[propertyName] = property;
    }
    if(_enum.length > 0) {
      schema['enum'] = _enum;
    }

    schema.required = existingModel.required;
    swagger.definitions[name] = schema;
  }
};

SwaggerSpecConverter.prototype.extractTag = function(resourcePath) {
  var pathString = resourcePath || 'default';
  if(pathString.indexOf('http:') === 0 || pathString.indexOf('https:') === 0) {
    pathString = pathString.split(['/']);
    pathString = pathString[pathString.length -1].substring();
  }
  if(pathString.endsWith('.json')) {
    pathString = pathString.substring(0, pathString.length - '.json'.length);
  }
  return pathString.replace('/','');
};

SwaggerSpecConverter.prototype.operations = function(path, resourcePath, obj, resourceLevelAuth, swagger) {
  if(!Array.isArray(obj)) {
    return;
  }
  var i;

  if(!swagger.paths) {
    swagger.paths = {};
  }

  var pathObj = swagger.paths[path] || {};
  var tag = this.extractTag(resourcePath);
  swagger.tags = swagger.tags || [];
  var matched = false;
  for(i = 0; i < swagger.tags.length; i++) {
    var tagObject = swagger.tags[i];
    if(tagObject.name === tag) {
      matched = true;
    }
  }
  if(!matched) {
    swagger.tags.push({name: tag});
  }

  for(i = 0; i < obj.length; i++) {
    var existingOperation = obj[i];
    var method = (existingOperation.method || existingOperation.httpMethod).toLowerCase();
    var operation = {tags: [tag]};
    var existingAuthorizations = existingOperation.authorizations;

    if(existingAuthorizations && Object.keys(existingAuthorizations).length === 0) {
      existingAuthorizations = resourceLevelAuth;
    }

    if(typeof existingAuthorizations !== 'undefined') {
      var scopesObject;
      for(var key in existingAuthorizations) {
        operation.security = operation.security || [];
        var scopes = existingAuthorizations[key];
        if(scopes) {
          var securityScopes = [];
          for(var j in scopes) {
            securityScopes.push(scopes[j].scope);
          }
          scopesObject = {};
          scopesObject[key] = securityScopes;
          operation.security.push(scopesObject);
        }
        else {
          scopesObject = {};
          scopesObject[key] = [];
          operation.security.push(scopesObject);
        }
      }
    }

    if(existingOperation.consumes) {
      operation.consumes = existingOperation.consumes;
    }
    else if(swagger.consumes) {
      operation.consumes = swagger.consumes;
    }
    if(existingOperation.produces) {
      operation.produces = existingOperation.produces;
    }
    else if(swagger.produces) {
      operation.produces = swagger.produces;
    }
    if(existingOperation.summary) {
      operation.summary = existingOperation.summary;
    }
    if(existingOperation.notes) {
      operation.description = existingOperation.notes;
    }
    if(existingOperation.nickname) {
      operation.operationId = existingOperation.nickname;
    }
    if(existingOperation.deprecated) {
      operation.deprecated = existingOperation.deprecated;
    }

    this.authorizations(existingAuthorizations, swagger);
    this.parameters(operation, existingOperation.parameters, swagger);
    this.responseMessages(operation, existingOperation, swagger);

    pathObj[method] = operation;
  }

  swagger.paths[path] = pathObj;
};

SwaggerSpecConverter.prototype.responseMessages = function(operation, existingOperation) {
  if(!_.isObject(existingOperation)) {
    return;
  }
  // build default response from the operation (1.x)
  var defaultResponse = {};
  this.dataType(existingOperation, defaultResponse);
  // TODO: look into the real problem of rendering responses in swagger-ui
  // ....should reponseType have an implicit schema?
  if(!defaultResponse.schema && defaultResponse.type) {
    defaultResponse = {schema: defaultResponse};
  }

  operation.responses = operation.responses || {};

  // grab from responseMessages (1.2)
  var has200 = false;
  if(Array.isArray(existingOperation.responseMessages)) {
    var i;
    var existingResponses = existingOperation.responseMessages;
    for(i = 0; i < existingResponses.length; i++) {
      var existingResponse = existingResponses[i];
      var response = { description: existingResponse.message };
      if(existingResponse.code === 200) {
        has200 = true;
      }
      // Convert responseModel -> schema{$ref: responseModel}
      if(existingResponse.responseModel) {
        response.schema = {'$ref': existingResponse.responseModel};
      }
      operation.responses['' + existingResponse.code] = response;
    }
  }

  if(has200) {
    operation.responses['default'] = defaultResponse;
  }
  else {
    operation.responses['200'] = defaultResponse;
  }
};

SwaggerSpecConverter.prototype.authorizations = function(obj) {
  // TODO
  if(!_.isObject(obj)) {
    return;
  }
};

SwaggerSpecConverter.prototype.parameters = function(operation, obj) {
  if(!Array.isArray(obj)) {
    return;
  }
  var i;
  for(i = 0; i < obj.length; i++) {
    var existingParameter = obj[i];
    var parameter = {};
    parameter.name = existingParameter.name;
    parameter.description = existingParameter.description;
    parameter.required = existingParameter.required;
    parameter.in = existingParameter.paramType;

    // per #168
    if(parameter.in === 'body') {
      parameter.name = 'body';
    }
    if(parameter.in === 'form') {
      parameter.in = 'formData';
    }

    if(existingParameter.enum) {
      parameter.enum = existingParameter.enum;
    }

    if(existingParameter.allowMultiple === true || existingParameter.allowMultiple === 'true') {
      var innerType = {};
      this.dataType(existingParameter, innerType);
      parameter.type = 'array';
      parameter.items = innerType;

      if(existingParameter.allowableValues) {
        var av = existingParameter.allowableValues;
        if(av.valueType === 'LIST') {
          parameter['enum'] = av.values;
        }
      }
    }
    else {
      this.dataType(existingParameter, parameter);
    }
    if(typeof existingParameter.defaultValue !== 'undefined') {
      parameter.default = existingParameter.defaultValue;
    }

    operation.parameters = operation.parameters || [];
    operation.parameters.push(parameter);
  }
};

SwaggerSpecConverter.prototype.dataType = function(source, target) {
  if(!_.isObject(source)) {
    return;
  }

  if(source.minimum) {
    target.minimum = source.minimum;
  }
  if(source.maximum) {
    target.maximum = source.maximum;
  }
  if (source.format) {
    target.format = source.format;
  }

  // default can be 'false'
  if(typeof source.defaultValue !== 'undefined') {
    target.default = source.defaultValue;
  }

  var jsonSchemaType = this.toJsonSchema(source);
  if(jsonSchemaType) {
    target = target || {};
    if(jsonSchemaType.type) {
      target.type = jsonSchemaType.type;
    }
    if(jsonSchemaType.format) {
      target.format = jsonSchemaType.format;
    }
    if(jsonSchemaType.$ref) {
      target.schema = {$ref: jsonSchemaType.$ref};
    }
    if(jsonSchemaType.items) {
      target.items = jsonSchemaType.items;
    }
  }
};

SwaggerSpecConverter.prototype.toJsonSchema = function(source) {
  if(!source) {
    return 'object';
  }
  var detectedType = (source.type || source.dataType || source.responseClass || '');
  var lcType = detectedType.toLowerCase();
  var format = (source.format || '').toLowerCase();

  if(lcType.indexOf('list[') === 0) {
    var innerType = detectedType.substring(5, detectedType.length - 1);
    var jsonType = this.toJsonSchema({type: innerType});
    return {type: 'array', items: jsonType};
  }
  else if(lcType === 'int' || (lcType === 'integer' && format === 'int32'))
    {return {type: 'integer', format: 'int32'};}
  else if(lcType === 'long' || (lcType === 'integer' && format === 'int64'))
    {return {type: 'integer', format: 'int64'};}
  else if(lcType === 'integer')
    {return {type: 'integer', format: 'int64'};}
  else if(lcType === 'float' || (lcType === 'number' && format === 'float'))
    {return {type: 'number', format: 'float'};}
  else if(lcType === 'double' || (lcType === 'number' && format === 'double'))
    {return {type: 'number', format: 'double'};}
  else if((lcType === 'string' && format === 'date-time') || (lcType === 'date'))
    {return {type: 'string', format: 'date-time'};}
  else if(lcType === 'string')
    {return {type: 'string'};}
  else if(lcType === 'file')
    {return {type: 'file'};}
  else if(lcType === 'boolean')
    {return {type: 'boolean'};}
  else if(lcType === 'array' || lcType === 'list') {
    if(source.items) {
      var it = this.toJsonSchema(source.items);
      return {type: 'array', items: it};
    }
    else {
      return {type: 'array', items: {type: 'object'}};
    }
  }
  else if(source.$ref) {
    return {$ref: '#/definitions/' + this.modelMap[source.$ref] || source.$ref};
  }
  else if(lcType === 'void' || lcType === '')
    {return {};}
  else {
    return {$ref: '#/definitions/' + this.modelMap[source.type] || source.type};
  }
};

SwaggerSpecConverter.prototype.resourceListing = function(obj, swagger, callback) {
  var i;
  var processedCount = 0;   // jshint ignore:line
  var self = this;          // jshint ignore:line
  var expectedCount = obj.apis.length;
  var _swagger = swagger;   // jshint ignore:line

  if(expectedCount === 0) {
    this.finish(callback, swagger);
  }

  for(i = 0; i < expectedCount; i++) {
    var api = obj.apis[i];
    var path = api.path;
    var absolutePath = this.getAbsolutePath(obj.swaggerVersion, this.docLocation, path);

    if(api.description) {
      swagger.tags = swagger.tags || [];
      swagger.tags.push({
        name : this.extractTag(api.path),
        description : api.description || ''
      });
    }
    var http = {
      url: absolutePath,
      headers: {accept: 'application/json'},
      on: {},
      method: 'get'
    };
    /* jshint ignore:start */
    http.on.response = function(data) {
      processedCount += 1;
      var obj = data.obj;
      if(obj) {
        self.declaration(obj, _swagger);
      }
      if(processedCount === expectedCount) {
        self.finish(callback, _swagger);
      }
    };
    http.on.error = function(data) {
      console.error(data);
      processedCount += 1;
      if(processedCount === expectedCount) {
        self.finish(callback, _swagger);
      }
    };
    /* jshint ignore:end */

    if(this.clientAuthorizations && typeof this.clientAuthorizations.apply === 'function') {
      this.clientAuthorizations.apply(http);
    }

    new SwaggerHttp().execute(http);
  }
};

SwaggerSpecConverter.prototype.getAbsolutePath = function(version, docLocation, path)  {
  if(version === '1.0') {
    if(docLocation.endsWith('.json')) {
      // get root path
      var pos = docLocation.lastIndexOf('/');
      if(pos > 0) {
        docLocation = docLocation.substring(0, pos);
      }
    }
  }

  var location = docLocation;
  if(path.indexOf('http://') === 0 || path.indexOf('https://') === 0) {
    location = path;
  }
  else {
    if(docLocation.endsWith('/')) {
      location = docLocation.substring(0, docLocation.length - 1);
    }
    location += path;
  }
  location = location.replace('{format}', 'json');
  return location;
};

SwaggerSpecConverter.prototype.securityDefinitions = function(obj, swagger) {
  if(obj.authorizations) {
    var name;
    for(name in obj.authorizations) {
      var isValid = false;
      var securityDefinition = {};
      var definition = obj.authorizations[name];
      if(definition.type === 'apiKey') {
        securityDefinition.type = 'apiKey';
        securityDefinition.in = definition.passAs;
        securityDefinition.name = definition.keyname || name;
        isValid = true;
      }
      else if(definition.type === 'oauth2') {
        var existingScopes = definition.scopes || [];
        var scopes = {};
        var i;
        for(i in existingScopes) {
          var scope = existingScopes[i];
          scopes[scope.scope] = scope.description;
        }
        securityDefinition.type = 'oauth2';
        if(i > 0) {
          securityDefinition.scopes = scopes;
        }
        if(definition.grantTypes) {
          if(definition.grantTypes.implicit) {
            var implicit = definition.grantTypes.implicit;
            securityDefinition.flow = 'implicit';
            securityDefinition.authorizationUrl = implicit.loginEndpoint;
            isValid = true;
          }
          /* jshint ignore:start */
          if(definition.grantTypes['authorization_code']) {
            if(!securityDefinition.flow) {
              // cannot set if flow is already defined
              var authCode = definition.grantTypes['authorization_code'];
              securityDefinition.flow = 'accessCode';
              securityDefinition.authorizationUrl = authCode.tokenRequestEndpoint.url;
              securityDefinition.tokenUrl = authCode.tokenEndpoint.url;
              isValid = true;
            }
          }
          /* jshint ignore:end */
        }
      }
      if(isValid) {
        swagger.securityDefinitions = swagger.securityDefinitions || {};
        swagger.securityDefinitions[name] = securityDefinition;
      }
    }
  }
};

SwaggerSpecConverter.prototype.apiInfo = function(obj, swagger) {
  // info section
  if(obj.info) {
    var info = obj.info;
    swagger.info = {};

    if(info.contact) {
      swagger.info.contact = {};
      swagger.info.contact.email = info.contact;
    }
    if(info.description) {
      swagger.info.description = info.description;
    }
    if(info.title) {
      swagger.info.title = info.title;
    }
    if(info.termsOfServiceUrl) {
      swagger.info.termsOfService = info.termsOfServiceUrl;
    }
    if(info.license || info.licenseUrl) {
      swagger.license = {};
      if(info.license) {
        swagger.license.name = info.license;
      }
      if(info.licenseUrl) {
        swagger.license.url = info.licenseUrl;
      }
    }
  }
  else {
    this.warnings.push('missing info section');
  }
};

SwaggerSpecConverter.prototype.finish = function (callback, obj) {
  callback(obj);
};
