﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Microsoft.Maui.Controls;
using Microsoft.Maui.Graphics;
using Microsoft.Maui.Layouts;
using Microsoft.Maui.Primitives;
using NSubstitute;
using Xunit;
using static Microsoft.Maui.UnitTests.Layouts.LayoutTestHelpers;

namespace Microsoft.Maui.UnitTests.Layouts
{
	[Category(TestCategory.Layout)]
	public class GridLayoutManagerTests
	{
		const string GridSpacing = "GridSpacing";
		const string GridAutoSizing = "GridAutoSizing";
		const string GridStarSizing = "GridStarSizing";
		const string GridAbsoluteSizing = "GridAbsoluteSizing";
		const string GridSpan = "GridSpan";

		IGridLayout CreateGridLayout(int rowSpacing = 0, int colSpacing = 0,
			string rows = null, string columns = null, IList<IView> children = null)
		{
			IEnumerable<IGridRowDefinition> rowDefs = null;
			IEnumerable<IGridColumnDefinition> colDefs = null;

			if (rows != null)
			{
				rowDefs = CreateTestRows(rows.Split(","));
			}

			if (columns != null)
			{
				colDefs = CreateTestColumns(columns.Split(","));
			}

			var grid = Substitute.For<IGridLayout>();

			grid.Height.Returns(Dimension.Unset);
			grid.Width.Returns(Dimension.Unset);
			grid.MinimumHeight.Returns(Dimension.Minimum);
			grid.MinimumWidth.Returns(Dimension.Minimum);
			grid.MaximumHeight.Returns(Dimension.Maximum);
			grid.MaximumWidth.Returns(Dimension.Maximum);

			grid.RowSpacing.Returns(rowSpacing);
			grid.ColumnSpacing.Returns(colSpacing);

			SubRowDefs(grid, rowDefs);
			SubColDefs(grid, colDefs);

			if (children != null)
			{
				SubstituteChildren(grid, children);
			}

			return grid;
		}

		void SubRowDefs(IGridLayout grid, IEnumerable<IGridRowDefinition> rows = null)
		{
			if (rows == null)
			{
				var rowDefs = new List<IGridRowDefinition>();
				grid.RowDefinitions.Returns(rowDefs);
			}
			else
			{
				grid.RowDefinitions.Returns(rows);
			}
		}

		void SubColDefs(IGridLayout grid, IEnumerable<IGridColumnDefinition> cols = null)
		{
			if (cols == null)
			{
				var colDefs = new List<IGridColumnDefinition>();
				grid.ColumnDefinitions.Returns(colDefs);
			}
			else
			{
				grid.ColumnDefinitions.Returns(cols);
			}
		}

		List<IGridColumnDefinition> CreateTestColumns(params string[] columnWidths)
		{
			var converter = new GridLengthTypeConverter();

			var colDefs = new List<IGridColumnDefinition>();

			foreach (var width in columnWidths)
			{
				var gridLength = converter.ConvertFromInvariantString(width);
				var colDef = Substitute.For<IGridColumnDefinition>();
				colDef.Width.Returns(gridLength);
				colDefs.Add(colDef);
			}

			return colDefs;
		}

		List<IGridRowDefinition> CreateTestRows(params string[] rowHeights)
		{
			var converter = new GridLengthTypeConverter();

			var rowDefs = new List<IGridRowDefinition>();

			foreach (var height in rowHeights)
			{
				var gridLength = converter.ConvertFromInvariantString(height);
				var rowDef = Substitute.For<IGridRowDefinition>();
				rowDef.Height.Returns(gridLength);
				rowDefs.Add(rowDef);
			}

			return rowDefs;
		}

		void SetLocation(IGridLayout grid, IView view, int row = 0, int col = 0, int rowSpan = 1, int colSpan = 1)
		{
			grid.GetRow(view).Returns(row);
			grid.GetRowSpan(view).Returns(rowSpan);
			grid.GetColumn(view).Returns(col);
			grid.GetColumnSpan(view).Returns(colSpan);
		}

		Size MeasureAndArrange(IGridLayout grid, double widthConstraint = double.PositiveInfinity, double heightConstraint = double.PositiveInfinity, double left = 0, double top = 0)
		{
			var manager = new GridLayoutManager(grid);
			var measuredSize = manager.Measure(widthConstraint, heightConstraint);
			manager.ArrangeChildren(new Rectangle(new Point(left, top), measuredSize));

			return measuredSize;
		}

		[Category(GridAutoSizing)]
		[Fact]
		public void OneAutoRowOneAutoColumn()
		{
			// A one-row, one-column grid
			var grid = CreateGridLayout();

			// A 100x100 IView
			var view = CreateTestView(new Size(100, 100));

			// Set up the grid to have a single child
			SubstituteChildren(grid, view);

			// Set up the row/column values and spans
			SetLocation(grid, view);

			MeasureAndArrange(grid, double.PositiveInfinity, double.PositiveInfinity);

			// No rows/columns were specified, so the implied */* is used; we're measuring with infinity, so
			// we expect that the view will be arranged at its measured size
			AssertArranged(view, 0, 0, 100, 100);
		}

		[Category(GridAbsoluteSizing)]
		[Fact]
		public void TwoAbsoluteColumnsOneAbsoluteRow()
		{
			var grid = CreateGridLayout(columns: "100, 100", rows: "10");

			var viewSize = new Size(10, 10);

			var view0 = CreateTestView(viewSize);
			var view1 = CreateTestView(viewSize);

			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0);
			SetLocation(grid, view1, col: 1);

			// Assuming no constraints on space
			MeasureAndArrange(grid, double.PositiveInfinity, double.NegativeInfinity);

			// Column width is 100, viewSize is less than that, so it should be able to layout out at full size
			AssertArranged(view0, 0, 0, 100, 10);

			// Since the first column is 100 wide, we expect the view in the second column to start at x = 100
			AssertArranged(view1, 100, 0, 100, 10);
		}

		[Category(GridAbsoluteSizing)]
		[Fact]
		public void TwoAbsoluteRowsAndColumns()
		{
			var grid = CreateGridLayout(columns: "100, 100", rows: "10, 30");

			var viewSize = new Size(10, 10);

			var view0 = CreateTestView(viewSize);
			var view1 = CreateTestView(viewSize);
			var view2 = CreateTestView(viewSize);
			var view3 = CreateTestView(viewSize);

			SubstituteChildren(grid, view0, view1, view2, view3);

			SetLocation(grid, view0);
			SetLocation(grid, view1, col: 1);
			SetLocation(grid, view2, row: 1);
			SetLocation(grid, view3, row: 1, col: 1);

			// Assuming no constraints on space
			MeasureAndArrange(grid, double.PositiveInfinity, double.NegativeInfinity);

			// Verify that the views are getting measured at all, and that they're being measured at 
			// the appropriate sizes
			view0.Received().Measure(Arg.Is<double>(100), Arg.Is<double>(10));
			view1.Received().Measure(Arg.Is<double>(100), Arg.Is<double>(10));
			view2.Received().Measure(Arg.Is<double>(100), Arg.Is<double>(30));
			view3.Received().Measure(Arg.Is<double>(100), Arg.Is<double>(30));

			AssertArranged(view0, 0, 0, 100, 10);

			// Since the first column is 100 wide, we expect the view in the second column to start at x = 100
			AssertArranged(view1, 100, 0, 100, 10);

			// First column, second row, so y should be 10
			AssertArranged(view2, 0, 10, 100, 30);

			// Second column, second row, so 100, 10
			AssertArranged(view3, 100, 10, 100, 30);
		}

		[Category(GridAbsoluteSizing), Category(GridAutoSizing)]
		[Fact]
		public void TwoAbsoluteColumnsOneAutoRow()
		{
			var grid = CreateGridLayout(columns: "100, 100");

			var viewSize = new Size(10, 10);

			var view0 = CreateTestView(viewSize);
			var view1 = CreateTestView(viewSize);

			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0);
			SetLocation(grid, view1, col: 1);

			// Assuming no constraints on space
			MeasureAndArrange(grid, double.PositiveInfinity, double.NegativeInfinity);

			// Column width is 100, viewSize is less, so it should be able to layout at full size
			AssertArranged(view0, 0, 0, 100, viewSize.Height);

			// Since the first column is 100 wide, we expect the view in the second column to start at x = 100
			AssertArranged(view1, 100, 0, 100, viewSize.Height);
		}

		[Category(GridAbsoluteSizing), Category(GridAutoSizing)]
		[Fact]
		public void TwoAbsoluteRowsOneAutoColumn()
		{
			var grid = CreateGridLayout(rows: "100, 100");

			var viewSize = new Size(10, 10);

			var view0 = CreateTestView(viewSize);
			var view1 = CreateTestView(viewSize);

			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0);
			SetLocation(grid, view1, row: 1);

			// Assuming no constraints on space
			MeasureAndArrange(grid, double.PositiveInfinity, double.NegativeInfinity);

			// Row height is 100, so full view should fit
			AssertArranged(view0, 0, 0, viewSize.Width, 100);

			// Since the first row is 100 tall, we expect the view in the second row to start at y = 100
			AssertArranged(view1, 0, 100, viewSize.Width, 100);
		}

		[Category(GridSpacing)]
		[Fact(DisplayName = "Row spacing shouldn't affect a single-row grid")]
		public void SingleRowIgnoresRowSpacing()
		{
			var grid = CreateGridLayout(rowSpacing: 10);
			var view = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			MeasureAndArrange(grid, double.PositiveInfinity, double.PositiveInfinity);
			AssertArranged(view, 0, 0, 100, 100);
		}

		[Category(GridSpacing)]
		[Fact(DisplayName = "Two rows should include the row spacing once")]
		public void TwoRowsWithSpacing()
		{
			var grid = CreateGridLayout(rows: "100, 100", rowSpacing: 10);
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view0, view1);
			SetLocation(grid, view0);
			SetLocation(grid, view1, row: 1);

			MeasureAndArrange(grid, double.PositiveInfinity, double.PositiveInfinity);
			AssertArranged(view0, 0, 0, 100, 100);

			// With column width 100 and spacing of 10, we expect the second column to start at 110
			AssertArranged(view1, 0, 110, 100, 100);
		}

		[Category(GridSpacing)]
		[Fact(DisplayName = "Measure should include row spacing")]
		public void MeasureTwoRowsWithSpacing()
		{
			var grid = CreateGridLayout(rows: "100, 100", rowSpacing: 10);
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view0, view1);
			SetLocation(grid, view0);
			SetLocation(grid, view1, row: 1);

			var manager = new GridLayoutManager(grid);
			var measure = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(100 + 100 + 10, measure.Height);
		}

		[Category(GridAutoSizing)]
		[Fact(DisplayName = "Auto rows without content have height zero")]
		public void EmptyAutoRowsHaveNoHeight()
		{
			var grid = CreateGridLayout(rows: "100, auto, 100");
			var view0 = CreateTestView(new Size(100, 100));
			var view2 = CreateTestView(new Size(100, 100));

			SubstituteChildren(grid, view0, view2);
			SetLocation(grid, view0);
			SetLocation(grid, view2, row: 2);

			var manager = new GridLayoutManager(grid);
			var measure = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);
			manager.ArrangeChildren(new Rectangle(0, 0, measure.Width, measure.Height));

			// Because the auto row has no content, we expect it to have height zero
			Assert.Equal(100 + 100, measure.Height);

			// Verify the offset for the third row
			AssertArranged(view2, 0, 100, 100, 100);
		}

		[Category(GridSpacing)]
		[Fact(DisplayName = "Empty rows should not incur additional row spacing")]
		public void RowSpacingForEmptyRows()
		{
			var grid = CreateGridLayout(rows: "100, auto, 100", rowSpacing: 10);
			var view0 = CreateTestView(new Size(100, 100));
			var view2 = CreateTestView(new Size(100, 100));

			SubstituteChildren(grid, view0, view2);
			SetLocation(grid, view0);
			SetLocation(grid, view2, row: 2);

			var manager = new GridLayoutManager(grid);
			var measure = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			// Because the auto row has no content, we expect it to have height zero
			// and we expect that it won't add more row spacing 
			Assert.Equal(100 + 100 + 10, measure.Height);
		}

		[Category(GridSpacing)]
		[Fact(DisplayName = "Column spacing shouldn't affect a single-column grid")]
		public void SingleColumnIgnoresColumnSpacing()
		{
			var grid = CreateGridLayout(colSpacing: 10);
			var view = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			MeasureAndArrange(grid, double.PositiveInfinity, double.PositiveInfinity);
			AssertArranged(view, 0, 0, 100, 100);
		}

		[Category(GridSpacing)]
		[Fact(DisplayName = "Two columns should include the column spacing once")]
		public void TwoColumnsWithSpacing()
		{
			var grid = CreateGridLayout(columns: "100, 100", colSpacing: 10);
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view0, view1);
			SetLocation(grid, view0);
			SetLocation(grid, view1, col: 1);

			MeasureAndArrange(grid, double.PositiveInfinity, double.PositiveInfinity);
			AssertArranged(view0, 0, 0, 100, 100);

			// With column width 100 and spacing of 10, we expect the second column to start at 110
			AssertArranged(view1, 110, 0, 100, 100);
		}

		[Fact(DisplayName = "Measure should include column spacing")]
		public void MeasureTwoColumnsWithSpacing()
		{
			var grid = CreateGridLayout(columns: "100, 100", colSpacing: 10);
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view0, view1);
			SetLocation(grid, view0);
			SetLocation(grid, view1, col: 1);

			var manager = new GridLayoutManager(grid);
			var measure = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(100 + 100 + 10, measure.Width);
		}

		[Category(GridAutoSizing)]
		[Fact(DisplayName = "Auto columns without content have width zero")]
		public void EmptyAutoColumnsHaveNoWidth()
		{
			var grid = CreateGridLayout(columns: "100, auto, 100");
			var view0 = CreateTestView(new Size(100, 100));
			var view2 = CreateTestView(new Size(100, 100));

			SubstituteChildren(grid, view0, view2);
			SetLocation(grid, view0);
			SetLocation(grid, view2, col: 2);

			var manager = new GridLayoutManager(grid);
			var measure = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);
			manager.ArrangeChildren(new Rectangle(0, 0, measure.Width, measure.Height));

			// Because the auto column has no content, we expect it to have width zero
			Assert.Equal(100 + 100, measure.Width);

			// Verify the offset for the third column
			AssertArranged(view2, 100, 0, 100, 100);
		}

		[Category(GridSpacing)]
		[Fact(DisplayName = "Empty columns should not incur additional column spacing")]
		public void ColumnSpacingForEmptyColumns()
		{
			var grid = CreateGridLayout(columns: "100, auto, 100", colSpacing: 10);
			var view0 = CreateTestView(new Size(100, 100));
			var view2 = CreateTestView(new Size(100, 100));

			SubstituteChildren(grid, view0, view2);
			SetLocation(grid, view0);
			SetLocation(grid, view2, col: 2);

			var manager = new GridLayoutManager(grid);
			var measure = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			// Because the auto column has no content, we expect it to have height zero
			// and we expect that it won't add more row spacing 
			Assert.Equal(100 + 100 + 10, measure.Width);
		}

		[Category(GridSpan)]
		[Fact(DisplayName = "Simple row spanning")]
		public void ViewSpansRows()
		{
			var grid = CreateGridLayout(rows: "auto, auto");
			var view0 = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view0);
			SetLocation(grid, view0, rowSpan: 2);

			var measuredSize = MeasureAndArrange(grid);

			AssertArranged(view0, 0, 0, 100, 100);
			Assert.Equal(100, measuredSize.Width);

			// We expect the rows to each get half the view height
			Assert.Equal(100, measuredSize.Height);
		}

		[Category(GridSpan)]
		[Fact(DisplayName = "Simple row spanning with multiple views")]
		public void ViewSpansRowsWhenOtherViewsPresent()
		{
			var grid = CreateGridLayout(rows: "auto, auto", columns: "auto, auto");
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(50, 50));
			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0, rowSpan: 2);
			SetLocation(grid, view1, row: 1, col: 1);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(100 + 50, measuredSize.Width);
			Assert.Equal(100, measuredSize.Height);

			AssertArranged(view0, 0, 0, 100, 100);
			AssertArranged(view1, 100, 25, 50, 75);
		}

		[Category(GridSpacing, GridSpan)]
		[Fact(DisplayName = "Row spanning with row spacing")]
		public void RowSpanningShouldAccountForSpacing()
		{
			var grid = CreateGridLayout(rows: "auto, auto", columns: "auto, auto", rowSpacing: 5);
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(50, 50));
			var view2 = CreateTestView(new Size(50, 50));
			SubstituteChildren(grid, view0, view1, view2);

			SetLocation(grid, view0, rowSpan: 2);
			SetLocation(grid, view1, row: 0, col: 1);
			SetLocation(grid, view2, row: 1, col: 1);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(150, measuredSize.Width);
			Assert.Equal(50 + 50 + 5, measuredSize.Height);

			// Starts a Y = 0
			AssertArranged(view1, 100, 0, 50, 50);

			// Starts at the first row's height + the row spacing value, so Y = 50 + 5 = 55
			AssertArranged(view2, 100, 55, 50, 50);

			// We expect the height for the view spanning the rows to include the space between the rows,
			// so 50 + 5 + 50 = 105
			AssertArranged(view0, 0, 0, 100, 105);
		}

		[Category(GridSpan)]
		[Fact(DisplayName = "Simple column spanning with multiple views")]
		public void ViewSpansColumnsWhenOtherViewsPresent()
		{
			var grid = CreateGridLayout(rows: "auto, auto", columns: "auto, auto");
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(50, 50));
			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0, colSpan: 2);
			SetLocation(grid, view1, row: 1, col: 1);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(100, measuredSize.Width);
			Assert.Equal(100 + 50, measuredSize.Height);

			AssertArranged(view0, 0, 0, 100, 100);
			AssertArranged(view1, 25, 100, 75, 50);
		}

		[Category(GridSpan)]
		[Category(GridSpacing)]
		[Fact(DisplayName = "Column spanning with column spacing")]
		public void ColumnSpanningShouldAccountForSpacing()
		{
			var grid = CreateGridLayout(rows: "auto, auto", columns: "auto, auto", colSpacing: 5);
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(50, 50));
			var view2 = CreateTestView(new Size(50, 50));
			SubstituteChildren(grid, view0, view1, view2);

			SetLocation(grid, view0, colSpan: 2);
			SetLocation(grid, view1, row: 1, col: 0);
			SetLocation(grid, view2, row: 1, col: 1);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(50 + 50 + 5, measuredSize.Width);
			Assert.Equal(100 + 50, measuredSize.Height);

			// Starts a X = 0
			AssertArranged(view1, 0, 100, 50, 50);
			// Starts at the first column's width + the column spacing, so X = 50 + 5 = 55
			AssertArranged(view2, 55, 100, 50, 50);

			// We expect the width for the view spanning the columns to include the space between the columns,
			// so 50 + 5 + 50 = 105
			AssertArranged(view0, 0, 0, 105, 100);
		}

		[Category(GridSpan)]
		[Fact(DisplayName = "Row-spanning views smaller than the views confined to the row should not affect row size")]
		public void SmallerSpanningViewsShouldNotAffectRowSize()
		{
			var grid = CreateGridLayout(rows: "auto, auto", columns: "auto, auto");
			var view0 = CreateTestView(new Size(30, 30));
			var view1 = CreateTestView(new Size(50, 50));
			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0, rowSpan: 2);
			SetLocation(grid, view1, row: 0, col: 1);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(30 + 50, measuredSize.Width);
			Assert.Equal(50, measuredSize.Height);

			AssertArranged(view0, 0, 0, 30, 50);
			AssertArranged(view1, 30, 0, 50, 50);
		}

		[Category(GridSpan)]
		[Fact(DisplayName = "Column-spanning views smaller than the views confined to the column should not affect column size")]
		public void SmallerSpanningViewsShouldNotAffectColumnSize()
		{
			var grid = CreateGridLayout(rows: "auto, auto", columns: "auto, auto");
			var view0 = CreateTestView(new Size(30, 30));
			var view1 = CreateTestView(new Size(50, 50));
			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0, colSpan: 2);
			SetLocation(grid, view1, row: 1, col: 0);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(50, measuredSize.Width);
			Assert.Equal(30 + 50, measuredSize.Height);

			AssertArranged(view0, 0, 0, 50, 30);
			AssertArranged(view1, 0, 30, 50, 50);
		}

		[Category(GridAbsoluteSizing)]
		[Fact(DisplayName = "Empty absolute rows/columns still affect Grid size")]
		public void EmptyAbsoluteRowsAndColumnsAffectSize()
		{
			var grid = CreateGridLayout(rows: "10, 40", columns: "15, 85");
			var view0 = CreateTestView(new Size(30, 30));
			SubstituteChildren(grid, view0);

			SetLocation(grid, view0, row: 1, col: 1);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(15 + 85, measuredSize.Width);
			Assert.Equal(10 + 40, measuredSize.Height);

			AssertArranged(view0, 15, 10, 85, 40);
		}

		[Category(GridSpan)]
		[Fact(DisplayName = "Row and column spans should be able to mix")]
		public void MixedRowAndColumnSpans()
		{
			var grid = CreateGridLayout(rows: "auto, auto", columns: "auto, auto");
			var view0 = CreateTestView(new Size(60, 30));
			var view1 = CreateTestView(new Size(30, 60));
			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0, row: 0, col: 0, colSpan: 2);
			SetLocation(grid, view1, row: 0, col: 1, rowSpan: 2);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(60, measuredSize.Width);
			Assert.Equal(60, measuredSize.Height);

			AssertArranged(view0, 0, 0, 60, 45);
			AssertArranged(view1, 15, 0, 45, 60);
		}

		[Category(GridSpan)]
		[Fact(DisplayName = "Row span including absolute row should not modify absolute size")]
		public void RowSpanShouldNotModifyAbsoluteRowSize()
		{
			var grid = CreateGridLayout(rows: "auto, 20", columns: "auto, auto");
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(50, 10));
			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0, rowSpan: 2);
			SetLocation(grid, view1, row: 1, col: 1);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(100 + 50, measuredSize.Width);
			Assert.Equal(100, measuredSize.Height);

			AssertArranged(view0, 0, 0, 100, 100);

			// The item in the second row starts at y = 80 because the auto row above had to distribute
			// all the extra space into row 0; row 1 is absolute, so no tinkering with it to make stuff fit
			AssertArranged(view1, 100, 80, 50, 20);
		}

		[Category(GridSpan)]
		[Fact(DisplayName = "Column span including absolute column should not modify absolute size")]
		public void ColumnSpanShouldNotModifyAbsoluteColumnSize()
		{
			var grid = CreateGridLayout(rows: "auto, auto", columns: "auto, 20");
			var view0 = CreateTestView(new Size(100, 100));
			var view1 = CreateTestView(new Size(50, 10));
			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0, colSpan: 2);
			SetLocation(grid, view1, row: 1, col: 1);

			var measuredSize = MeasureAndArrange(grid);

			Assert.Equal(100, measuredSize.Width);
			Assert.Equal(100 + 10, measuredSize.Height);

			AssertArranged(view0, 0, 0, 100, 100);

			// The item in the second row starts at x = 80 because the auto column before it had to distribute
			// all the extra space into column 0; column 1 is absolute, so no tinkering with it to make stuff fit
			AssertArranged(view1, 80, 100, 20, 10);
		}

		[Category(GridSpan)]
		[Fact]
		public void CanSpanAbsoluteColumns()
		{
			var grid = CreateGridLayout(rows: "auto", columns: "100,100");
			var view0 = CreateTestView(new Size(150, 100));
			SubstituteChildren(grid, view0);
			SetLocation(grid, view0, colSpan: 2);
			var manager = new GridLayoutManager(grid);

			manager.Measure(200, 100);
			manager.ArrangeChildren(new Rectangle(0, 0, 200, 100));

			// View should be arranged to span both columns (200 points)
			AssertArranged(view0, 0, 0, 200, 100);
		}

		[Category(GridSpan)]
		[Fact]
		public void CanSpanAbsoluteRows()
		{
			var grid = CreateGridLayout(rows: "100,100", columns: "auto");
			var view0 = CreateTestView(new Size(100, 150));

			SubstituteChildren(grid, view0);
			SetLocation(grid, view0, rowSpan: 2);
			var manager = new GridLayoutManager(grid);

			manager.Measure(100, 200);
			manager.ArrangeChildren(new Rectangle(0, 0, 100, 200));

			// View should be arranged to span both rows (200 points)
			AssertArranged(view0, 0, 0, 100, 200);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Single star column consumes all horizontal space")]
		public void SingleStarColumn()
		{
			var screenWidth = 400;
			var screenHeight = 600;

			var grid = CreateGridLayout(rows: "auto", columns: $"*");
			var view0 = CreateTestView(new Size(100, 100));

			SubstituteChildren(grid, view0);
			SetLocation(grid, view0);

			MeasureAndArrange(grid, screenWidth, screenHeight);

			// Row height is auto, so it gets the height of the view
			// Column is *, so it should get the whole width
			AssertArranged(view0, 0, 0, screenWidth, 100);
		}

		[Category(GridStarSizing)]
		[Fact]
		public void SingleWeightedStarColumn()
		{
			var screenWidth = 400;
			var screenHeight = 600;

			var grid = CreateGridLayout(rows: "auto", columns: $"3*");
			var view0 = CreateTestView(new Size(100, 100));

			SubstituteChildren(grid, view0);
			SetLocation(grid, view0);

			MeasureAndArrange(grid, screenWidth, screenHeight);

			// Row height is auto, so it gets the height of the view
			// The column is 3*, but it's the only column, so it should get the full width
			AssertArranged(view0, 0, 0, screenWidth, 100);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Multiple star columns consume equal space")]
		public void MultipleStarColumns()
		{
			var screenWidth = 300;
			var screenHeight = 600;
			var viewSize = new Size(50, 50);

			var grid = CreateGridLayout(rows: "auto", columns: $"*,*,*");
			var view0 = CreateTestView(viewSize);
			var view1 = CreateTestView(viewSize);
			var view2 = CreateTestView(viewSize);

			SubstituteChildren(grid, view0, view1, view2);

			SetLocation(grid, view0);
			SetLocation(grid, view1, col: 1);
			SetLocation(grid, view2, col: 2);

			MeasureAndArrange(grid, screenWidth, screenHeight);

			// Row height is auto, so it gets the height of the view
			// Columns are *,*,*, so each view should be arranged at 1/3 the width
			var expectedWidth = screenWidth / 3;
			var expectedHeight = viewSize.Height;

			// Make sure that the views in the columns are actually getting measured at the column width,
			// and not just at the width of the whole grid
			view1.Received().Measure(Arg.Is<double>(expectedWidth), Arg.Any<double>());
			view2.Received().Measure(Arg.Is<double>(expectedWidth), Arg.Any<double>());

			AssertArranged(view0, 0, 0, expectedWidth, expectedHeight);
			AssertArranged(view1, expectedWidth, 0, expectedWidth, expectedHeight);
			AssertArranged(view2, expectedWidth * 2, 0, expectedWidth, expectedHeight);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Weighted star column gets proportional space")]
		public void WeightedStarColumn()
		{
			var screenWidth = 300;
			var screenHeight = 600;
			var viewSize = new Size(50, 50);

			var grid = CreateGridLayout(rows: "auto", columns: $"*,2*");
			var view0 = CreateTestView(viewSize);
			var view1 = CreateTestView(viewSize);

			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0);
			SetLocation(grid, view1, col: 1);

			MeasureAndArrange(grid, screenWidth, screenHeight);

			// Row height is auto, so it gets the height of the view
			// First column should get 1/3 of the width, second should get 2/3
			var expectedWidth0 = screenWidth / 3;
			var expectedWidth1 = expectedWidth0 * 2;
			var expectedHeight = viewSize.Height;
			AssertArranged(view0, 0, 0, expectedWidth0, expectedHeight);
			AssertArranged(view1, expectedWidth0, 0, expectedWidth1, expectedHeight);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Totally empty star columns measured at infinite width have zero width")]
		public void EmptyStarColumnInfiniteWidthMeasure()
		{
			var grid = CreateGridLayout(rows: "auto", columns: $"*");
			var manager = new GridLayoutManager(grid);
			var measuredSize = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(0, measuredSize.Width);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Single star column with a view measured at infinite width gets width of the view")]
		public void StarColumnWithViewInfiniteWidthMeasure()
		{
			var grid = CreateGridLayout(rows: "auto", columns: $"*");
			var view0 = CreateTestView(new Size(100, 50));

			SubstituteChildren(grid, view0);
			SetLocation(grid, view0);

			var manager = new GridLayoutManager(grid);
			var measuredSize = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(100, measuredSize.Width);
			Assert.Equal(50, measuredSize.Height);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Multiple star columns with views measured at infinite width get the width of the widest view")]
		public void MultipleStarColumnsWithViewsInfiniteWidthMeasure()
		{
			var grid = CreateGridLayout(rows: "auto", columns: $"*,*,*");
			var view0 = CreateTestView(new Size(50, 50));
			var view1 = CreateTestView(new Size(75, 50));
			var view2 = CreateTestView(new Size(50, 50));

			SubstituteChildren(grid, view0, view1, view2);

			SetLocation(grid, view0);
			SetLocation(grid, view1, col: 1);
			SetLocation(grid, view2, col: 2);

			MeasureAndArrange(grid, double.PositiveInfinity, double.PositiveInfinity);

			// Row height is auto, so it gets the height of the tallest view
			// The widest view has width 75, so we expect all three * columns to have 75 width
			var expectedWidth = 75;
			AssertArranged(view0, 0, 0, expectedWidth, 50);
			AssertArranged(view1, expectedWidth, 0, expectedWidth, 50);
			AssertArranged(view2, expectedWidth * 2, 0, expectedWidth, 50);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Single star row consumes all vertical space")]
		public void SingleStarRow()
		{
			var screenWidth = 400;
			var screenHeight = 600;

			var grid = CreateGridLayout(rows: "*", columns: "auto");
			var view0 = CreateTestView(new Size(100, 100));

			SubstituteChildren(grid, view0);
			SetLocation(grid, view0);

			MeasureAndArrange(grid, screenWidth, screenHeight);

			// Column width is auto, so it gets the width of the view
			// Row is *, so it should get the whole height
			AssertArranged(view0, 0, 0, 100, screenHeight);
		}

		[Category(GridStarSizing)]
		[Fact]
		public void SingleWeightedStarRow()
		{
			var screenWidth = 400;
			var screenHeight = 600;

			var grid = CreateGridLayout(rows: "3*", columns: "auto");
			var view0 = CreateTestView(new Size(100, 100));

			SubstituteChildren(grid, view0);
			SetLocation(grid, view0);

			MeasureAndArrange(grid, screenWidth, screenHeight);

			// Column width is auto, so it gets the width of the view
			// The row is 3*, but it's the only row, so it should get the full height
			AssertArranged(view0, 0, 0, 100, screenHeight);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Multiple star rows consume equal space")]
		public void MultipleStarRows()
		{
			var screenWidth = 300;
			var screenHeight = 600;
			var viewSize = new Size(50, 50);

			var grid = CreateGridLayout(rows: "*,*,*", columns: "auto");
			var view0 = CreateTestView(viewSize);
			var view1 = CreateTestView(viewSize);
			var view2 = CreateTestView(viewSize);

			SubstituteChildren(grid, view0, view1, view2);

			SetLocation(grid, view0);
			SetLocation(grid, view1, row: 1);
			SetLocation(grid, view2, row: 2);

			MeasureAndArrange(grid, screenWidth, screenHeight);

			// Column width is auto, so it gets the width of the view
			// Rows are *,*,*, so each view should be arranged at 1/3 the height
			var expectedHeight = screenHeight / 3;
			var expectedWidth = viewSize.Width;
			AssertArranged(view0, 0, 0, expectedWidth, expectedHeight);
			AssertArranged(view1, 0, expectedHeight, expectedWidth, expectedHeight);
			AssertArranged(view2, 0, expectedHeight * 2, expectedWidth, expectedHeight);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Weighted star row gets proportional space")]
		public void WeightedStarRow()
		{
			var screenWidth = 300;
			var screenHeight = 600;
			var viewSize = new Size(50, 50);

			var grid = CreateGridLayout(rows: "*,2*", columns: "auto");
			var view0 = CreateTestView(viewSize);
			var view1 = CreateTestView(viewSize);

			SubstituteChildren(grid, view0, view1);

			SetLocation(grid, view0);
			SetLocation(grid, view1, row: 1);

			MeasureAndArrange(grid, screenWidth, screenHeight);

			// Column width is auto, so it gets the width of the view
			// First row should get 1/3 of the height, second should get 2/3
			var expectedHeight0 = screenHeight / 3;
			var expectedHeight1 = expectedHeight0 * 2;
			var expectedWidth = viewSize.Width;
			AssertArranged(view0, 0, 0, expectedWidth, expectedHeight0);
			AssertArranged(view1, 0, expectedHeight0, expectedWidth, expectedHeight1);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Totally empty star rows measured at infinite height have zero height")]
		public void EmptyStarRowInfiniteHeightMeasure()
		{
			var grid = CreateGridLayout(rows: "*", columns: $"auto");
			var manager = new GridLayoutManager(grid);
			var measuredSize = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(0, measuredSize.Height);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Single star row with a view measured at infinite height gets height of the view")]
		public void StarRowWithViewInfiniteHeightMeasure()
		{
			var grid = CreateGridLayout(rows: "*", columns: $"auto");
			var view0 = CreateTestView(new Size(100, 50));

			SubstituteChildren(grid, view0);
			SetLocation(grid, view0);

			var manager = new GridLayoutManager(grid);
			var measuredSize = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(100, measuredSize.Width);
			Assert.Equal(50, measuredSize.Height);
		}

		[Category(GridStarSizing)]
		[Fact(DisplayName = "Multiple star rows with views measured at infinite height get the height of the tallest view")]
		public void MultipleStarRowsWithViewsInfiniteHeightMeasure()
		{
			var grid = CreateGridLayout(rows: "*,*,*", columns: "auto");
			var view0 = CreateTestView(new Size(50, 50));
			var view1 = CreateTestView(new Size(50, 75));
			var view2 = CreateTestView(new Size(50, 50));

			SubstituteChildren(grid, view0, view1, view2);

			SetLocation(grid, view0);
			SetLocation(grid, view1, row: 1);
			SetLocation(grid, view2, row: 2);

			MeasureAndArrange(grid, double.PositiveInfinity, double.PositiveInfinity);

			// Column width is auto, so it gets the width of the widest view
			// The tallest view has height 75, so we expect all three * rows to have 75 height
			var expectedHeight = 75;
			AssertArranged(view0, 0, 0, 50, expectedHeight);
			AssertArranged(view1, 0, expectedHeight, 50, expectedHeight);
			AssertArranged(view2, 0, expectedHeight * 2, 50, expectedHeight);
		}

		[Category(GridAbsoluteSizing)]
		[Category(GridStarSizing)]
		[Fact]
		public void MixStarsAndExplicitSizes()
		{
			var screenWidth = 300;
			var screenHeight = 600;
			var viewSize = new Size(50, 50);

			var grid = CreateGridLayout(rows: "auto", columns: $"3*,100,*");
			var view0 = CreateTestView(viewSize);
			var view1 = CreateTestView(viewSize);
			var view2 = CreateTestView(viewSize);

			SubstituteChildren(grid, view0, view1, view2);

			SetLocation(grid, view0);
			SetLocation(grid, view1, col: 1);
			SetLocation(grid, view2, col: 2);

			MeasureAndArrange(grid, screenWidth, screenHeight);

			// Row height is auto, so it gets the height of the view
			// Columns are 3*,100,* 
			// So we expect the center column to be 100, leaving 500 for the stars
			// 3/4 of that goes to the first column, so 375; the remaining 125 is the last column
			var expectedStarWidth = (screenWidth - 100) / 4;
			var expectedHeight = viewSize.Height;

			AssertArranged(view0, 0, 0, expectedStarWidth * 3, expectedHeight);
			AssertArranged(view1, expectedStarWidth * 3, 0, 100, expectedHeight);
			AssertArranged(view2, (expectedStarWidth * 3) + 100, 0, expectedStarWidth, expectedHeight);
		}

		[Fact]
		public void UsesImpliedRowAndColumnIfNothingDefined()
		{
			var grid = CreateGridLayout();
			var view0 = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view0);
			SetLocation(grid, view0);

			// Using 300,300 - the implied row/column are GridLength.Star
			MeasureAndArrange(grid, 300, 300);

			// Since it's using GridLength.Star, we expect the view to be arranged at the full size of the grid
			AssertArranged(view0, 0, 0, 300, 300);
		}

		[Fact]
		public void IgnoresCollapsedViews()
		{
			var view = LayoutTestHelpers.CreateTestView(new Size(100, 100));
			var collapsedView = LayoutTestHelpers.CreateTestView(new Size(100, 100));
			collapsedView.Visibility.Returns(Visibility.Collapsed);

			var grid = CreateGridLayout(children: new List<IView>() { view, collapsedView });

			var manager = new GridLayoutManager(grid);
			var measure = manager.Measure(100, double.PositiveInfinity);
			manager.ArrangeChildren(new Rectangle(Point.Zero, measure));

			// View is visible, so we expect it to be measured and arranged
			view.Received().Measure(Arg.Any<double>(), Arg.Any<double>());
			view.Received().Arrange(Arg.Any<Rectangle>());

			// View is collapsed, so we expect it not to be measured or arranged
			collapsedView.DidNotReceive().Measure(Arg.Any<double>(), Arg.Any<double>());
			collapsedView.DidNotReceive().Arrange(Arg.Any<Rectangle>());
		}

		[Fact]
		public void DoesNotIgnoreHiddenViews()
		{
			var view = LayoutTestHelpers.CreateTestView(new Size(100, 100));
			var hiddenView = LayoutTestHelpers.CreateTestView(new Size(100, 100));
			hiddenView.Visibility.Returns(Visibility.Hidden);

			var grid = CreateGridLayout(children: new List<IView>() { view, hiddenView });

			var manager = new GridLayoutManager(grid);
			var measure = manager.Measure(100, double.PositiveInfinity);
			manager.ArrangeChildren(new Rectangle(Point.Zero, measure));

			// View is visible, so we expect it to be measured and arranged
			view.Received().Measure(Arg.Any<double>(), Arg.Any<double>());
			view.Received().Arrange(Arg.Any<Rectangle>());

			// View is hidden, so we expect it to be measured and arranged (since it'll need to take up space)
			hiddenView.Received().Measure(Arg.Any<double>(), Arg.Any<double>());
			hiddenView.Received().Arrange(Arg.Any<Rectangle>());
		}

		IGridLayout BuildPaddedGrid(Thickness padding, double viewWidth, double viewHeight)
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(viewWidth, viewHeight));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.Padding.Returns(padding);

			return grid;
		}

		[Theory]
		[InlineData(0, 0, 0, 0)]
		[InlineData(10, 10, 10, 10)]
		[InlineData(10, 0, 10, 0)]
		[InlineData(0, 10, 0, 10)]
		[InlineData(23, 5, 3, 15)]
		public void MeasureAccountsForPadding(double left, double top, double right, double bottom)
		{
			var viewWidth = 100d;
			var viewHeight = 100d;
			var padding = new Thickness(left, top, right, bottom);

			var expectedHeight = padding.VerticalThickness + viewHeight;
			var expectedWidth = padding.HorizontalThickness + viewWidth;

			var grid = BuildPaddedGrid(padding, viewWidth, viewHeight);

			var manager = new GridLayoutManager(grid);
			var measuredSize = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(expectedHeight, measuredSize.Height);
			Assert.Equal(expectedWidth, measuredSize.Width);
		}

		[Theory]
		[InlineData(0, 0, 0, 0)]
		[InlineData(10, 10, 10, 10)]
		[InlineData(10, 0, 10, 0)]
		[InlineData(0, 10, 0, 10)]
		[InlineData(23, 5, 3, 15)]
		public void ArrangeAccountsForPadding(double left, double top, double right, double bottom)
		{
			var viewWidth = 100d;
			var viewHeight = 100d;
			var padding = new Thickness(left, top, right, bottom);

			var grid = BuildPaddedGrid(padding, viewWidth, viewHeight);

			var manager = new GridLayoutManager(grid);
			var measuredSize = manager.Measure(double.PositiveInfinity, double.PositiveInfinity);
			manager.ArrangeChildren(new Rectangle(Point.Zero, measuredSize));

			AssertArranged(grid[0], padding.Left, padding.Top, viewWidth, viewHeight);
		}

		[Category(GridStarSizing)]
		[Fact]
		public void StarValuesAreMeasuredTwiceWhenConstraintsAreInfinite()
		{
			// A one-row, one-column grid
			var grid = CreateGridLayout();

			// A 100x100 IView
			var view = CreateTestView(new Size(100, 100));

			// Set up the grid to have a single child
			SubstituteChildren(grid, view);

			// Set up the row/column values and spans
			SetLocation(grid, view);

			MeasureAndArrange(grid, double.PositiveInfinity, double.PositiveInfinity);

			// View must be measured to figure out the Auto value
			view.Received().Measure(Arg.Is(double.PositiveInfinity), Arg.Is(double.PositiveInfinity));

			// And again at the final size
			view.Received().Measure(Arg.Is<double>(100), Arg.Is<double>(100));
		}

		[Category(GridAbsoluteSizing)]
		[Fact]
		public void GridMeasureShouldUseExplicitHeight()
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(10, 10));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.Height.Returns(50);

			var gridLayoutManager = new GridLayoutManager(grid);
			var measure = gridLayoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(50, measure.Height);
		}

		[Category(GridAbsoluteSizing)]
		[Fact]
		public void GridMeasureShouldUseExplicitWidth()
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(10, 10));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.Width.Returns(50);

			var gridLayoutManager = new GridLayoutManager(grid);
			var measure = gridLayoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(50, measure.Width);
		}

		[Theory]
		// at 0, 0
		[InlineData(1, 1, 0, 0, 0, 0)]
		[InlineData(1, 2, 0, 0, 0, 0)]
		[InlineData(2, 1, 0, 0, 0, 0)]
		[InlineData(2, 2, 0, 0, 0, 0)]
		// at 1, 0
		[InlineData(1, 1, 1, 0, 0, 0)]
		[InlineData(1, 2, 1, 0, 0, 0)]
		[InlineData(2, 1, 1, 0, 1, 0)]
		[InlineData(2, 2, 1, 0, 1, 0)]
		// at 0, 1
		[InlineData(1, 1, 0, 1, 0, 0)]
		[InlineData(1, 2, 0, 1, 0, 1)]
		[InlineData(2, 1, 0, 1, 0, 0)]
		[InlineData(2, 2, 0, 1, 0, 1)]
		// at 1, 1
		[InlineData(1, 1, 1, 1, 0, 0)]
		[InlineData(1, 2, 1, 1, 0, 1)]
		[InlineData(2, 1, 1, 1, 1, 0)]
		[InlineData(2, 2, 1, 1, 1, 1)]
		public void ViewOutsideRowsAndColsClampsToGrid(int rows, int cols, int row, int col, int actualRow, int actualCol)
		{
			var r = string.Join(",", Enumerable.Repeat("100", rows));
			var c = string.Join(",", Enumerable.Repeat("100", cols));

			var grid = CreateGridLayout(rows: r, columns: c);
			var view0 = CreateTestView(new Size(10, 10));
			SubstituteChildren(grid, view0);
			SetLocation(grid, view0, row, col);

			MeasureAndArrange(grid, 100 * cols, 100 * rows);

			AssertArranged(view0, 100 * actualCol, 100 * actualRow, 100, 100);
		}

		[Theory]
		// normal
		[InlineData(0, 0, 1, 1, 0, 0, 1, 1)]
		[InlineData(1, 1, 1, 1, 1, 1, 1, 1)]
		[InlineData(1, 1, 2, 1, 1, 1, 2, 1)]
		// negative origin
		[InlineData(-1, 0, 1, 1, 0, 0, 1, 1)]
		[InlineData(0, -1, 1, 1, 0, 0, 1, 1)]
		[InlineData(-1, -1, 1, 1, 0, 0, 1, 1)]
		// negative span
		[InlineData(1, 1, -1, 0, 1, 1, 1, 1)]
		[InlineData(1, 1, 0, -1, 1, 1, 1, 1)]
		[InlineData(1, 1, -1, -1, 1, 1, 1, 1)]
		// positive origin
		[InlineData(5, 0, 1, 1, 3, 0, 1, 1)]
		[InlineData(0, 5, 1, 1, 0, 3, 1, 1)]
		[InlineData(5, 5, 1, 1, 3, 3, 1, 1)]
		// positive span
		[InlineData(0, 0, 1, 5, 0, 0, 1, 4)]
		[InlineData(0, 0, 5, 1, 0, 0, 4, 1)]
		[InlineData(0, 0, 5, 5, 0, 0, 4, 4)]
		// normal origin + positive span
		[InlineData(1, 1, 1, 5, 1, 1, 1, 3)]
		[InlineData(1, 1, 5, 1, 1, 1, 3, 1)]
		[InlineData(1, 1, 5, 5, 1, 1, 3, 3)]
		// positive origin + positive span
		[InlineData(5, 5, 1, 5, 3, 3, 1, 1)]
		[InlineData(5, 5, 5, 1, 3, 3, 1, 1)]
		[InlineData(5, 5, 5, 5, 3, 3, 1, 1)]
		public void SpansOutsideRowsAndColsClampsToGrid(int row, int col, int rowSpan, int colSpan, int actualRow, int actualCol, int actualRowSpan, int actualColSpan)
		{
			const int GridSize = 4;
			var r = string.Join(",", Enumerable.Repeat("100", GridSize));
			var c = string.Join(",", Enumerable.Repeat("100", GridSize));

			var grid = CreateGridLayout(rows: r, columns: c);
			var view0 = CreateTestView(new Size(10, 10));
			SubstituteChildren(grid, view0);
			SetLocation(grid, view0, row, col, rowSpan, colSpan);

			MeasureAndArrange(grid, 100 * GridSize, 100 * GridSize);

			AssertArranged(
				view0,
				100 * actualCol,
				100 * actualRow,
				100 * actualColSpan,
				100 * actualRowSpan);
		}

		[Fact]
		public void ArrangeRespectsBounds()
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(100, 100));

			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			var measure = MeasureAndArrange(grid, double.PositiveInfinity, double.PositiveInfinity, 10, 15);

			var expectedRectangle = new Rectangle(10, 15, measure.Width, measure.Height);

			view.Received().Arrange(Arg.Is(expectedRectangle));
		}

		[Category(GridAbsoluteSizing)]
		[Theory]
		[InlineData(50, 100, 50)]
		[InlineData(100, 100, 100)]
		[InlineData(100, 50, 50)]
		[InlineData(0, 50, 0)]
		[InlineData(-1, 50, 50)]
		public void MeasureRespectsMaxHeight(double maxHeight, double viewHeight, double expectedHeight)
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(100, viewHeight));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.MaximumHeight.Returns(maxHeight);

			var layoutManager = new GridLayoutManager(grid);
			var measure = layoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(expectedHeight, measure.Height);
		}

		[Category(GridAbsoluteSizing)]
		[Theory]
		[InlineData(50, 100, 50)]
		[InlineData(100, 100, 100)]
		[InlineData(100, 50, 50)]
		[InlineData(0, 50, 0)]
		[InlineData(-1, 50, 50)]
		public void MeasureRespectsMaxWidth(double maxWidth, double viewWidth, double expectedWidth)
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(viewWidth, 100));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.MaximumWidth.Returns(maxWidth);

			var layoutManager = new GridLayoutManager(grid);
			var measure = layoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(expectedWidth, measure.Width);
		}

		[Category(GridAbsoluteSizing)]
		[Theory]
		[InlineData(50, 10, 50)]
		[InlineData(100, 100, 100)]
		[InlineData(10, 50, 50)]
		[InlineData(-1, 50, 50)]
		public void MeasureRespectsMinHeight(double minHeight, double viewHeight, double expectedHeight)
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(100, viewHeight));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.MinimumHeight.Returns(minHeight);

			var layoutManager = new GridLayoutManager(grid);
			var measure = layoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(expectedHeight, measure.Height);
		}

		[Category(GridAbsoluteSizing)]
		[Theory]
		[InlineData(50, 10, 50)]
		[InlineData(100, 100, 100)]
		[InlineData(10, 50, 50)]
		[InlineData(-1, 50, 50)]
		public void MeasureRespectsMinWidth(double minWidth, double viewWidth, double expectedWidth)
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(viewWidth, 100));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.MinimumWidth.Returns(minWidth);

			var layoutManager = new GridLayoutManager(grid);
			var measure = layoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			Assert.Equal(expectedWidth, measure.Width);
		}

		[Fact]
		[Category(GridAbsoluteSizing)]
		public void MaxWidthDominatesWidth()
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.Width.Returns(75);
			grid.MaximumWidth.Returns(50);

			var layoutManager = new GridLayoutManager(grid);
			var measure = layoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			// The maximum value beats out the explicit value
			Assert.Equal(50, measure.Width);
		}

		[Fact]
		[Category(GridAbsoluteSizing)]
		public void MinWidthDominatesMaxWidth()
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.MinimumWidth.Returns(75);
			grid.MaximumWidth.Returns(50);

			var layoutManager = new GridLayoutManager(grid);
			var measure = layoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			// The minimum value should beat out the maximum value
			Assert.Equal(75, measure.Width);
		}

		[Fact]
		[Category(GridAbsoluteSizing)]
		public void MaxHeightDominatesHeight()
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.Height.Returns(75);
			grid.MaximumHeight.Returns(50);

			var layoutManager = new GridLayoutManager(grid);
			var measure = layoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			// The maximum value beats out the explicit value
			Assert.Equal(50, measure.Height);
		}

		[Fact]
		[Category(GridAbsoluteSizing)]
		public void MinHeightDominatesMaxHeight()
		{
			var grid = CreateGridLayout();
			var view = CreateTestView(new Size(100, 100));
			SubstituteChildren(grid, view);
			SetLocation(grid, view);

			grid.MinimumHeight.Returns(75);
			grid.MaximumHeight.Returns(50);

			var layoutManager = new GridLayoutManager(grid);
			var measure = layoutManager.Measure(double.PositiveInfinity, double.PositiveInfinity);

			// The minimum value should beat out the maximum value
			Assert.Equal(75, measure.Height);
		}
	}
}
