using System;
using System.Collections.Immutable;
using System.Text;

namespace Orleans.Runtime
{
    [GenerateSerializer, Immutable]
    internal sealed class MembershipTableSnapshot
    {
        public MembershipTableSnapshot(
            MembershipVersion version,
            ImmutableDictionary<SiloAddress, MembershipEntry> entries)
        {
            this.Version = version;
            this.Entries = entries;
        }

        public static MembershipTableSnapshot Create(MembershipTableData table)
        {
            ArgumentNullException.ThrowIfNull(table);

            var entries = ImmutableDictionary.CreateBuilder<SiloAddress, MembershipEntry>();
            if (table.Members != null)
            {
                foreach (var item in table.Members)
                {
                    var entry = item.Item1;
                    entries.Add(entry.SiloAddress, entry);
                }
            }

            var version = (table.Version.Version == 0 && table.Version.VersionEtag == "0")
              ? MembershipVersion.MinValue
              : new MembershipVersion(table.Version.Version);
            return new MembershipTableSnapshot(version, entries.ToImmutable());
        }

        public static MembershipTableSnapshot Create(MembershipTableSnapshot snapshot)
        {
            ArgumentNullException.ThrowIfNull(snapshot);

            var entries = ImmutableDictionary.CreateBuilder<SiloAddress, MembershipEntry>();
            if (snapshot.Entries != null)
            {
                foreach (var item in snapshot.Entries)
                {
                    var entry = item.Value;
                    entries.Add(entry.SiloAddress, entry);
                }
            }

            return new MembershipTableSnapshot(snapshot.Version, entries.ToImmutable());
        }

        [Id(0)]
        public MembershipVersion Version { get; }
        
        [Id(1)]
        public ImmutableDictionary<SiloAddress, MembershipEntry> Entries { get; }

        public int ActiveNodeCount
        {
            get
            {
                var count = 0;
                foreach (var entry in this.Entries)
                {
                    if (entry.Value.Status == SiloStatus.Active)
                    {
                        ++count;
                    }
                }

                return count;
            }
        }

        public SiloStatus GetSiloStatus(SiloAddress silo)
        {
            var status = this.Entries.TryGetValue(silo, out var entry) ? entry.Status : SiloStatus.None;
            if (status == SiloStatus.None)
            {
                foreach (var member in this.Entries)
                {
                    if (member.Key.IsSuccessorOf(silo))
                    {
                        status = SiloStatus.Dead;
                        break;
                    }
                }
            }

            return status;
        }

        public override string ToString()
        {
            var sb = new StringBuilder();
            sb.Append($"[Version: {this.Version}, {this.Entries.Count} silos");
            foreach (var entry in this.Entries) sb.Append($", {entry.Value}");
            sb.Append(']');
            return sb.ToString();
        }
    }
}
