#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
import os
from tempfile import NamedTemporaryFile

import pytest

from tests.providers.google.cloud.utils.gcp_authenticator import GCP_AI_KEY, GCP_GCS_KEY
from tests.test_utils.gcp_system_helpers import CLOUD_DAG_FOLDER, GoogleSystemTest, provide_gcp_context

GCP_BUCKET_NAME = os.environ.get('GCP_VISION_BUCKET_NAME', 'vision-bucket-system-test')
GCP_REFERENCE_IMAGE_URL = os.environ.get('GCP_VISION_REFERENCE_IMAGE_URL', 'gs://bucket-name/image.png')
GCP_ANNOTATE_IMAGE_URL = os.environ.get('GCP_VISION_ANNOTATE_IMAGE_URL', 'gs://bucket-name/image.png')
GCP_VIDEO_SOURCE_URL = os.environ.get('GCP_VISION_SOURCE_IMAGE_URL', "http://google.com/image.jpg")


@pytest.mark.system("google.cloud")
@pytest.mark.credential_file(GCP_AI_KEY)
class CloudVisionExampleDagsSystemTest(GoogleSystemTest):
    @provide_gcp_context(GCP_AI_KEY)
    def setUp(self):
        super().setUp()
        self.create_gcs_bucket(GCP_BUCKET_NAME, location="europe-north1")
        with NamedTemporaryFile(suffix=".png") as file:
            self.execute_cmd(["curl", "-s", GCP_VIDEO_SOURCE_URL, "-o", file.name])
            self.execute_with_ctx(['gsutil', 'cp', file.name, GCP_REFERENCE_IMAGE_URL], key=GCP_GCS_KEY)
            self.execute_with_ctx(['gsutil', 'cp', file.name, GCP_ANNOTATE_IMAGE_URL], key=GCP_GCS_KEY)

    @provide_gcp_context(GCP_AI_KEY)
    def tearDown(self):
        self.delete_gcs_bucket(GCP_BUCKET_NAME)
        super().tearDown()

    @provide_gcp_context(GCP_AI_KEY)
    def test_run_example_gcp_vision_autogenerated_id_dag(self):
        self.run_dag('example_gcp_vision_autogenerated_id', CLOUD_DAG_FOLDER)

    @provide_gcp_context(GCP_AI_KEY)
    def test_run_example_gcp_vision_explicit_id_dag(self):
        self.run_dag('example_gcp_vision_explicit_id', CLOUD_DAG_FOLDER)

    @provide_gcp_context(GCP_AI_KEY)
    def test_run_example_gcp_vision_annotate_image_dag(self):
        self.run_dag('example_gcp_vision_annotate_image', CLOUD_DAG_FOLDER)
