/*
 * Copyright 2018 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.shamrock.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Indicates that this {@link BuildStep} method will also output recorded bytecode.
 *
 * If this annotation is present at least one method parameter must be a template object
 * (i.e. a runtime object annotated with {@code @Template}). Any invocations made against
 * this object will be recorded, and written out to bytecode be be invoked at runtime.
 *
 * The {@link #value()} element determines when the generated bytcode is executed. If this
 * is {@link ExecutionTime#STATIC_INIT} then it will be executed from a static init method,
 * so will run at native image generation time.
 *
 * If this is {@link ExecutionTime#RUNTIME_INIT} then it will run from a main method at application
 * start.
 */
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.SOURCE)
public @interface Record {

    /**
     * The time to execute the recorded bytecode
     */
    ExecutionTime value();

    /**
     * If this is true then the bytecode produced by this method will be considered to be optional,
     * and will only be created if this build step also produces another {@link org.jboss.builder.item.BuildItem}
     * that is consumed by the build.
     *
     * If a method is optional it must be capable of producing at least one other item
     *
     */
    boolean optional() default false;

}
