use crate::GitverHashes;
use quote::quote;

pub fn write_mod(gitver_hashes: GitverHashes) {
    let githashes: Vec<_> = gitver_hashes
        .map
        .into_iter()
        .map(|(k, gh)| {
            let key = k.display().to_string();
            let pathspec = gh.pathspec.display().to_string();
            let commit = gh.commit;
            let tree = gh.tree;
            let sha256 = gh.sha256;

            quote! {
                (PathBuf::from_str(#key).unwrap(), GitHash {
                    pathspec: PathBuf::from_str(#pathspec).unwrap(),
                    commit: #commit.to_string(),
                    tree: #tree.to_string(),
                    sha256: #sha256.to_string(),
                })
            }
        })
        .collect();

    let head_commit = &gitver_hashes.head.commit;
    let head_tree = &gitver_hashes.head.tree;

    let code = quote! {
        use gitver::{GitverMap, GitHead, GitverHashes};

        #[cfg(not(debug_assertions))]
        use gitver::GitHash;

        #[cfg(not(debug_assertions))]
        use std::{path::PathBuf, str::FromStr};

        use std::sync::OnceLock;
        static GITVER_HASHES: OnceLock<GitverHashes> = OnceLock::new();

        pub fn get() -> &'static GitverHashes {
            GITVER_HASHES.get_or_init(|| {
                GitverHashes {
                    map: GitverMap::from([
                        #(#githashes),*
                    ]),
                    head: GitHead {
                        commit: #head_commit.to_string(),
                        tree: #head_tree.to_string()
                    },
                }})
        }

        /// Print all pre-generated hashes from the git repo.
        ///
        /// The hashes are generated by the build script, and can be used for auditing purposes.
        pub fn print_all() {
            println!("{}",get());
        }
    };

    let path = format!("{}/gitver_hashes.rs", std::env::var("OUT_DIR").unwrap());
    std::fs::write(path, format!("{code}")).unwrap();
}
