<?php
/**
 * BEdita, API-first content management framework
 * Copyright 2020 ChannelWeb Srl, Chialab Srl
 *
 * This file is part of BEdita: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * See LICENSE.LGPL or <http://gnu.org/licenses/lgpl-3.0.html> for more details.
 */

namespace App\Form;

use Cake\Core\Configure;
use Cake\Utility\Hash;
use Cake\Utility\Inflector;

/**
 * Control class provides methods for form controls, customized by control type.
 * Used by SchemaHelper to build control options for a property schema (@see \App\View\Helper\SchemaHelper::controlOptions)
 */
class Control
{
    /**
     * Control types
     */
    public const CONTROL_TYPES = [
        'json',
        'richtext',
        'plaintext',
        'date-time',
        'date',
        'checkbox',
        'checkboxNullable',
        'enum',
        'categories',
    ];

    /**
     * Map JSON schema properties to HTML attributes
     */
    public const JSON_SCHEMA_HTML_PROPERTIES_MAP = [
        'default' => 'value',
        'minimum' => 'min',
        'maximum' => 'max',
    ];

    /**
     * Get control by options
     *
     * @param array $options The options
     * @return array
     */
    public static function control(array $options): array
    {
        $type = $options['propertyType'];
        $oneOf = self::oneOf((array)$options['schema']);
        $format = (string)Hash::get($oneOf, 'format');
        $controlOptions = array_intersect_key($options, array_flip(['label', 'disabled', 'readonly', 'step', 'value']));
        foreach (self::JSON_SCHEMA_HTML_PROPERTIES_MAP as $key => $attribute) {
            if (array_key_exists($key, $oneOf)) {
                $controlOptions[$attribute] = $oneOf[$key];
            }
        }
        $defaultValue = Hash::get($controlOptions, 'value');
        if ($type === 'text' && in_array($format, ['email', 'uri'])) {
            $result = call_user_func_array(Form::getMethod(self::class, $type, $format), [$options]);

            return array_merge($controlOptions, $result);
        }
        if (!in_array($type, self::CONTROL_TYPES)) {
            $result = compact('type') + ['value' => $options['value'] === null && $defaultValue ? $defaultValue : $options['value']];

            return array_merge($controlOptions, $result);
        }
        $result = call_user_func_array(Form::getMethod(self::class, $type), [$options]);

        return array_merge($controlOptions, $result);
    }

    /**
     * Get format from schema
     *
     * @param array $schema The schema
     * @return string
     */
    public static function format(array $schema): string
    {
        return (string)Hash::get(self::oneOf($schema), 'format');
    }

    /**
     * Get oneOf from schema, excluding null values
     *
     * @param array $schema The schema
     * @return array
     */
    public static function oneOf(array $schema): array
    {
        $oneOf = array_filter(
            (array)Hash::get($schema, 'oneOf'),
            function ($item) {
                return !empty($item) && Hash::get($item, 'type') !== 'null';
            }
        );

        return (array)Hash::get(array_values($oneOf), 0);
    }

    /**
     * Control for json data
     *
     * @param array $options The options
     * @return array
     */
    public static function json(array $options): array
    {
        return [
            'type' => 'textarea',
            'v-jsoneditor' => 'true',
            'class' => 'json',
            'value' => json_encode(Hash::get($options, 'value')),
        ];
    }

    /**
     * Control for plaintext
     *
     * @param array $options The options
     * @return array
     */
    public static function plaintext(array $options): array
    {
        return [
            'type' => 'textarea',
            'value' => Hash::get($options, 'value'),
        ];
    }

    /**
     * Control for richtext
     *
     * @param array $options The options
     * @return array
     */
    public static function richtext(array $options): array
    {
        $schema = (array)Hash::get($options, 'schema');
        $value = Hash::get($options, 'value');
        $richeditorKey = !empty($schema['placeholders']) ? 'v-richeditor.placeholders' : 'v-richeditor';
        $override = !empty($options[$richeditorKey]);
        $toolbar = $override ? $options[$richeditorKey] : json_encode(Configure::read('RichTextEditor.default.toolbar', ''));

        return [
            'type' => 'textarea',
            $richeditorKey => $toolbar,
            'value' => $value,
        ];
    }

    /**
     * Control for datetime
     *
     * @param array $options The options
     * @return array
     */
    public static function datetime(array $options): array
    {
        return [
            'type' => 'text',
            'v-datepicker' => 'true',
            'date' => 'true',
            'time' => 'true',
            'value' => Hash::get($options, 'value'),
            'templates' => [
                'inputContainer' => '<div class="input datepicker {{type}}{{required}}">{{content}}</div>',
            ],
        ];
    }

    /**
     * Control for date
     *
     * @param array $options The options
     * @return array
     */
    public static function date(array $options): array
    {
        return [
            'type' => 'text',
            'v-datepicker' => 'true',
            'date' => 'true',
            'value' => Hash::get($options, 'value'),
            'templates' => [
                'inputContainer' => '<div class="input datepicker {{type}}{{required}}">{{content}}</div>',
            ],
        ];
    }

    /**
     * Control for categories
     *
     * @param array $options The options
     * @return array
     */
    public static function categories(array $options): array
    {
        $schema = (array)Hash::get($options, 'schema');
        $value = Hash::get($options, 'value');
        $categories = array_values(array_filter(
            (array)Hash::get($schema, 'categories'),
            function ($category) {
                return (bool)Hash::get($category, 'enabled');
            }
        ));
        $options = array_map(
            function ($category) {
                return [
                    'value' => Hash::get($category, 'name'),
                    'text' => empty($category['label']) ? $category['name'] : $category['label'],
                ];
            },
            $categories
        );

        $checked = [];
        if (!empty($value)) {
            $names = (array)Hash::extract($value, '{n}.name');
            foreach ($categories as $category) {
                if (in_array($category['name'], $names)) {
                    $checked[] = $category['name'];
                }
            }
        }

        return [
            'type' => 'select',
            'options' => $options,
            'multiple' => 'checkbox',
            'value' => $checked,
        ];
    }

    /**
     * Control for checkbox
     *
     * @param array $options The options
     * @return array
     */
    public static function checkbox(array $options): array
    {
        $schema = (array)Hash::get($options, 'schema');
        $value = Hash::get($options, 'value');
        if (empty($schema['oneOf'])) {
            return [
                'type' => 'checkbox',
                'checked' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
                'value' => '1',
            ];
        }

        $options = [];
        foreach ($schema['oneOf'] as $one) {
            self::oneOptions($options, $one);
        }
        if (!empty($options)) {
            return [
                'type' => 'select',
                'options' => $options,
                'multiple' => 'checkbox',
            ];
        }

        return [
            'type' => 'checkbox',
            'checked' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'value' => '1',
        ];
    }

    /**
     * Control for checkbox nullable
     *
     * @param array $options The options
     * @return array
     */
    public static function checkboxNullable(array $options): array
    {
        return [
            'type' => 'select',
            'options' => [
                Form::NULL_VALUE => '',
                '1' => __('Yes'),
                '0' => __('No'),
            ],
            'value' => Hash::get($options, 'value'),
        ];
    }

    /**
     * Set options for one of `oneOf` items.
     *
     * @param array $options The options to update
     * @param array $one The one item to check
     * @return void
     */
    public static function oneOptions(array &$options, array $one): void
    {
        $type = Hash::get($one, 'type');
        if ($type !== 'array') {
            return;
        }
        $options = array_map(
            function ($item) {
                return ['value' => $item, 'text' => Inflector::humanize($item)];
            },
            (array)Hash::extract($one, 'items.enum')
        );
    }

    /**
     * Control for enum
     *
     * @param array $options The options
     * @return array
     */
    public static function enum(array $options): array
    {
        $schema = (array)Hash::get($options, 'schema');
        $value = Hash::get($options, 'value');
        $objectType = Hash::get($options, 'objectType');
        $property = Hash::get($options, 'property');

        if (!empty($schema['oneOf'])) {
            foreach ($schema['oneOf'] as $one) {
                if (!empty($one['enum'])) {
                    $schema['enum'] = $one['enum'];
                    array_unshift($schema['enum'], '');
                }
            }
        }

        return [
            'type' => 'select',
            'options' => array_map(
                function (string $value) use ($objectType, $property) {
                    $text = self::label((string)$objectType, (string)$property, $value);

                    return compact('text', 'value');
                },
                $schema['enum']
            ),
            'value' => $value,
        ];
    }

    /**
     * Control for email
     *
     * @param array $options The options
     * @return array
     */
    public static function email(array $options): array
    {
        return [
            'type' => 'text',
            'v-email' => 'true',
            'class' => 'email',
            'value' => Hash::get($options, 'value'),
        ];
    }

    /**
     * Control for uri
     *
     * @param array $options The options
     * @return array
     */
    public static function uri(array $options): array
    {
        return [
            'type' => 'text',
            'v-uri' => 'true',
            'class' => 'uri',
            'value' => Hash::get($options, 'value'),
        ];
    }

    /**
     * Label for property.
     * If set in config Properties.<type>.labels.options.<property>, return it.
     * Return humanize of value, otherwise.
     *
     * @param string $type The object type
     * @param string $property The property name
     * @param string $value The value
     * @return string
     */
    public static function label(string $type, string $property, string $value): string
    {
        $label = Configure::read(sprintf('Properties.%s.labels.options.%s', $type, $property));
        if (empty($label)) {
            return Inflector::humanize($value);
        }
        $labelVal = (string)Configure::read(sprintf('Properties.%s.labels.options.%s.%s', $type, $property, $value));
        if (empty($labelVal)) {
            return Inflector::humanize($value);
        }

        return $labelVal;
    }
}
