/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
import * as vscode from 'vscode';
import { expect } from '@jest/globals';
import { restartLanguageServer, waitForExpectedResult } from './integrationHelpers';

function sortDiagnostics(diagnostics: vscode.Diagnostic[]): vscode.Diagnostic[] {
    return diagnostics.sort((a, b) => {
        const rangeCompare = a.range.start.compareTo(b.range.start);
        if (rangeCompare !== 0) {
            return rangeCompare;
        }

        return getCode(a).localeCompare(getCode(b));
    });
}

export async function waitForExpectedDiagnostics(
    assertExpectedDiagnostics: (input: [vscode.Uri, vscode.Diagnostic[]][]) => void,
    file?: vscode.Uri
): Promise<void> {
    const duration = 30 * 1000;
    const step = 500;
    await waitForExpectedResult<[vscode.Uri, vscode.Diagnostic[]][]>(
        () => {
            const diagnostics: [vscode.Uri, vscode.Diagnostic[]][] = [];
            if (file) {
                diagnostics.push([file, sortDiagnostics(vscode.languages.getDiagnostics(file))]);
            } else {
                const allDiagnostics = vscode.languages.getDiagnostics();
                for (const [uri, uriDiagnostics] of allDiagnostics) {
                    diagnostics.push([uri, sortDiagnostics(uriDiagnostics)]);
                }
            }

            return diagnostics;
        },
        duration,
        step,
        assertExpectedDiagnostics
    );
}

export async function setDiagnosticSettings(options: {
    compilerScope: string;
    analyzerScope: string;
    reportInformationAsHint: boolean;
}): Promise<void> {
    const dotnetConfig = vscode.workspace.getConfiguration('dotnet');
    await dotnetConfig.update('backgroundAnalysis.compilerDiagnosticsScope', options.compilerScope);
    await dotnetConfig.update('backgroundAnalysis.analyzerDiagnosticsScope', options.analyzerScope);
    await dotnetConfig.update('diagnostics.reportInformationAsHint', options.reportInformationAsHint);

    // Restart the language server to ensure diagnostics are reported with the correct configuration.
    // While in normal user scenarios it isn't necessary to restart the server to pickup diagnostic config changes,
    // we need to do it in integration tests for two reasons:
    //  1. We don't know when the server finishes processing the config change (its sent as a notification from client -> server).
    //  2. Even if we processed the config change, the VSCode API does not provide a way to re-request diagnostics after the config change.
    await restartLanguageServer();
}

export function getCode(diagnostic: vscode.Diagnostic): string {
    const code: {
        value: string | number;
        target: vscode.Uri;
    } = diagnostic.code as any;
    expect(code).toBeDefined();
    return code.value.toString();
}
