/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/

import * as vscode from 'vscode';
import * as path from 'path';
import * as semver from 'semver';
import { CSharpExtensionExports } from '../../../src/csharpExtensionExports';
import { existsSync } from 'fs';
import { ServerState } from '../../../src/lsptoolshost/server/languageServerEvents';
import testAssetWorkspace from './testAssets/testAssetWorkspace';
import { EOL, platform } from 'os';
import { describe, expect, test } from '@jest/globals';
import { WaitForAsyncOperationsRequest } from './testHooks';

export async function activateCSharpExtension(): Promise<CSharpExtensionExports> {
    const csharpExtension = vscode.extensions.getExtension<CSharpExtensionExports>('ms-dotnettools.csharp');
    if (!csharpExtension) {
        throw new Error('Failed to find installation of ms-dotnettools.csharp');
    }

    let shouldRestart = false;

    const csDevKitExtension = vscode.extensions.getExtension<CSharpExtensionExports>('ms-dotnettools.csdevkit');
    if (usingDevKit()) {
        if (!csDevKitExtension) {
            throw new Error('Failed to find installation of ms-dotnettools.csdevkit');
        }

        // Ensure C# Dev Kit has a minimum version.
        const version = csDevKitExtension.packageJSON.version;
        const minimumVersion = '1.10.18';
        if (semver.lt(version, minimumVersion)) {
            throw new Error(`C# Dev Kit version ${version} is below required minimum of ${minimumVersion}`);
        }
    } else {
        // Run a restore manually to make sure the project is up to date since we don't have automatic restore.
        await testAssetWorkspace.restoreLspToolsHostAsync();

        // If the extension is already active, we need to restart it to ensure we start with a clean server state.
        // For example, a previous test may have changed configs, deleted restored packages or made other changes that would put it in an invalid state.
        if (csharpExtension.isActive) {
            shouldRestart = true;
        }
    }

    // Explicitly await the extension activation even if completed so that we capture any errors it threw during activation.
    await csharpExtension.activate();
    await csharpExtension.exports.initializationFinished();
    console.log('ms-dotnettools.csharp activated');
    console.log(`Extension Log Directory: ${csharpExtension.exports.logDirectory}`);

    if (shouldRestart) {
        await restartLanguageServer();
    }

    return csharpExtension.exports;
}

export function usingDevKit(): boolean {
    return vscode.workspace.getConfiguration().get<boolean>('dotnet.preferCSharpExtension') !== true;
}

export async function openFileInWorkspaceAsync(relativeFilePath: string): Promise<vscode.Uri> {
    const uri = getFilePath(relativeFilePath);
    await vscode.commands.executeCommand('vscode.open', uri);
    return uri;
}

export function getFilePath(relativeFilePath: string): vscode.Uri {
    const root = vscode.workspace.workspaceFolders![0].uri.fsPath;
    const filePath = path.join(root, relativeFilePath);
    if (!existsSync(filePath)) {
        throw new Error(`File ${filePath} does not exist`);
    }

    return vscode.Uri.file(filePath);
}

export async function closeAllEditorsAsync(): Promise<void> {
    await vscode.commands.executeCommand('workbench.action.closeAllEditors');
}

/**
 * Reverts any unsaved changes to the active file.
 * Useful to reset state between tests without fully reloading everything.
 */
export async function revertActiveFile(): Promise<void> {
    await vscode.commands.executeCommand('workbench.action.files.revert');
}

export async function restartLanguageServer(): Promise<void> {
    if (usingDevKit()) {
        // Restarting the server will cause us to lose all project information when using C# Dev Kit.
        throw new Error('Cannot restart language server when using the C# Dev Kit');
    }
    const csharpExtension = vscode.extensions.getExtension<CSharpExtensionExports>('ms-dotnettools.csharp');
    // Register to wait for initialization events and restart the server.
    const waitForInitialProjectLoad = new Promise<void>((resolve, _) => {
        csharpExtension!.exports.experimental.languageServerEvents.onServerStateChange(async (e) => {
            if (e.state === ServerState.ProjectInitializationComplete) {
                resolve();
            }
        });
    });
    await vscode.commands.executeCommand('dotnet.restartServer');
    await waitForInitialProjectLoad;
}

export function isRazorWorkspace(workspace: typeof vscode.workspace) {
    return isGivenSln(workspace, 'RazorApp');
}

export function isSlnWithGenerator(workspace: typeof vscode.workspace) {
    return isGivenSln(workspace, 'slnWithGenerator');
}

export async function getCompletionsAsync(
    position: vscode.Position,
    triggerCharacter: string | undefined,
    completionsToResolve: number
): Promise<vscode.CompletionList> {
    const activeEditor = vscode.window.activeTextEditor;
    if (!activeEditor) {
        throw new Error('No active editor');
    }

    return await vscode.commands.executeCommand(
        'vscode.executeCompletionItemProvider',
        activeEditor.document.uri,
        position,
        triggerCharacter,
        completionsToResolve
    );
}

export async function getCodeLensesAsync(): Promise<vscode.CodeLens[]> {
    const activeEditor = vscode.window.activeTextEditor;
    if (!activeEditor) {
        throw new Error('No active editor');
    }

    // The number of code lens items to resolve.  Set to a high number so we get pretty much everything in the document.
    const resolvedItemCount = 100;

    let tryCount = 0;
    const maxRetryCount = 3;
    do {
        try {
            const result = await executeCodeLensProviderAsync(activeEditor, resolvedItemCount);
            return result;
        } catch (e) {
            tryCount++;
            // It is totally possible that the code lens request is cancelled due to the server returning a content modified error.
            // This is not an error condition - it just means that the server snapshot has moved on and we need to retry the request.
            //
            // This error is not thrown as an error type that matches ours, so we'll check the name of the error to determine if it was a cancellation.
            if (Object.prototype.hasOwnProperty.call(e, 'name') && (e as any).name === 'Canceled') {
                console.log('CodeLens provider was cancelled, retrying in 1 second');
                await sleep(1000);
            } else {
                console.log('CodeLens provider encountered unexpected error');
                console.log(JSON.stringify(e));
                throw e;
            }
        }
    } while (tryCount < maxRetryCount);
    throw new Error(`Failed to get code lenses after ${maxRetryCount} retries`);
}

async function executeCodeLensProviderAsync(
    activeEditor: vscode.TextEditor,
    resolvedItemCount: number
): Promise<vscode.CodeLens[]> {
    const codeLenses = <vscode.CodeLens[]>(
        await vscode.commands.executeCommand(
            'vscode.executeCodeLensProvider',
            activeEditor.document.uri,
            resolvedItemCount
        )
    );
    return codeLenses.sort((a, b) => {
        const rangeCompare = a.range.start.compareTo(b.range.start);
        if (rangeCompare !== 0) {
            return rangeCompare;
        }
        return a.command!.title.localeCompare(b.command!.command);
    });
}

export async function navigate(
    originalPosition: vscode.Position,
    definitionLocations: vscode.Location[],
    expectedFileName: string
): Promise<void> {
    const windowChanged = new Promise<void>((resolve, _) => {
        vscode.window.onDidChangeActiveTextEditor((_e) => {
            if (_e?.document.fileName.includes(expectedFileName)) {
                resolve();
            }
        });
    });

    await vscode.commands.executeCommand(
        'editor.action.goToLocations',
        vscode.window.activeTextEditor!.document.uri,
        originalPosition,
        definitionLocations,
        'goto',
        'Failed to navigate'
    );

    // Navigation happens asynchronously when a different file is opened, so we need to wait for the window to change.
    await windowChanged;

    expect(vscode.window.activeTextEditor?.document.fileName).toContain(expectedFileName);
}

export function sortLocations(locations: vscode.Location[]): vscode.Location[] {
    return locations.sort((a, b) => {
        const uriCompare = a.uri.fsPath.localeCompare(b.uri.fsPath);
        if (uriCompare !== 0) {
            return uriCompare;
        }

        return a.range.start.compareTo(b.range.start);
    });
}

export function findRangeOfString(editor: vscode.TextEditor, stringToFind: string): vscode.Range[] {
    const text = editor.document.getText();
    const matches = [...text.matchAll(new RegExp(stringToFind, 'gm'))];
    const ranges = matches.map((match) => {
        const startPos = editor.document.positionAt(match.index!);
        const endPos = editor.document.positionAt(match.index! + stringToFind.length);
        return new vscode.Range(startPos, endPos);
    });
    return ranges;
}

function isGivenSln(workspace: typeof vscode.workspace, expectedProjectFileName: string) {
    const primeWorkspace = workspace.workspaceFolders![0];
    const projectFileName = primeWorkspace.uri.fsPath.split(path.sep).pop();

    return projectFileName === expectedProjectFileName;
}

export async function waitForExpectedResult<T>(
    getValue: () => Promise<T> | T,
    duration: number,
    step: number,
    expression: (input: T) => void
): Promise<void> {
    let value: T;
    let error: any = undefined;

    while (duration > 0) {
        value = await getValue();

        try {
            expression(value);
            return;
        } catch (e) {
            error = e;
            // Wait for a bit and try again.
            await new Promise((r) => setTimeout(r, step));
            duration -= step;
        }
    }

    throw new Error(`Polling did not succeed within the alotted duration: ${error}`);
}

export async function sleep(ms = 0) {
    return new Promise((r) => setTimeout(r, ms));
}

export async function expectText(document: vscode.TextDocument, expectedLines: string[]) {
    const expectedText = expectedLines.join(EOL);
    expect(document.getText()).toBe(expectedText);
}

export function expectPath(expected: vscode.Uri, actual: vscode.Uri) {
    if (isLinux()) {
        expect(actual.path).toBe(expected.path);
    } else {
        const actualPath = actual.path.toLowerCase();
        const expectedPath = expected.path.toLocaleLowerCase();
        expect(actualPath).toBe(expectedPath);
    }
}

export const describeIfCSharp = describeIf(!usingDevKit());
export const describeIfDevKit = describeIf(usingDevKit());
export const describeIfNotMacOS = describeIf(!isMacOS());
export const describeIfWindows = describeIf(isWindows());
export const testIfCSharp = testIf(!usingDevKit());
export const testIfDevKit = testIf(usingDevKit());
export const testIfNotMacOS = testIf(!isMacOS());
export const testIfWindows = testIf(isWindows());

const runFileBasedProgramsTests = process.env['ROSLYN_SKIP_TEST_FILE_BASED_PROGRAMS'] !== 'true';
console.log(`process.env.ROSLYN_SKIP_TEST_FILE_BASED_PROGRAMS: ${process.env.ROSLYN_SKIP_TEST_FILE_BASED_PROGRAMS}`);
export const describeIfFileBasedPrograms = describeIf(runFileBasedProgramsTests);

function describeIf(condition: boolean) {
    return condition ? describe : describe.skip;
}

function testIf(condition: boolean) {
    return condition ? test : test.skip;
}

function isMacOS() {
    const currentPlatform = platform();
    return currentPlatform === 'darwin';
}

function isWindows() {
    const currentPlatform = platform();
    return currentPlatform === 'win32';
}

function isLinux() {
    return !(isMacOS() || isWindows());
}

export async function waitForAllAsyncOperationsAsync(exports: CSharpExtensionExports): Promise<void> {
    const source = new vscode.CancellationTokenSource();
    await exports.experimental.sendServerRequest(WaitForAsyncOperationsRequest.type, { operations: [] }, source.token);
}
