/*
 Copyright 2020-2022 Enock Nicholaus <3nock@protonmail.com>. All rights reserved.
 Use of this source code is governed by GPL-3.0 LICENSE that can be found in the LICENSE file.

 @brief :
*/

#include "SSLTool.h"
#include "ui_SSLTool.h"

#include "src/dialogs/FailedScansDialog.h"
#include "src/modules/active/SSLScanner.h"


void SSLTool::startScan(){
    ui->buttonStop->setEnabled(true);
    ui->buttonStart->setText(tr("Pause"));

    /* status */
    status->isRunning = true;
    status->isNotActive = false;
    status->isStopped = false;
    status->isPaused = false;

    /* ressetting and setting new values */
    ui->progressBar->show();
    ui->progressBar->reset();
    m_failedScans.clear();

    /*
     if the numner of threads is greater than the number of wordlists, set the
     number of threads to use to the number of wordlists available to avoid
     creating more threads than needed...
    */
    if(m_scanArgs->config->threads > m_scanArgs->targets.length())
        status->activeScanThreads = m_scanArgs->targets.length();
    else
        status->activeScanThreads = m_scanArgs->config->threads;

    /* renewing scan statistics */
    m_scanStats->failed = 0;
    m_scanStats->resolved = 0;
    m_scanStats->threads = status->activeScanThreads;
    m_scanStats->targets = m_scanArgs->targets.length();

    /* set progressbar maximum value */
    ui->progressBar->setMaximum(m_scanArgs->targets.length());
    m_scanArgs->progress = 0;

    /* start timer */
    m_timer.start();

    switch (ui->comboBoxOutput->currentIndex()) {
        case 0: // subdomain
            m_scanArgs->output = ssl::OUTPUT::SUBDOMAIN;
            break;
        case 1: // cert id
            switch (ui->comboBoxOption->currentIndex()) {
                case 0: // SHA1
                    m_scanArgs->output = ssl::OUTPUT::SHA1;
                    break;
                case 1: // SHA256
                    m_scanArgs->output = ssl::OUTPUT::SHA256;
                    break;
            }
            break;
        case 2: // cert raw
            m_scanArgs->output = ssl::OUTPUT::RAW;
            break;
    }

    switch (ui->comboBoxProtocal->currentIndex()) {
    case 0: // HTTPS
        m_scanArgs->port = ssl::PORT::HTTPS;
        break;
    case 1: // FTP
        m_scanArgs->port = ssl::PORT::FTP;
        break;
    case 2: // FTPs
        m_scanArgs->port = ssl::PORT::FTPs;
        break;
    case 3: // SSH
        m_scanArgs->port = ssl::PORT::SSH;
        break;
    }

    /* loop to create threads for enumeration... */
    for(int i = 0; i < status->activeScanThreads; i++)
    {
        ssl::Scanner *scanner = new ssl::Scanner(m_scanArgs);
        QThread *cThread = new QThread;
        scanner->startScan(cThread);
        scanner->moveToThread(cThread);

        /* results signals & slots... */
        switch (ui->comboBoxOutput->currentIndex()) {
        case 0: // subdomain
            connect(scanner, &ssl::Scanner::resultSubdomain, this, &SSLTool::onScanResultSubdomain);
            break;
        case 1: // cert id
            switch (ui->comboBoxOption->currentIndex()) {
            case 0: // SHA1
                connect(scanner, &ssl::Scanner::resultSHA1, this, &SSLTool::onScanResultSHA1);
                break;
            case 1: // SHA256
                connect(scanner, &ssl::Scanner::resultSHA256, this, &SSLTool::onScanResultSHA256);
                break;
            }
            break;
        case 2: // raw cert
            connect(scanner, &ssl::Scanner::resultRaw, this, &SSLTool::onScanResultRaw);
            break;
        }
        connect(scanner, &ssl::Scanner::scanProgress, ui->progressBar, &QProgressBar::setValue);
        connect(scanner, &ssl::Scanner::scanLog, this, &SSLTool::onScanLog);
        connect(cThread, &QThread::finished, this, &SSLTool::onScanThreadEnded);
        connect(cThread, &QThread::finished, scanner, &ssl::Scanner::deleteLater);
        connect(cThread, &QThread::finished, cThread, &QThread::deleteLater);
        connect(this, &SSLTool::stopScanThread, scanner, &ssl::Scanner::onStopScan, Qt::DirectConnection);
        connect(this, &SSLTool::pauseScanThread, scanner, &ssl::Scanner::onPauseScan, Qt::DirectConnection);
        connect(this, &SSLTool::resumeScanThread, scanner, &ssl::Scanner::onResumeScan, Qt::DirectConnection);
        cThread->start();
    }
    status->isRunning = true;
}

void SSLTool::onReScan(QQueue<QString> targets){
    if(targets.isEmpty())
        return;

    /* clear */
    m_scanArgs->targets.clear();

    /* get targets */
    m_scanArgs->targets = targets;

    /* start scan */
    this->startScan();

    /* logs */
    log("----------------- Re-Scan ---------------\n");
    qInfo() << "[SSL] Re-Scan Started";
}

void SSLTool::onScanThreadEnded(){
    status->activeScanThreads--;

    /* if all Scan Threads have finished... */
    if(status->activeScanThreads == 0)
    {
        /* display the scan summary on logs */
        this->scanSummary();

        if(status->isStopped)
            log("---------------- Stopped ------------\n");
        else
            log("------------------ End --------------\n");

        qInfo() << "[SSL] Scan Ended";

        /* set the progress bar to 100% just in case... */
        if(!status->isStopped)
            ui->progressBar->setValue(ui->progressBar->maximum());

        m_scanArgs->targets.clear();

        status->isNotActive = true;
        status->isPaused = false;
        status->isStopped = false;
        status->isRunning = false;

        ui->buttonStart->setText("Start");
        ui->buttonStop->setDisabled(true);

        // launching the failed scans dialog if there were failed scans
        if(!m_failedScans.isEmpty()){
            FailedScansDialog *failedScansDialog = new FailedScansDialog(this, m_failedScans);
            failedScansDialog->setAttribute(Qt::WA_DeleteOnClose, true);

            connect(failedScansDialog, &FailedScansDialog::reScan, this, &SSLTool::onReScan);
            failedScansDialog->show();
        }
    }
}

void SSLTool::scanSummary(){
    /* elapsed time */
    QTime time = QTime::fromMSecsSinceStartOfDay(m_timer.elapsed());

    /* write to log file */
    ui->plainTextEditLogs->appendHtml("<font color=\"white\">  [ Scan Summary ]</font>");
    ui->plainTextEditLogs->appendHtml("[ Resolved ]    : <font color=\"green\">"+QString::number(m_scanStats->resolved)+"</font>");
    ui->plainTextEditLogs->appendHtml("[ Failed ]      : <font color=\"red\">"+QString::number(m_scanStats->failed)+"</font>");
    ui->plainTextEditLogs->appendHtml("[ Threads ]     : <font color=\"green\">"+QString::number(m_scanStats->threads)+"</font>");
    ui->plainTextEditLogs->appendHtml("[ Targets ]     : <font color=\"green\">"+QString::number(m_scanStats->targets)+"</font>");
    ui->plainTextEditLogs->appendHtml("[ Time (hh:mm:ss:zzz) ]        : <font color=\"green\">"+time.toString("hh:mm:ss:zzz")+"</font>");
}
