package br.com.etyllica.gui;

import java.awt.Color;

import br.com.etyllica.core.event.Action;
import br.com.etyllica.core.event.GUIEvent;
import br.com.etyllica.core.event.KeyEvent;
import br.com.etyllica.core.event.MouseButton;
import br.com.etyllica.core.event.PointerEvent;
import br.com.etyllica.core.graphics.Graphics;
import br.com.etyllica.gui.base.BaseButton;
import br.com.etyllica.gui.icon.DownArrow;
import br.com.etyllica.gui.icon.UpArrow;
import br.com.etyllica.gui.panel.ScrollBackground;

/**
 * 
 * @author yuripourre
 * 
 */

public class ScrollView extends View {

    private View component;

    private int buttonSize = 20;
    private int scrollAmount = 10;
    private int scrollCursor = 0;
    private int steps = 0;
    private float scrollFactor = 1;
    private float offset = 0;
    private float knobPosition = 0;

    private BaseButton upButton;
    private BaseButton downButton;
    private BaseButton knob;
    private ScrollBackground track;

    public ScrollView(int x, int y, int w, int h) {
        super(x, y, w, h);

        clipOnDraw = true;
        build();
    }
    
    private void build() {
    	upButton = new BaseButton(x + w - buttonSize, y, buttonSize, buttonSize);
        upButton.setLabel(new UpArrow((-buttonSize / 3), -buttonSize / 3, buttonSize / 2));
        upButton.addAction(GUIEvent.MOUSE_LEFT_BUTTON_UP, new Action(this, "scrollUp"));
        upButton.setVisible(false);

        downButton = new BaseButton(x + w - buttonSize, y + h - buttonSize, buttonSize, buttonSize);
        downButton.setLabel(new DownArrow(-buttonSize / 3, -buttonSize / 3, buttonSize / 2));
        downButton.addAction(GUIEvent.MOUSE_LEFT_BUTTON_UP, new Action(this, "scrollDown"));
        downButton.setVisible(false);

        track = new ScrollBackground(x + w - buttonSize, y + buttonSize, buttonSize, h - buttonSize * 2);
        track.setVisible(false);

        add(track);
        add(upButton);
        add(downButton);
    }
    
    @Override
    public void resize() {
    	upButton.setBounds(x + w - buttonSize, y, buttonSize, buttonSize);
    	downButton.setBounds(x + w - buttonSize, y + h - buttonSize, buttonSize, buttonSize);
    	track.setBounds(x + w - buttonSize, y + buttonSize, buttonSize, h - buttonSize * 2);
    	
    	/*if(component != null) {
    		setComponent(component);
    	}*/
    }

    @Override
    public void draw(Graphics g) {
        g.setColor(Color.WHITE);
        g.fillRect(left(), top(), width(), height());
    }
    
    @Override
    public void drawWithChildren(Graphics g) {
    	if (clipOnDraw) {
    		g.setClip(x, y, w, h);
    	}
    	super.drawWithChildren(g);
    	if (clipOnDraw) {
    		g.resetClip();	
    	}
	}
    
    @Override
    public void update(GUIEvent event) {

    }

    @Override
    public GUIEvent updateMouse(PointerEvent event) {

        if (mouseOver) {

            if (event.isButtonDown(MouseButton.MOUSE_WHEEL_DOWN)) {

                for (int i = 0; i < event.getAmount(); i++) {
                    scrollDown();
                }
            }

            if (event.isButtonDown(MouseButton.MOUSE_WHEEL_UP)) {

                for (int i = event.getAmount(); i < 0; i++) {
                    scrollUp();
                }
            }

            if (knob.isMouseOver()) {

                if (event.isButtonDown(MouseButton.MOUSE_BUTTON_LEFT)) {
                    //TODO Mouse dragged with knob move scroll
                }

            }
        }

        return GUIEvent.NONE;
    }

    @Override
    public GUIEvent updateKeyboard(KeyEvent event) {

        if (event.isKeyDown(KeyEvent.VK_TAB)) {
            return GUIEvent.NEXT_COMPONENT;
        }

        return GUIEvent.NONE;
    }

    public void setComponent(View component) {
    	views.clear();
    	views.add(component);
    	build();
    	this.component = component;
    	component.cascadeClipOnDraw(false);
    	    	
        knobPosition = buttonSize;
        resetScroll();
        this.component.setBounds(x, y, component.getW(), component.getH());
        this.component.resize();
    }

    private void resetScroll() {
        if (component.getH() <= h) {
            return;
        }

        scrollCursor = 0;
        scrollFactor = ((float) (h) / (float) component.getH());
        
        float utilScrollH = (h - buttonSize * 2);
        float scrollSize = utilScrollH * scrollFactor;

        float dif = component.getH() - h;
        steps = (int) Math.abs(dif / scrollAmount);
        
        offset = (utilScrollH-scrollSize) / steps;

        if (knob == null) {
            knob = new BaseButton(x + w - buttonSize, y + (int) (knobPosition), buttonSize, ((int) (scrollSize)));
            add(knob);
        } else {
            knob.setBounds(x + w - buttonSize, y + (int) (knobPosition), buttonSize, ((int) (scrollSize)));
        }
        knob.setVisible(false);

        showButtons();
    }

    private void showButtons() {
        track.setVisible(true);
        upButton.setVisible(true);
        downButton.setVisible(true);
        knob.setVisible(true);
    }

    public void scrollDown() {
        if (scrollCursor < steps) {
            scrollCursor++;
            component.setOffsetY(-scrollAmount);
            component.resize();
            knobPosition = buttonSize+offset*scrollCursor;
            knob.setY(y + (int) knobPosition);
        }
    }

    public void scrollUp() {

        if (scrollCursor > 0) {
            scrollCursor--;
            component.setOffsetY(+scrollAmount);
            component.resize();
            knobPosition = buttonSize+offset*scrollCursor;
            knob.setY(y + (int) knobPosition);
        }
    }

    public BaseButton getKnob() {
        return knob;
    }

    public int getButtonSize() {
        return buttonSize;
    }

    public void setButtonSize(int buttonSize) {
        this.buttonSize = buttonSize;
    }

}