/**
 * This file is part of DCD, a development tool for the D programming language.
 * Copyright (C) 2014 Brian Schott
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

module server;

import std.socket;
import std.stdio;
import std.getopt;
import std.algorithm;
import std.path;
import std.file;
import std.array;
import std.process;
import std.datetime;
import std.conv;
import std.allocator;

import core.memory;

import msgpack;

import messages;
import autocomplete;
import modulecache;
import stupidlog;
import actypes;
import core.memory;
import dcd_version;

enum CONFIG_FILE_NAME = "dcd.conf";

version(linux) version = useXDG;
version(BSD) version = useXDG;
version(FreeBSD) version = useXDG;
version(OSX) version = useXDG;

int main(string[] args)
{
	Log.info("Starting up...");
	StopWatch sw = StopWatch(AutoStart.yes);

	Log.output = stdout;
	Log.level = LogLevel.trace;

	ushort port = 9166;
	bool help;
	bool printVersion;
	string[] importPaths;

	try
	{
		getopt(args, "port|p", &port, "I", &importPaths, "help|h", &help,
			"version", & printVersion);
	}
	catch (ConvException e)
	{
		Log.fatal(e.msg);
		printHelp(args[0]);
		return 1;
	}

	if (printVersion)
	{
		writeln(DCD_VERSION);
		return 0;
	}

	if (help)
	{
		printHelp(args[0]);
		return 0;
	}

	importPaths ~= loadConfiguredImportDirs();

	auto socket = new TcpSocket(AddressFamily.INET);
	socket.blocking = true;
	socket.setOption(SocketOptionLevel.SOCKET, SocketOption.REUSEADDR, true);
	socket.bind(new InternetAddress("localhost", port));
	socket.listen(0);
	scope (exit)
	{
		Log.info("Shutting down sockets...");
		socket.shutdown(SocketShutdown.BOTH);
		socket.close();
		Log.info("Sockets shut down.");
	}

	ModuleCache.addImportPaths(importPaths);
	Log.info("Import directories: ", ModuleCache.getImportPaths());

	ubyte[] buffer = cast(ubyte[]) Mallocator.it.allocate(1024 * 1024 * 4); // 4 megabytes should be enough for anybody...
	scope(exit) Mallocator.it.deallocate(buffer);

	sw.stop();
	Log.info(ModuleCache.symbolsAllocated, " symbols cached.");
	Log.info("Startup completed in ", sw.peek().to!("msecs", float), " milliseconds.");
	import core.memory : GC;
	GC.minimize();


	// No relative paths
	version (Posix) chdir("/");

	serverLoop: while (true)
	{
		auto s = socket.accept();
		s.blocking = true;

		// TODO: Restrict connections to localhost

		scope (exit)
		{
			s.shutdown(SocketShutdown.BOTH);
			s.close();
		}
		ptrdiff_t bytesReceived = s.receive(buffer);

		auto requestWatch = StopWatch(AutoStart.yes);

		size_t messageLength;
		// bit magic!
		(cast(ubyte*) &messageLength)[0..size_t.sizeof] = buffer[0..size_t.sizeof];
		while (bytesReceived < messageLength + size_t.sizeof)
		{
			auto b = s.receive(buffer[bytesReceived .. $]);
			if (b == Socket.ERROR)
			{
				bytesReceived = Socket.ERROR;
				break;
			}
			bytesReceived += b;
		}

		if (bytesReceived == Socket.ERROR)
		{
			Log.error("Socket recieve failed");
			break;
		}

		AutocompleteRequest request;
		msgpack.unpack(buffer[size_t.sizeof .. bytesReceived], request);
		if (request.kind & RequestKind.clearCache)
		{
			Log.info("Clearing cache.");
			ModuleCache.clear();
		}
		else if (request.kind & RequestKind.shutdown)
		{
			Log.info("Shutting down.");
			break serverLoop;
		}
		else if (request.kind & RequestKind.query)
		{
			AutocompleteResponse response;
			response.completionType = "ack";
			ubyte[] responseBytes = msgpack.pack(response);
			s.send(responseBytes);
		}
		if (request.kind & RequestKind.addImport)
		{
			ModuleCache.addImportPaths(request.importPaths);
			GC.minimize();
		}
		if (request.kind & RequestKind.autocomplete)
		{
			Log.info("Getting completions");
			AutocompleteResponse response = complete(request);
			ubyte[] responseBytes = msgpack.pack(response);
			s.send(responseBytes);
		}
		else if (request.kind & RequestKind.doc)
		{
			Log.info("Getting doc comment");
			try
			{
				AutocompleteResponse response = getDoc(request);
				ubyte[] responseBytes = msgpack.pack(response);
				s.send(responseBytes);
			}
			catch (Exception e)
			{
				Log.error("Could not get DDoc information", e.msg);
			}
		}
		else if (request.kind & RequestKind.symbolLocation)
		{
			try
			{
				AutocompleteResponse response = findDeclaration(request);
				ubyte[] responseBytes = msgpack.pack(response);
				s.send(responseBytes);
			}
			catch (Exception e)
			{
				Log.error("Could not get symbol location", e.msg);
			}
		}
		else if (request.kind & RequestKind.search)
		{
			AutocompleteResponse response = symbolSearch(request);
			ubyte[] responseBytes = msgpack.pack(response);
			s.send(responseBytes);
		}
		Log.info("Request processed in ", requestWatch.peek().to!("msecs", float), " milliseconds");
	}
	return 0;
}

/**
 * Locates the configuration file
 */
string getConfigurationLocation()
{
	version (useXDG)
	{
		string configDir = environment.get("XDG_CONFIG_HOME", null);
		if (configDir is null)
		{
			configDir = environment.get("HOME", null);
			if (configDir is null)
				throw new Exception("Both $XDG_CONFIG_HOME and $HOME are unset");
			configDir = buildPath(configDir, ".config", "dcd", CONFIG_FILE_NAME);
		}
		else
		{
			configDir = buildPath(configDir, "dcd", CONFIG_FILE_NAME);
		}
		return configDir;
	}
	else version(Windows)
	{
		return CONFIG_FILE_NAME;
	}
}

void warnAboutOldConfigLocation()
{
	version (linux) if ("~/.config/dcd".expandTilde().exists()
		&& "~/.config/dcd".expandTilde().isFile())
	{
		Log.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
		Log.error("!! Upgrade warning:");
		Log.error("!! '~/.config/dcd' should be moved to '$XDG_CONFIG_HOME/dcd/dcd.conf'");
		Log.error("!! or '$HOME/.config/dcd/dcd.conf'");
		Log.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
	}
}

/**
 * Loads import directories from the configuration file
 */
string[] loadConfiguredImportDirs()
{
	warnAboutOldConfigLocation();
	immutable string configLocation = getConfigurationLocation();
	if (!configLocation.exists())
		return [];
	Log.info("Loading configuration from ", configLocation);
	File f = File(configLocation, "rt");
	return f.byLine(KeepTerminator.no)
		.filter!(a => a.length > 0 && existanceCheck(a))
		.map!(a => a.idup)
		.array();
}

void printHelp(string programName)
{
    writefln(
`
    Usage: %s options

options:
    -I PATH
        Includes PATH in the listing of paths that are searched for file
        imports.

    --help | -h
        Prints this help message.

    --version
        Prints the version number and then exits.

    --port PORTNUMBER | -pPORTNUMBER
        Listens on PORTNUMBER instead of the default port 9166.`, programName);
}
