package main

import (
	"encoding/json"
	"fmt"
)

// for tcp
const (
	MaxBodySize = int32(1 << 10)
)

const (
	// size
	PackSize      = 4
	HeaderSize    = 2
	VerSize       = 2
	OperationSize = 4
	SeqIdSize     = 4
	RawHeaderSize = PackSize + HeaderSize + VerSize + OperationSize + SeqIdSize
	MaxPackSize   = MaxBodySize + int32(RawHeaderSize)
	// offset
	PackOffset      = 0
	HeaderOffset    = PackOffset + PackSize
	VerOffset       = HeaderOffset + HeaderSize
	OperationOffset = VerOffset + VerSize
	SeqIdOffset     = OperationOffset + OperationSize
	EndOffset       = SeqIdOffset + SeqIdSize
)

var (
	emptyProto = Proto{}
)

// Proto is a request&response written before every goim connect.  It is used internally
// but documented here as an aid to debugging, such as when analyzing
// network traffic.
// tcp:
// binary codec
// websocket & http:
// raw codec, with http header stored ver, operation, seqid
type Proto struct {
	PackLen   int32               `json:"-"`    // pack length
	HeaderLen int16               `json:"-"`    // header length
	Ver       int16               `json:"ver"`  // protocol version
	Operation int32               `json:"op"`   // operation for request
	SeqId     int32               `json:"seq"`  // sequence number chosen by client
	Body      json.RawMessage     `json:"body"` // binary body bytes(json.RawMessage is []byte)
	Readbuf   [MaxBodySize]byte   `json:"-"`    // read buffer
	Writebuf  [RawHeaderSize]byte `json:"-"`    // write buffer
}

func (p *Proto) Reset() {
	*p = emptyProto
}

func (p *Proto) String() string {
	return fmt.Sprintf("\n-------- proto --------\npack: %d\nheader: %d\nver: %d\nop: %d\nseq: %d\nbody: %s\n-----------------------", p.PackLen, p.HeaderLen, p.Ver, p.Operation, p.SeqId, string(p.Body))
}
