---
layout: post
title: "4.2 RESTier Publishers [<=0.3.0-beta2]"
description: ""
category: "4. Deep in RESTier"
---

The publisher component of RESTier exposes the domain-specific data as various APIs which could be easily consumed by clients. Now RESTier has one publisher implemented upon Web API for OData. The **primary part** of the publisher is the `ODataDomainController<T>` class which derives from the `ODataController` class in Web API for OData. One of the big advantages of RESTier is that we need to write **only one controller** which handles all resources. To figure out how it works, we may want to take a look at the source code.

In Visual Studio, open the solution file **RESTier.sln**. From the **Solution Explorer** window, open the project `Microsoft.Restier.WebApi` under **RESTier > src > Publishers**. Then double-click each source file under the project to view its content.

![]({{site.baseurl}}/images/publisher.png)

This section will dive into two scenarios and go through the main source code of the publisher:

 - Application startup
 - Processing a request
 <br />

### Application Startup
Usually we put a call to `MapODataDomainRoute<TController>` in the **WebApiConfig.cs** file which contains the application startup code. The following code is an example taken from our **AdventureWorksLT** sample.

{% highlight csharp %}
public static async void RegisterAdventureWorks(HttpConfiguration config, HttpServer server)
{
    await config.MapODataDomainRoute<AdventureWorksController>(
        "AdventureWorksLT",
        "AdventureWorksLT",
        new ODataDomainBatchHandler(server)
        );
}
{% endhighlight %}

The **key part** of the implementation of `MapODataDomainRoute<TController>` is as follows. This method first calls `CreateODataDomainRoutingConventions<TController>` to create routing conventions for the domain controller. Then it creates an `ODataRoute` based on the conventions and adds it into Web API.

{% highlight csharp %}
public static async Task<ODataRoute> MapODataDomainRoute<TController>(
    this HttpConfiguration config, string routeName, string routePrefix,
    Func<IDomain> domainFactory,
    ODataDomainBatchHandler batchHandler = null)
    where TController : ODataDomainController, new()
{
    ...
    var model = await domain.GetModelAsync();
    var conventions = CreateODataDomainRoutingConventions<TController>(config, model);
    ...
    var routeConstraint = new DefaultODataPathRouteConstraint(odataPathHandler, model, routeName, conventions);
    var route = new ODataRoute(routePrefix, routeConstraint);
    routes.Add(routeName, route);
    return route;
}
{% endhighlight %}

The `CreateODataDomainRoutingConventions<TController>` method is also worth mentioning. It first adds a `DefaultODataRoutingConvention` (for domain controller routing) somewhere in the convention list. Then it inserts an `AttributeRoutingConvention` (for attribute routing) **at the first position** of the convention list. This is why we can **override the default route** in the domain controller using attribute routing in *Section 3.3*.

{% highlight csharp %}
public static IList<IODataRoutingConvention> CreateODataDomainRoutingConventions<TController>(
    this HttpConfiguration config, IEdmModel model)
    where TController : ODataDomainController, new()
{
    var conventions = ODataRoutingConventions.CreateDefault();
    var index = 0;
    for (; index < conventions.Count; index++)
    {
        var unmapped = conventions[index] as UnmappedRequestRoutingConvention;
        if (unmapped != null)
        {
            break;
        }
    }

    conventions.Insert(index, new DefaultODataRoutingConvention(typeof(TController).Name));
    conventions.Insert(0, new AttributeRoutingConvention(model, config));
    return conventions;
}
{% endhighlight %}

### Processing a Request
Studying how a request is processed step by step is the best way to be familiar with the RESTier internals. In this section, we will only look at a typical **GET** request to illustrate the main thread of the framework. When a request comes to the service, Web API for OData will dispatch it to the `ODataRoute` instance created above. To route a request, we must first determine the controller then the action under it.

#### Determine the Controller
So through a call chain, the `DefaultODataPathRouteConstraint.SelectControllerName` method is called with the request.

{% highlight csharp %}
protected override string SelectControllerName(ODataPath path, HttpRequestMessage request)
{
    var controllers = request.GetConfiguration().Services.GetHttpControllerSelector().GetControllerMapping();
    foreach (var routingConvention in RoutingConventions)
    {
        var controllerName = routingConvention.SelectController(path, request);
        if (controllerName != null)
        {
            HttpControllerDescriptor descriptor;
            if (controllers.TryGetValue(controllerName, out descriptor) && descriptor != null)
            {
                return controllerName;
            }
        }
    }

    return null;
}
{% endhighlight %}

This method matches the request against all routing conventions and returns the selected controller name. If the request is to be handled by the domain controller, the `DefaultODataRoutingConvention.SelectController` method would be called.

{% highlight csharp %}
public string SelectController(ODataPath odataPath, HttpRequestMessage request)
{
    ...
    return this.controllerName;
}
{% endhighlight %}

This method simply returns the controller name passed through the `DefaultODataRoutingConvention` constructor, which is usually **the name of the domain controller**.

#### Determine the Action
Then through another call chain, the `DefaultODataRoutingConvention.SelectAction` method is called to determine the action name.

{% highlight csharp %}
public string SelectAction(ODataPath odataPath, HttpControllerContext controllerContext, ILookup<string, HttpActionDescriptor> actionMap)
{
    ...
    HttpMethod method = controllerContext.Request.Method;

    if (method == HttpMethod.Get && !IsMetadataPath(odataPath))
    {
        return "Get";
    }

    ODataPathSegment lastSegment = odataPath.Segments.LastOrDefault();
    if (lastSegment != null && lastSegment.SegmentKind == ODataSegmentKinds.UnboundAction)
    {
        return "PostAction";
    }

    // Let WebAPI select default action
    return null;
}
{% endhighlight %}

This method basically selects the `ODataDomainController.Get` method for HTTP **GET** requests (except those to **service document** and **metadata**) or the `ODataDomainController.PostAction` method for unbound actions. Otherwise fallback to Web API to select a default action under that controller.

#### Execute the Action
The `ODataDomainController.Get` method relies on the domain to execute the query and returns a response message with `EntityResult` or `EntityCollectionResult`.

{% highlight csharp %}
public async Task<HttpResponseMessage> Get(
    CancellationToken cancellationToken)
{
    ...
    IQueryable queryable = this.GetQuery(odataProperties);
    var result = await Domain.QueryAsync(new QueryRequest(queryable), cancellationToken);
    ...
    return this.CreateQueryResponse(result.Results.AsQueryable(), path.EdmType);
}
{% endhighlight %}

**Notably** there are also `Post`, `Patch` and `Put` method in the `ODataDomainController` class and they will be selected by Web API according to the request method.

#### Serialize the Result
Web API for OData will call the domain serializer provider in RESTier to serialize the result returned by the action. This is specified in the `ODataDomainFormattingAttribute` on the `ODataDomainController` class.

{% highlight csharp %}
[ODataDomainFormatting]
[ODataDomainExceptionFilter]
public abstract class ODataDomainController : ODataController { ... }

[AttributeUsage(AttributeTargets.Class, AllowMultiple = false, Inherited = true)]
public sealed class ODataDomainFormattingAttribute : Attribute, IControllerConfiguration
{
    public void Initialize(HttpControllerSettings controllerSettings, HttpControllerDescriptor controllerDescriptor)
    {
        ...
        odataFormatters = ODataMediaTypeFormatters.Create(
            new DefaultODataDomainSerializerProvider(),
            new DefaultODataDeserializerProvider());
        controllerFormatters.InsertRange(0, odataFormatters);
    }
}
{% endhighlight %}

The `ODataDomainFormattingAttribute.Initialize` method will be called when adding the controller to Web API. In this method, a pair of formatters are created and inserted to **the first place** of the controller formatter list. We use the default deserializer provider from Web API for OData and use the `DefaultODataDomainSerializerProvider` defined in RESTier as the serializer provider.

{% highlight csharp %}
public class DefaultODataDomainSerializerProvider : DefaultODataSerializerProvider
{
    ...
    public override ODataSerializer GetODataPayloadSerializer(IEdmModel model, Type type, HttpRequestMessage request)
    {
        ODataSerializer serializer = base.GetODataPayloadSerializer(model, type, request);

        if (serializer == null)
        {
            if (type == typeof(EntityCollectionResult))
            {
                serializer = this.feedSerializer;
            }
            else if (type == typeof(EntityResult))
            {
                serializer = this.entityTypeSerializer;
            }
        }

        return serializer;
    }
    ...
}
{% endhighlight %}

Basically the `DefaultODataDomainSerializerProvider` would select the `ODataDomainFeedSerializer` (or the `ODataDomainEntityTypeSerializer`) if the result is `EntityCollectionResult` (or `EntityResult`). We will not list the source of the `ODataDomainFeedSerializer` or the `ODataDomainEntityTypeSerializer` class here because they are almost simple wrappers of the corresponding serializers in Web API for OData. If you have interest, please take a look at them yourself.