---
layout: post
title: "3.4 Use Edm.Date in RESTier [>=0.3.0-beta2]"
description: ""
category: "3. Extensions"
---

RESTier now supports `Edm.Date` as property type. However the way of supporting is a little special compared to other primitive types because there is no straightforward CLR type mapped to `Edm.Date`. In Entity Framework, `System.DateTime` is chosen as the object type for the database type `DATE` along with a certain data annotation. To ensure compatibility, RESTier also follows this convention. In RESTier, the `Edm.Date` support covers most scenarios including query, serialization and deserialization though with a few limitations that will be mentioned below.

### Add Edm.Date property
Suppose you have an entity class `Person`, please add the following code to define an `Edm.Date` property `BirthDate`. The data annotation indicates that the property is mapped to a database column `BirthDate` whose database type is `Date`. Note that currently we **DO NOT** support the type to be `System.DateTime?` due to a limitation in OData Web API - note: OData v4 Web API 5.8 will support this - refer to issue [#482](https://github.com/OData/WebApi/issues/482).

{% highlight csharp %}
using System;
using System.ComponentModel.DataAnnotations.Schema;

public class Person
{
    [Column("BirthDate", TypeName = "Date")]
    public DateTime BirthDate { get; set; }
}
{% endhighlight %}

Actually you are all done to add an `Edm.Date` property! If you have the need to override `ODataPayloadValueConverter`, please now change to override `ODataDomainPayloadValueConverter ` (=0.3.0-beta2) or `RestierPayloadValueConverter` (>0.3.0-beta2) instead in order not to break the payload value conversion specialized for `Edm.Date`.

### Query Edm.Date property
Here are some examples:

{% highlight text %}
GET ~/People?$filter=BirthDate eq 1986-02-09
GET ~/People?$filter=year(BirthDate) eq 1986
{% endhighlight %}

In RESTier 0.3.0-beta2, please add the following code to your controller to support querying single property:

{% highlight csharp %}
public class TrippinController : ODataDomainController<TrippinDomain>
{
    [ODataRoute("People({key})/BirthDate")]
    [ODataRoute("People({key})/BirthDate/$value")]
    public Microsoft.OData.Edm.Library.Date GetPersonBirthDate([FromODataUri]int key)
    {
        return DbContext.People.Where(c => c.PersonId == key).Select(c => c.BirthDate).FirstOrDefault();
    }
}
{% endhighlight %}

Note that you **CANNOT** use `System.DateTime` as the return type of the controller action because in `EdmLib` world there is a fixed CLR type for `Edm.Date` called `Microsoft.OData.Edm.Library.Date`. The conversion from `System.DateTime` to `Microsoft.OData.Edm.Library.Date` is automatically done by an implicit conversion defined in `EdmLib`. 

There is also a good news that in the future versions (>0.3.0-beta2), you don't need to write the controller action or even the controller itself at all! The property access feature will be integrated into RESTier [#234](https://github.com/OData/RESTier/issues/234).
