---
layout: post
title: "4.3 RESTier Hook Handler (0.4.0)"
description: ""
category: "4. Deep in RESTier"
---

The hook handler in RESTier provides a way for setting custom extension points.

### Implement a single hook handler
Basically we have the following interface for all hook handlers
{% highlight csharp %}
public interface IHookHandler
{
}
{% endhighlight %}

For a specific hook handler, we will also provide the corresponding interface.
For example, for model builder, we have the following interface hook handler for building up a service model:
{% highlight csharp %}
public interface IModelBuilder : IHookHandler
{
 
	Task<IEdmModel> GetModelAsync(InvocationContext context, CancellationToken cancellationToken);
}
{% endhighlight %}

To support this hook handler, user could write his custom class and implement such interface.

### Implement a chained hook handler
Before 0.3.0-beta2, RESTier supports two kinds of hook (singleton hook and multi-cast hook point handler).
For singleton hook, users can specifiy one single hook implementation, and the invoker will call the hook once.
For multi-cast hook, users can add the hook implementation in sequence, while the invoker will call all the hooks in certain order (it may vary for different hook.)

But later we introduced the current hook mechanism, which allows the user to maintain the hook chain. This is also something called Matryoshka doll model.

If user wants to support hook hanlder chain, he could choose to implement the following interface besides the IHookHandler.  
{% highlight csharp %}
public interface IDelegateHookHandler<T> where T : IHookHandler
{
	
	T InnerHandler { get; set; }
}
{% endhighlight %}

For example, to support a chain of model producer, we can have the following class: 
{% highlight csharp %}
public class MyModelBuilder : IModelBuilder, IDelegateHookHandler<IModelBuilder>
{
	public IModelBuilder InnerHandler { get; set; }
	
	public async Task<IEdmModel> GetModelAsync(InvocationContext context, CancellationToken cancellationToken)
	{
		EdmModel model = null;
		if (this.InnerHandler != null)
		{
			model = await this.InnerHandler.GetModelAsync(context, cancellationToken) as EdmModel;
		}

		if (model == null)
		{
			// We don't plan to extend an empty model with operations.
			return null;
		}

		// some model extender
		return model;
	}
}
{% endhighlight %}

The InnnerHandler would be auto set to a previous hook handler if present, during the time this hook handler was added.

### Set a hook handler
Then we have got the following two APIs for setting hook handlers:

{% highlight csharp %}
public class ApiConfiguration 
{
	public ApiConfiguration AddHookHandler<T>(T handler) where T : class, IHookHandler
}
{% endhighlight %}

For each Api class instance, it will have a corresponding configuration. And we can set the hook handlers on the ApiConfiguration instance.

