---
layout: post
title: "4.4 RESTier API Service (0.5.0)"
description: ""
category: "4. Deep in RESTier"
---

Users can inject their custom API services into RESTier to extend various functionalities. There is a big progress since 0.4.0. Now the concept of **hook handler** has become **API service** in 0.5.0. We have removed the old interfaces `IHookHandler` and `IDelegateHookHandler` to adapt to the concept change. Thus the implementation of any custom API service (previously known as hook handler) should also be changed accordingly.

All API services registered as **one specific type** (either a class or an interface) are organized in a consistently chained (or nested) way. Each API service in a chain can choose whether to call the next (or inner) API service. The last API service registered is always invoked first.

As a practical example in RESTier, there is an API service interface called `IModelBuilder` to build or extend an EDM model. By default, RESTier will register two model builders for `IModelBuilder`. The model builder from the data provider (e.g., `ModelProducer` in RESTier EF) is always registered first. The `ConventionBasedApiModelBuilder` is always registered last. Any custom model builder will be registered sequentially between the two built-in model builders. When the API service `IModelBuilder` is invoked, the outermost `ConventionBasedApiModelBuilder` is always invoked first. It first invokes the inner API service which could possibly be the model builder from the data provider or some custom model builder (if any). The custom model builder can choose to extend the model returned from an inner builder, or otherwise it can simply choose not to call the inner one and directly return a new model. The model builder from the data provider is typically innermost and thus has no inner builder to call.

This subsection shows how to implement custom API services and inject them into RESTier.

### Implement an API service
The following sample code is to implement a custom model builder. Please note that if you want to call the inner builder, you need to put a **settable property** of `IModelBuilder` into your builder class. **The accessibility and the name of the property doesn't matter here**. Then try to call the inner builder in the service implementation. If you don't want to call any inner builder, you can just **omit the property** and remove the related logic.

{% highlight csharp %}
public class MyModelBuilder : IModelBuilder
{
    public IModelBuilder InnerBuilder { get; set; }

    public async Task<IEdmModel> GetModelAsync(InvocationContext context, CancellationToken cancellationToken)
    {
        IEdmModel model = null;
        if (this.InnerBuilder != null)
        {
            // Call the inner builder to build a model.
            model = await this.InnerBuilder.GetModelAsync(context, cancellationToken);
        }
        
        if (model != null)
        {
            // Do something to extend the model.
        }

        return model;
    }
}
{% endhighlight %}

### Register an API service
We need to register `MyModelBuilder` into the API to make it work. You can override the `ConfigureApi` method in your API class to do so. Here is the sample code. If you want to call the inner builder, you need to register it using `ChainPrevious` otherwise use `CutoffPrevious` instead. There are also overloads for the two methods that take an existing service instance or a service factory method. By the way, all those methods are fluent API so you can call them in a chained way.

{% highlight csharp %}
public class MyApi : ApiBase
{
    protected override IServiceCollection ConfigureApi(IServiceCollection services)
    {
        return base.ConfigureApi(services)
            .ChainPrevious<IModelBuilder, MyModelBuilder>();
    }
}
{% endhighlight %}

If you want to register a lambda expression (usually for advanced usage) as an API service, you can use the `AddContributor<T>` method. Here is an example. The parameter `sp` is of `IServiceProvider` and the parameter `next` is of `Func<T>` which represents the factory of the last API service registered. This means you can call `next()` to get the last API service instance.

{% highlight csharp %}
public class MyApi : ApiBase
{
    protected override IServiceCollection ConfigureApi(IServiceCollection services)
    {
        return base.ConfigureApi(services)
            .AddContributor<IModelBuilder>((sp, next) => /* return a model builder */);
    }
}
{% endhighlight %}

In the service implementation, the parameter type `IServiceCollection` is actually a container builder from Microsoft Dependency Injection Framework (DI). You can do whatever applicable to a normal DI container here. It is notable that you can also take advantage of the powerful **scope** feature in DI here! RESTier will create a new scope for each individual request in `ApiContext` which enables you to register scoped services whose lifetime is per-request.

Please visit [https://docs.asp.net/en/latest/fundamentals/dependency-injection.html](https://docs.asp.net/en/latest/fundamentals/dependency-injection.html) to grasp some basic understanding about DI before proceeding.

The following example is to register a scoped `MyDbContext` service so that you have a new `MyDbContext` instance for each request.

{% highlight csharp %}
public class MyDbContext : DbContext {...}

public class MyApi : ApiBase
{
    protected override IServiceCollection ConfigureApi(IServiceCollection services)
    {
        return base.ConfigureApi(services)
            .ChainPrevious<IModelBuilder, MyModelBuilder>()
            .AddScoped<MyDbContext>(sp => sp.GetService<T>());
    }
}
{% endhighlight %}

You can also make a specific API service singleton, scoped or transient (though not common) by calling `MakeSingleton`, `MakeScoped` or `MakeTransient`. Here is a sample which is to make `IModelBuilder` scoped.

{% highlight csharp %}
public class MyApi : ApiBase
{
    protected override IServiceCollection ConfigureApi(IServiceCollection services)
    {
        return base.ConfigureApi(services)
            .ChainPrevious<IModelBuilder, MyModelBuilder>()
            .MakeScoped<IModelBuilder>();
    }
}
{% endhighlight %}

### Get an API service
No matter in which way you register an API service of `T`, the only and unified way to get that service out is to use `(ApiContext|ApiConfiguration).GetApiService<T>` from RESTier or `IServiceProvider.GetService<T>` from DI. Please choose a proper one based on the current code context.