﻿// Copyright (c) Microsoft.  All Rights Reserved.  Licensed under the Apache License, Version 2.0.  See License.txt in the project root for license information.

using System.Collections.Generic;
using System.Collections.Immutable;
using System.Composition;
using System.Linq;
using System.Threading;
using Microsoft.CodeAnalysis.CSharp.CodeStyle;
using Microsoft.CodeAnalysis.Execution;
using Microsoft.CodeAnalysis.Host.Mef;
using Microsoft.CodeAnalysis.Options;
using Roslyn.Utilities;

namespace Microsoft.CodeAnalysis.CSharp.Execution
{
    [ExportLanguageService(typeof(IOptionsSerializationService), LanguageNames.CSharp), Shared]
    internal class CSharpOptionsSerializationService : AbstractOptionsSerializationService
    {
        public override void WriteTo(CompilationOptions options, ObjectWriter writer, CancellationToken cancellationToken)
        {
            WriteCompilationOptionsTo(options, writer, cancellationToken);

            var csharpOptions = (CSharpCompilationOptions)options;
            writer.WriteValue(csharpOptions.Usings.ToArray());
            writer.WriteBoolean(csharpOptions.AllowUnsafe);
        }

        public override void WriteTo(ParseOptions options, ObjectWriter writer, CancellationToken cancellationToken)
        {
            WriteParseOptionsTo(options, writer, cancellationToken);

            var csharpOptions = (CSharpParseOptions)options;
            writer.WriteInt32((int)csharpOptions.LanguageVersion);
            writer.WriteValue(options.PreprocessorSymbolNames.ToArray());
        }

        public override void WriteTo(OptionSet options, ObjectWriter writer, CancellationToken cancellationToken)
        {
            WriteOptionSetTo(options, LanguageNames.CSharp, writer, cancellationToken);

            foreach (var option in CSharpCodeStyleOptions.GetCodeStyleOptions())
            {
                WriteOptionTo(options, option, writer, cancellationToken);
            }
        }

        public override OptionSet ReadOptionSetFrom(ObjectReader reader, CancellationToken cancellationToken)
        {
            OptionSet options = new SerializedPartialOptionSet();

            options = ReadOptionSetFrom(options, LanguageNames.CSharp, reader, cancellationToken);

            foreach (var option in CSharpCodeStyleOptions.GetCodeStyleOptions())
            {
                options = ReadOptionFrom(options, option, reader, cancellationToken);
            }

            return options;
        }

        public override CompilationOptions ReadCompilationOptionsFrom(ObjectReader reader, CancellationToken cancellationToken)
        {
            ReadCompilationOptionsFrom(
                reader,
                out var outputKind, out var reportSuppressedDiagnostics, out var moduleName, out var mainTypeName, out var scriptClassName,
                out var optimizationLevel, out var checkOverflow, out var cryptoKeyContainer, out var cryptoKeyFile, out var cryptoPublicKey,
                out var delaySign, out var platform, out var generalDiagnosticOption, out var warningLevel, out var specificDiagnosticOptions,
                out var concurrentBuild, out var deterministic, out var publicSign, out var xmlReferenceResolver, out var sourceReferenceResolver,
                out var metadataReferenceResolver, out var assemblyIdentityComparer, out var strongNameProvider, cancellationToken);

            var usings = reader.ReadArray<string>();
            var allowUnsafe = reader.ReadBoolean();

            return new CSharpCompilationOptions(
                outputKind, reportSuppressedDiagnostics, moduleName, mainTypeName, scriptClassName, usings, optimizationLevel, checkOverflow, allowUnsafe,
                cryptoKeyContainer, cryptoKeyFile, cryptoPublicKey, delaySign, platform, generalDiagnosticOption, warningLevel, specificDiagnosticOptions, concurrentBuild,
                deterministic, xmlReferenceResolver, sourceReferenceResolver, metadataReferenceResolver, assemblyIdentityComparer, strongNameProvider, publicSign);
        }

        public override ParseOptions ReadParseOptionsFrom(ObjectReader reader, CancellationToken cancellationToken)
        {
            ReadParseOptionsFrom(reader, out var kind, out var documentationMode, out var features, cancellationToken);

            var languageVersion = (LanguageVersion)reader.ReadInt32();
            var preprocessorSymbolNames = reader.ReadArray<string>();

            var options = new CSharpParseOptions(languageVersion, documentationMode, kind);

            // use WithPreprocessorSymbols instead of constructor to bypass preprocessor validation.
            // https://github.com/dotnet/roslyn/issues/15797
            return options.WithPreprocessorSymbols(preprocessorSymbolNames)
                          .WithFeatures(features);
        }
    }
}
