﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Xunit;
using Xunit.Abstractions;

namespace Orleans.Transactions.Tests
{
    public abstract class GrainFaultTransactionTestRunner : TransactionTestRunnerBase
    {
        public GrainFaultTransactionTestRunner(IGrainFactory grainFactory, ITestOutputHelper output)
        : base(grainFactory, output)
        { }

        [SkippableTheory]
        [InlineData(TransactionTestConstants.SingleStateTransactionalGrain)]
        public async Task AbortTransactionOnExceptions(string transactionTestGrainClassName)
        {
            const int expected = 5;

            ITransactionTestGrain grain = RandomTestGrain(transactionTestGrainClassName);
            ITransactionCoordinatorGrain coordinator = this.grainFactory.GetGrain<ITransactionCoordinatorGrain>(Guid.NewGuid());

            await coordinator.MultiGrainSet(new List<ITransactionTestGrain> { grain }, expected);
            await Assert.ThrowsAsync<OrleansTransactionAbortedException>(() => coordinator.AddAndThrow(grain, expected));

            await TestAfterDustSettles(async () =>
            {
                int actual = await grain.Get();
                Assert.Equal(expected, actual);
            });
        }

        [SkippableTheory]
        [InlineData(TransactionTestConstants.SingleStateTransactionalGrain)]
        public async Task MultiGrainAbortTransactionOnExceptions(string transactionTestGrainClassName)
        {
            const int grainCount = TransactionTestConstants.MaxCoordinatedTransactions - 1;
            const int expected = 5;

            ITransactionTestGrain throwGrain = RandomTestGrain(transactionTestGrainClassName);
            List<ITransactionTestGrain> grains =
                Enumerable.Range(0, grainCount)
                    .Select(i => RandomTestGrain(transactionTestGrainClassName))
                    .ToList();
            ITransactionCoordinatorGrain coordinator = this.grainFactory.GetGrain<ITransactionCoordinatorGrain>(Guid.NewGuid());

            await throwGrain.Set(expected);
            await coordinator.MultiGrainSet(grains, expected);
            await Assert.ThrowsAsync<OrleansTransactionAbortedException>(() => coordinator.MultiGrainAddAndThrow(throwGrain, grains, expected));

            grains.Add(throwGrain);

            await TestAfterDustSettles(async () =>
            {
                foreach (var grain in grains)
                {
                    int actual = await grain.Get();
                    Assert.Equal(expected, actual);
                }
            });
        }

        [SkippableTheory(Skip = "Intermittent failure, jbragg investigating")]
        [InlineData(TransactionTestConstants.SingleStateTransactionalGrain)]
        public async Task AbortTransactionOnOrphanCalls(string transactionTestGrainClassName)
        {
            const int expected = 5;

            ITransactionTestGrain grain = RandomTestGrain(transactionTestGrainClassName);
            ITransactionCoordinatorGrain coordinator = this.grainFactory.GetGrain<ITransactionCoordinatorGrain>(Guid.NewGuid());

            await grain.Set(expected);
            await Assert.ThrowsAsync<OrleansOrphanCallException>(() => coordinator.OrphanCallTransaction(grain));

            //await Task.Delay(20000); // give time for GC

            await TestAfterDustSettles(async () =>
            {
                int actual = await grain.Get();
                Assert.Equal(expected, actual);
            });
        }

        private async Task TestAfterDustSettles(Func<Task> what)
        {
            int tries = 2;
            while (tries-- > 0)
            {
                try
                {
                    await what();
                }
                catch (OrleansCascadingAbortException)
                {
                    // due to optimistic reading we may read state of aborted transactions
                    // which causes cascading abort
                }
            }
        }
    }
}
