// Copyright 2014 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.devtools.build.lib.skyframe;

import com.google.devtools.build.lib.util.io.TimestampGranularityMonitor;
import com.google.devtools.build.lib.vfs.RootedPath;
import com.google.devtools.build.skyframe.SkyFunction;
import com.google.devtools.build.skyframe.SkyFunctionException;
import com.google.devtools.build.skyframe.SkyKey;
import com.google.devtools.build.skyframe.SkyValue;

import java.io.IOException;

/**
 * A {@link SkyFunction} for {@link FileStateValue}s.
 *
 * <p>Merely calls FileStateValue#create, but also has special handling for files outside the
 * package roots (see {@link ExternalFilesHelper}).
 */
public class FileStateFunction implements SkyFunction {

  private final TimestampGranularityMonitor tsgm;
  private final ExternalFilesHelper externalFilesHelper;

  public FileStateFunction(TimestampGranularityMonitor tsgm,
      ExternalFilesHelper externalFilesHelper) {
    this.tsgm = tsgm;
    this.externalFilesHelper = externalFilesHelper;
  }

  @Override
  public SkyValue compute(SkyKey skyKey, Environment env) throws FileStateFunctionException {
    RootedPath rootedPath = (RootedPath) skyKey.argument();

    try {
      externalFilesHelper.maybeHandleExternalFile(rootedPath, env);
      if (env.valuesMissing()) {
        return null;
      }
      return FileStateValue.create(rootedPath, tsgm);
    } catch (FileOutsidePackageRootsException e) {
      throw new FileStateFunctionException(e);
    } catch (IOException e) {
      throw new FileStateFunctionException(e);
    } catch (InconsistentFilesystemException e) {
      throw new FileStateFunctionException(e);
    }
  }

  @Override
  public String extractTag(SkyKey skyKey) {
    return null;
  }

  /**
   * Used to declare all the exception types that can be wrapped in the exception thrown by
   * {@link FileStateFunction#compute}.
   */
  private static final class FileStateFunctionException extends SkyFunctionException {
    public FileStateFunctionException(IOException e) {
      super(e, Transience.TRANSIENT);
    }

    public FileStateFunctionException(InconsistentFilesystemException e) {
      super(e, Transience.TRANSIENT);
    }

    public FileStateFunctionException(FileOutsidePackageRootsException e) {
      super(e, Transience.PERSISTENT);
    }
  }
}
