# Commands
Commands are represented as an array of `Command` objects. A `Command` object must at least have an `id`,`label` and `action` property. Here is an example:

::alert{type="info"}
It is **important** to have a **unique** `id` for each command.
::

```vue
<script setup lang="ts">
import { type Command, CmdBar } from 'cmd-bar'
  
const commands: Command[] = [
  {
    key: '1',
    label: 'Command 1',
    action: () => {
      console.log('Command 1')
    },
  },
  {
    key: '2',
    label: 'Command 2',
    action: () => {}
  },
  {
    key: '3',
    label: 'Command 3',
    action: () => {},
  },
]
</script>

<template>
  <CmdBar :groups="[{key: 'people', commands: 'people'}]">
    <CmdBarDialog>
      <template #header>
        <CmdBarInput placeholder="search for anything />
      </template>
      <template #content>
        <CmdBarList>
          <template #default="{ command }">
            <div class="leading">
              <img :src="command.leading" alt="icon" />
              {{ command.label }}
            </div>
          </template>
      </CmdBarList>
    </template>
  </CmdBarDialog>
</template>
```

::alert{type="info"}
You can add any property to your commands and just render them as you wish.
::

## DefineCommand

`defineCommand()` is a helper function to get type safety, that takes an object and returns a `Command` object.

Here is an example of fetching commands from a remote API:

```ts
const users = ref<Command[]>([])

async function fetchUsers() {
  const { data } = await useFetch(
    'https://dummyjson.com/users?limit=10&select=id,firstName,lastName'
  ).json()
  
  users.value = data.value.users.map((user: Record<string, any>) => {
    return defineCommand({
      key: 'user-' + user.id.toString(),
      leading: './src/assets/icons/user_new.svg',
      label: `${user.firstName} ${user.lastName}`,
      action: () => {
        // Define your action here.
      }
    })
  })
}
```

