package io.quarkus.devtools.commands.handlers;

import static io.quarkus.devtools.commands.handlers.QuarkusCommandHandlers.select;
import static java.util.Arrays.asList;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

import io.quarkus.devtools.commands.data.SelectionResult;
import io.quarkus.maven.ArtifactCoords;
import io.quarkus.registry.catalog.Extension;

class QuarkusCommandHandlersTest {

    @Test
    void testMultiMatchByLabels() {
        Extension e1 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "e1", "1.0"))
                .setName("some extension 1")
                .setMetadata(Extension.MD_KEYWORDS, Arrays.asList("foo", "bar"));

        Extension e2 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "e2", "1.0"))
                .setName("some extension 2")
                .setMetadata(Extension.MD_KEYWORDS, Arrays.asList("foo", "bar", "baz"));

        Extension e3 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "e3", "1.0"))
                .setName("unrelated")
                .setMetadata(Extension.MD_KEYWORDS, Arrays.asList("bar"));

        List<Extension> extensions = asList(e1, e2, e3);
        Collections.shuffle(extensions);
        SelectionResult matches = select("foo", extensions, true);
        Assertions.assertFalse(matches.matches());
        Assertions.assertEquals(2, matches.getExtensions().size());

        matches = select("foo", extensions, false);
        Assertions.assertFalse(matches.matches());
        Assertions.assertEquals(0, matches.getExtensions().size());
    }

    @Test
    void testThatSingleLabelMatchIsNotAMatch() {
        Extension e1 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "e1", "1.0"))
                .setName("e1")
                .setMetadata(Extension.MD_KEYWORDS, Arrays.asList("foo", "bar"));

        Extension e2 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "e2", "1.0"))
                .setName("e2")
                .setMetadata(Extension.MD_KEYWORDS, Arrays.asList("bar", "baz"));

        List<Extension> extensions = asList(e1, e2);
        Collections.shuffle(extensions);
        SelectionResult matches = select("foo", extensions, true);
        Assertions.assertFalse(matches.matches());
        Assertions.assertEquals(1, matches.getExtensions().size());
    }

    @Test
    void testMultiMatchByArtifactIdsAndNames() {
        Extension e1 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "e1", "1.0"))
                .setName("foo")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo", "bar"));
        Extension e2 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "quarkus-foo", "1.0"))
                .setName("some foo bar")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo", "bar", "baz"));
        Extension e3 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "e3", "1.0"))
                .setName("unrelated")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo"));

        List<Extension> extensions = asList(e1, e2, e3);
        Collections.shuffle(extensions);
        SelectionResult matches = select("foo", extensions, false);
        Assertions.assertFalse(matches.matches(), " " + matches.getExtensions().size());
        Assertions.assertEquals(2, matches.getExtensions().size());

        matches = select("foo", extensions, true);
        Assertions.assertFalse(matches.matches());
        Assertions.assertEquals(3, matches.getExtensions().size());

    }

    @Test
    void testShortNameSelection() {
        Extension e1 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "some-complex-seo-unaware-artifactid", "1.0"))
                .setName("some complex seo unaware name")
                .setMetadata(Extension.MD_SHORT_NAME, "foo")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo", "bar"));
        Extension e2 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "some-foo-bar", "1.0"))
                .setName("some foo bar")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo", "bar", "baz"));
        Extension e3 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "unrelated", "1.0"))
                .setName("unrelated")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo"));

        List<Extension> extensions = asList(e1, e2, e3);
        Collections.shuffle(extensions);
        SelectionResult matches = select("foo", extensions, false);
        Assertions.assertTrue(matches.matches());
        Assertions.assertEquals(1, matches.getExtensions().size());
        Assertions.assertTrue(matches.iterator().hasNext());
        Assertions
                .assertTrue(matches.iterator().next().getArtifact().getArtifactId()
                        .equalsIgnoreCase("some-complex-seo-unaware-artifactid"));
    }

    @Test
    void testArtifactIdSelectionWithQuarkusPrefix() {
        Extension e1 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "quarkus-foo", "1.0"))
                .setName("some complex seo unaware name")
                .setMetadata(Extension.MD_SHORT_NAME, "foo")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo", "bar"));
        Extension e2 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "quarkus-foo-bar", "1.0"))
                .setName("some foo bar")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo", "bar", "baz"));
        Extension e3 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "quarkus-unrelated", "1.0"))
                .setName("unrelated")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo"));

        List<Extension> extensions = asList(e1, e2, e3);
        Collections.shuffle(extensions);
        SelectionResult matches = select("foo", extensions, false);
        Assertions.assertEquals(1, matches.getExtensions().size());
        Assertions.assertTrue(matches.iterator().hasNext());
        Assertions.assertTrue(matches.iterator().next().getArtifact().getArtifactId().equalsIgnoreCase("quarkus-foo"));
    }

    @Test
    void testListedVsUnlisted() {
        Extension e1 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "quarkus-foo-unlisted", "1.0"))
                .setName("some complex seo unaware name")
                .setMetadata(Extension.MD_SHORT_NAME, "foo")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo", "bar"))
                .setMetadata("unlisted", "true");

        Extension e2 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "quarkus-foo-bar", "1.0"))
                .setName("some foo bar")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo", "bar", "baz"))
                .setMetadata("unlisted", "false");

        Extension e3 = Extension.builder()
                .setArtifact(new ArtifactCoords("org.acme", "quarkus-foo-baz", "1.0"))
                .setName("unrelated")
                .setMetadata(Extension.MD_KEYWORDS, asList("foo"));

        List<Extension> extensions = asList(e1, e2, e3);
        Collections.shuffle(extensions);
        SelectionResult matches = select("quarkus-foo", extensions, true);
        Assertions.assertEquals(2, matches.getExtensions().size());

        matches = select("quarkus-foo-unlisted", extensions, true);
        Assertions.assertEquals(1, matches.getExtensions().size());

    }
}
