package io.quarkus.flyway.test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import jakarta.inject.Inject;

import org.flywaydb.core.Flyway;
import org.flywaydb.core.api.MigrationVersion;
import org.flywaydb.core.api.migration.Context;
import org.flywaydb.core.api.migration.JavaMigration;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

import db.migration.V1_0_1__Update;
import db.migration.V1_0_2__Update;
import io.agroal.api.AgroalDataSource;
import io.quarkus.test.QuarkusUnitTest;

public class FlywayExtensionCleanAndMigrateAtStartWithJavaMigrationTest {

    @Inject
    Flyway flyway;

    @Inject
    AgroalDataSource defaultDataSource;

    @RegisterExtension
    static final QuarkusUnitTest config = new QuarkusUnitTest()
            .withApplicationRoot((jar) -> jar
                    .addClasses(V1_0_1__Update.class, V1_0_2__Update.class, V9_9_9__Update.class)
                    .addAsResource("db/migration/V1.0.0__Quarkus.sql")
                    .addAsResource("clean-and-migrate-at-start-config.properties", "application.properties"));

    @Test
    @DisplayName("Clean and migrate at start correctly")
    public void testFlywayConfigInjection() throws SQLException {

        try (Connection connection = defaultDataSource.getConnection(); Statement stat = connection.createStatement()) {
            try (ResultSet countQuery = stat.executeQuery("select count(1) from quarked_flyway")) {
                assertTrue(countQuery.first());
                assertEquals(2,
                        countQuery.getInt(1),
                        "Table 'quarked_flyway' does not contain the expected number of rows");
            }
        }
        String currentVersion = flyway.info().current().getVersion().toString();
        assertEquals("1.0.2", currentVersion, "Expected to be 1.0.2 as there is a SQL and two Java migration scripts");
    }

    public static class V9_9_9__Update implements JavaMigration {
        @Override
        public MigrationVersion getVersion() {
            return MigrationVersion.fromVersion("9.9.9");
        }

        @Override
        public String getDescription() {
            return getClass().getSimpleName();
        }

        @Override
        public Integer getChecksum() {
            return null;
        }

        @Override
        public boolean canExecuteInTransaction() {
            return true;
        }

        @Override
        public void migrate(Context context) throws Exception {
            try (Statement statement = context.getConnection().createStatement()) {
                statement.executeUpdate("INSERT INTO quarked_flyway VALUES (9999, 'should-not-be-added')");
            }
        }
    }

}
