"Provide access to a BSD tar"

load(":utf8_environment.bzl", "Utf8EnvironmentInfo")

BSDTAR_PLATFORMS = {
    "darwin_amd64": struct(
        compatible_with = [
            "@platforms//os:osx",
            "@platforms//cpu:x86_64",
        ],
    ),
    "darwin_arm64": struct(
        compatible_with = [
            "@platforms//os:osx",
            "@platforms//cpu:aarch64",
        ],
    ),
    "linux_amd64": struct(
        compatible_with = [
            "@platforms//os:linux",
            "@platforms//cpu:x86_64",
        ],
    ),
    "linux_arm64": struct(
        compatible_with = [
            "@platforms//os:linux",
            "@platforms//cpu:aarch64",
        ],
    ),
    "windows_amd64": struct(
        compatible_with = [
            "@platforms//os:windows",
            "@platforms//cpu:x86_64",
        ],
    ),
    "windows_arm64": struct(
        compatible_with = [
            "@platforms//os:windows",
            "@platforms//cpu:aarch64",
        ],
    ),
}

BSDTAR_PREBUILT = {
    "darwin_amd64": (
        "https://github.com/aspect-build/bsdtar-prebuilt/releases/download/v3.8.1/tar_darwin_amd64",
        "921118f1d043aff08a8d7d7b477217781efb9dacad111646add724bc51575c6a",
    ),
    "darwin_arm64": (
        "https://github.com/aspect-build/bsdtar-prebuilt/releases/download/v3.8.1/tar_darwin_arm64",
        "9e78a0b3e21bc05c67e54004e5b29c2b19c3a9f16ccec4de2a227b1e01aea5fd",
    ),
    "linux_amd64": (
        "https://github.com/aspect-build/bsdtar-prebuilt/releases/download/v3.8.1/tar_linux_amd64",
        "a703af6fc8df1a89f1ca864c651a9003b75069dd6b80bd32dcd94a7d255df07d",
    ),
    "linux_arm64": (
        "https://github.com/aspect-build/bsdtar-prebuilt/releases/download/v3.8.1/tar_linux_arm64",
        "663f498baab2a9b7758e46d0c377b311c5b058758a37958372a0503c5dda4028",
    ),
    "windows_arm64": (
        "https://github.com/aspect-build/bsdtar-prebuilt/releases/download/v3.8.1-fix.1/tar_windows_arm64.exe",
        "130d69268b0a387bca387d00663821779dd1915557caf7fcbfd5ded3f41074f3",
    ),
    "windows_amd64": (
        "https://github.com/aspect-build/bsdtar-prebuilt/releases/download/v3.8.1-fix.1/tar_windows_x86_64.exe",
        "f48c81e1812956adb4906c6f057ca856dd280a455e7867d77800e6d5ef9fc81d",
    ),
}

def _bsdtar_binary_repo(rctx):
    (url, sha256) = BSDTAR_PREBUILT[rctx.attr.platform]
    binary = "tar.exe" if rctx.attr.platform.startswith("windows") else "tar"
    rctx.download(
        url = url,
        output = binary,
        executable = True,
        sha256 = sha256,
    )

    rctx.file("BUILD.bazel", """\
# @generated by @aspect_bazel_lib//lib/private:tar_toolchain.bzl

load("@aspect_bazel_lib//lib/private:tar_toolchain.bzl", "tar_toolchain")

package(default_visibility = ["//visibility:public"])

tar_toolchain(name = "bsdtar_toolchain", binary = "{}")
""".format(binary))

bsdtar_binary_repo = repository_rule(
    implementation = _bsdtar_binary_repo,
    attrs = {
        "platform": attr.string(mandatory = True, values = BSDTAR_PLATFORMS.keys()),
    },
)

TarInfo = provider(
    doc = "Provide info for executing BSD tar",
    fields = {
        # environment appears on the toolchain since it's platform-specific, and we want to pair it with the tool.
        # Currently known to be required only for the extract mode; see
        # https://github.com/bazelbuild/bazel-central-registry/issues/2256
        "default_env": "environment variables which should be set when spawning tar, to ensure reproducible results",
        "binary": "tar executable",
    },
)

def _tar_toolchain_impl(ctx):
    binary = ctx.executable.binary

    # Make the $(BSDTAR_BIN) variable available in places like genrules.
    # See https://docs.bazel.build/versions/main/be/make-variables.html#custom_variables
    template_variables = platform_common.TemplateVariableInfo({
        "BSDTAR_BIN": binary.path,
    })

    default_info = DefaultInfo(
        files = depset(ctx.files.binary + ctx.files.files),
    )
    tarinfo = TarInfo(
        default_env = ctx.attr._utf8_environment[Utf8EnvironmentInfo].environment,
        binary = binary,
    )

    # Export all the providers inside our ToolchainInfo
    # so the resolved_toolchain rule can grab and re-export them.
    toolchain_info = platform_common.ToolchainInfo(
        tarinfo = tarinfo,
        template_variables = template_variables,
        default = default_info,
    )

    return [toolchain_info, template_variables, default_info]

tar_toolchain = rule(
    implementation = _tar_toolchain_impl,
    attrs = {
        "binary": attr.label(
            doc = "a command to find on the system path",
            allow_files = True,
            executable = True,
            cfg = "exec",
        ),
        "_utf8_environment": attr.label(
            default = ":utf8_environment",
            cfg = "exec",
        ),
        "files": attr.label_list(allow_files = True),
    },
)

def _tar_toolchains_repo_impl(rctx):
    # Expose a concrete toolchain which is the result of Bazel resolving the toolchain
    # for the execution or target platform.
    # Workaround for https://github.com/bazelbuild/bazel/issues/14009
    starlark_content = """\
# @generated by @aspect_bazel_lib//lib/private:tar_toolchain.bzl

# Forward all the providers
def _resolved_toolchain_impl(ctx):
    toolchain_info = ctx.toolchains["@aspect_bazel_lib//lib:tar_toolchain_type"]
    return [
        toolchain_info,
        toolchain_info.default,
        toolchain_info.tarinfo,
        toolchain_info.template_variables,
    ]

# Copied from java_toolchain_alias
# https://cs.opensource.google/bazel/bazel/+/master:tools/jdk/java_toolchain_alias.bzl
resolved_toolchain = rule(
    implementation = _resolved_toolchain_impl,
    toolchains = ["@aspect_bazel_lib//lib:tar_toolchain_type"],
)
"""
    rctx.file("defs.bzl", starlark_content)

    build_content = """# @generated by @aspect_bazel_lib//lib/private:tar_toolchain.bzl
load(":defs.bzl", "resolved_toolchain")

resolved_toolchain(name = "resolved_toolchain", visibility = ["//visibility:public"])"""

    for [platform, meta] in BSDTAR_PLATFORMS.items():
        build_content += """
toolchain(
    name = "{platform}_toolchain",
    exec_compatible_with = {compatible_with},
    toolchain = "@{user_repository_name}_{platform}//:bsdtar_toolchain",
    toolchain_type = "@aspect_bazel_lib//lib:tar_toolchain_type",
)
""".format(
            platform = platform,
            user_repository_name = rctx.attr.user_repository_name,
            compatible_with = meta.compatible_with,
        )

    rctx.file("BUILD.bazel", build_content)

tar_toolchains_repo = repository_rule(
    _tar_toolchains_repo_impl,
    doc = """Creates a repository that exposes a tar_toolchain_type target.""",
    attrs = {
        "user_repository_name": attr.string(doc = "Base name for toolchains repository"),
    },
)
