"Provide access to ZSTD"

ZSTD_PLATFORMS = {
    "darwin_amd64": struct(
        compatible_with = [
            "@platforms//os:osx",
            "@platforms//cpu:x86_64",
        ],
    ),
    "darwin_arm64": struct(
        compatible_with = [
            "@platforms//os:osx",
            "@platforms//cpu:aarch64",
        ],
    ),
    "linux_amd64": struct(
        compatible_with = [
            "@platforms//os:linux",
            "@platforms//cpu:x86_64",
        ],
    ),
    "linux_arm64": struct(
        compatible_with = [
            "@platforms//os:linux",
            "@platforms//cpu:aarch64",
        ],
    ),
    "windows_amd64": struct(
        compatible_with = [
            "@platforms//os:windows",
            "@platforms//cpu:x86_64",
        ],
    ),
}

ZSTD_PREBUILT = {
    "darwin_amd64": (
        "https://github.com/aspect-build/zstd-prebuilt/releases/download/v1.5.6-bcr1/zstd_darwin_amd64",
        "E4D517212005CF26F8B8D657455D1380318B071CB52A3FFD9DFBDF4C2BA71A13",
    ),
    "darwin_arm64": (
        "https://github.com/aspect-build/zstd-prebuilt/releases/download/v1.5.6-bcr1/zstd_darwin_arm64",
        "6E210EEAE08FB6BA38C3AC2D1857075C28113AEF68296F7E396F1180F7E894B9",
    ),
    "linux_amd64": (
        "https://github.com/aspect-build/zstd-prebuilt/releases/download/v1.5.6-bcr1/zstd_linux_amd64",
        "0F0BD1193509A598629D7FA745C4B0B6D5FA6719E0C94C01EF0F20E466D801A7",
    ),
    "linux_arm64": (
        "https://github.com/aspect-build/zstd-prebuilt/releases/download/v1.5.6-bcr1/zstd_linux_arm64",
        "82AACF8F1C67FF3C94E04AFB0721A848BBBA70FBF8249EE4BC4C9085AFB84548",
    ),
    "windows_amd64": (
        "https://github.com/aspect-build/zstd-prebuilt/releases/download/v1.5.6-bcr1/zstd_windows_x86_64.exe",
        "A944FDE13D01892C7CF0970FE4271F9561373F5E0A46ECEA756B6AE9A31E056C",
    ),
    "windows_arm64": (
        "https://github.com/aspect-build/zstd-prebuilt/releases/download/v1.5.6-bcr1/zstd_windows_arm64.exe",
        "8CA462473BCCD13C9D9A952F09E5F628153785323949D539E8858B2ABB85E246",
    ),
}

def _zstd_binary_repo(rctx):
    (url, sha256) = ZSTD_PREBUILT[rctx.attr.platform]
    rctx.download(
        url = url,
        output = "zstd",
        executable = True,
        sha256 = sha256,
    )
    binary = "zstd"

    rctx.file("BUILD.bazel", """\
# @generated by @aspect_bazel_lib//lib/private:zstd_toolchain.bzl

load("@aspect_bazel_lib//lib/private:zstd_toolchain.bzl", "zstd_toolchain")

package(default_visibility = ["//visibility:public"])

zstd_toolchain(name = "zstd_toolchain", binary = "{}")
""".format(binary))

zstd_binary_repo = repository_rule(
    implementation = _zstd_binary_repo,
    attrs = {
        "platform": attr.string(mandatory = True, values = ZSTD_PLATFORMS.keys()),
    },
)

ZstdInfo = provider(
    doc = "Provide info for executing zstd",
    fields = {
        "binary": "zstd executable",
    },
)

def _zstd_toolchain_impl(ctx):
    binary = ctx.executable.binary

    # Make the $(ZSTD_BIN) variable available in places like genrules.
    # See https://docs.bazel.build/versions/main/be/make-variables.html#custom_variables
    template_variables = platform_common.TemplateVariableInfo({
        "ZSTD_BIN": binary.path,
    })

    default_info = DefaultInfo(
        files = depset(ctx.files.binary + ctx.files.files),
    )
    zstdinfo = ZstdInfo(
        binary = binary,
    )

    # Export all the providers inside our ToolchainInfo
    # so the resolved_toolchain rule can grab and re-export them.
    toolchain_info = platform_common.ToolchainInfo(
        zstdinfo = zstdinfo,
        template_variables = template_variables,
        default = default_info,
    )

    return [toolchain_info, template_variables, default_info]

zstd_toolchain = rule(
    implementation = _zstd_toolchain_impl,
    attrs = {
        "binary": attr.label(
            doc = "a command to find on the system path",
            allow_files = True,
            executable = True,
            cfg = "exec",
        ),
        "files": attr.label_list(allow_files = True),
    },
)

def _zstd_toolchains_repo_impl(rctx):
    # Expose a concrete toolchain which is the result of Bazel resolving the toolchain
    # for the execution or zstdget platform.
    # Workaround for https://github.com/bazelbuild/bazel/issues/14009
    starlark_content = """\
# @generated by @aspect_bazel_lib//lib/private:zstd_toolchain.bzl

# Forward all the providers
def _resolved_toolchain_impl(ctx):
    toolchain_info = ctx.toolchains["@aspect_bazel_lib//lib:zstd_toolchain_type"]
    return [
        toolchain_info,
        toolchain_info.default,
        toolchain_info.zstdinfo,
        toolchain_info.template_variables,
    ]

# Copied from java_toolchain_alias
# https://cs.opensource.google/bazel/bazel/+/master:tools/jdk/java_toolchain_alias.bzl
resolved_toolchain = rule(
    implementation = _resolved_toolchain_impl,
    toolchains = ["@aspect_bazel_lib//lib:zstd_toolchain_type"],
)
"""
    rctx.file("defs.bzl", starlark_content)

    build_content = """# @generated by @aspect_bazel_lib//lib/private:zstd_toolchain.bzl
load(":defs.bzl", "resolved_toolchain")

resolved_toolchain(name = "resolved_toolchain", visibility = ["//visibility:public"])"""

    for [platform, meta] in ZSTD_PLATFORMS.items():
        build_content += """
toolchain(
    name = "{platform}_toolchain",
    exec_compatible_with = {compatible_with},
    toolchain = "@{user_repository_name}_{platform}//:zstd_toolchain",
    toolchain_type = "@aspect_bazel_lib//lib:zstd_toolchain_type",
)
""".format(
            platform = platform,
            user_repository_name = rctx.attr.user_repository_name,
            compatible_with = meta.compatible_with,
        )

    rctx.file("BUILD.bazel", build_content)

zstd_toolchains_repo = repository_rule(
    _zstd_toolchains_repo_impl,
    doc = """Creates a repository that exposes a zstd_toolchain_type zstdget.""",
    attrs = {
        "user_repository_name": attr.string(doc = "Base name for toolchains repository"),
    },
)
