"""Repository rules for fetching cosign"""

load("//cosign/private:versions.bzl", "COSIGN_VERSIONS")

# buildifier: disable=bzl-visibility
load("//oci/private:toolchains_repo.bzl", "PLATFORMS", "toolchains_repo")

COSIGN_BUILD_TMPL = """\
# Generated by container/repositories.bzl
load("@rules_oci//cosign:toolchain.bzl", "cosign_toolchain")
cosign_toolchain(
    name = "cosign_toolchain", 
    cosign = "cosign"
)
"""

def _cosign_repo_impl(repository_ctx):
    platform = repository_ctx.attr.platform.replace("x86_64", "amd64").replace("_", "-")
    url = "https://github.com/sigstore/cosign/releases/download/{version}/cosign-{platform}".format(
        version = repository_ctx.attr.cosign_version,
        platform = platform,
    )
    repository_ctx.download(
        url = url,
        output = "cosign",
        executable = True,
        integrity = COSIGN_VERSIONS[repository_ctx.attr.cosign_version][platform],
    )
    repository_ctx.file("BUILD.bazel", COSIGN_BUILD_TMPL)

cosign_repositories = repository_rule(
    _cosign_repo_impl,
    doc = "Fetch external tools needed for cosign toolchain",
    attrs = {
        "cosign_version": attr.string(mandatory = True, values = COSIGN_VERSIONS.keys()),
        "platform": attr.string(mandatory = True, values = PLATFORMS.keys()),
    },
)

# Wrapper macro around everything above, this is the primary API
def cosign_register_toolchains(name, register = True):
    """Convenience macro for users which does typical setup.

    - create a repository for each built-in platform like "cosign_linux_amd64" -
      this repository is lazily fetched when node is needed for that platform.
    - create a repository exposing toolchains for each platform like "oci_platforms"
    - register a toolchain pointing at each platform
    Users can avoid this macro and do these steps themselves, if they want more control.
    Args:
        name: base name for cosign repository, like "oci_cosign"
    """
    toolchain_name = "{name}_toolchains".format(name = name)

    for platform in PLATFORMS.keys():
        cosign_repositories(
            name = "{name}_{platform}".format(name = name, platform = platform),
            platform = platform,
            cosign_version = COSIGN_VERSIONS.keys()[0],
        )
        if register:
            native.register_toolchains("@{}//:{}_toolchain".format(toolchain_name, platform))

    toolchains_repo(
        name = toolchain_name,
        toolchain_type = "@rules_oci//cosign:toolchain_type",
        # avoiding use of .format since {platform} is formatted by toolchains_repo for each platform.
        toolchain = "@%s_{platform}//:cosign_toolchain" % name,
    )
