import { ChangeDetectionStrategy, Component, OnDestroy } from '@angular/core';
import { MatLegacyTableDataSource } from '@angular/material/legacy-table';
import { ActivatedRoute } from '@angular/router';
import { BehaviorSubject, Subscription } from 'rxjs';
import { StorageClient, Transfer, TransferSubscription } from '../client';
import { Synchronizer } from '../core/services/Synchronizer';
import { YamcsService } from '../core/services/YamcsService';
import { TransferItem } from './TransferItem';

@Component({
  templateUrl: './FailedTransfersTab.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class FailedTransfersTab implements OnDestroy {

  serviceName$ = new BehaviorSubject<string | null>(null);
  dataSource = new MatLegacyTableDataSource<TransferItem>();

  private storageClient: StorageClient;
  private transfersById = new Map<number, TransferItem>();
  private transferSubscription: TransferSubscription;

  private dirty = false;
  private syncSubscription: Subscription;

  private queryParamSubscription: Subscription;

  hasTransferType = false;

  constructor(
    private yamcs: YamcsService,
    route: ActivatedRoute,
    synchronizer: Synchronizer,
  ) {
    this.hasTransferType = history.state.hasTransferType;
    this.storageClient = yamcs.createStorageClient();
    this.queryParamSubscription = route.queryParamMap.subscribe(params => {
      const service = params.get('service');
      this.serviceName$.next(service);
      this.switchService(service);
    });
    this.syncSubscription = synchronizer.sync(() => {
      if (this.dirty) {
        const values = [...this.transfersById.values()];
        values.sort((a, b) => this.compareTransfers(a.transfer, b.transfer));
        this.dataSource.data = values;
        this.dirty = false;
      }
    });
  }

  private compareTransfers(a: Transfer, b: Transfer) {
    const time1 = a.creationTime || a.startTime || "";
    const time2 = b.creationTime || b.startTime || "";
    // most recent transfers on top
    return time2.localeCompare(time1);
  }

  private switchService(service: string | null) {
    // Clear state
    this.dirty = false;
    this.transfersById.clear();
    this.dataSource.data = [];
    if (this.transferSubscription) {
      this.transferSubscription.cancel();
    }

    if (service) {
      this.transferSubscription = this.yamcs.yamcsClient.createTransferSubscription({
        instance: this.yamcs.instance!,
        serviceName: service,
      }, transfer => {
        switch (transfer.state) {
          case 'FAILED':
            this.setOrUpdate(transfer);
            break;
        }

        // throttle updates, it can get spammy
        this.dirty = true;
      });
    }
  }

  // Do our best to preserve top-level object identity
  // It improves change detection behaviour
  private setOrUpdate(transfer: Transfer) {
    let item = this.transfersById.get(transfer.id);
    if (item) {
      item.updateTransfer(transfer);
    } else {
      const objectUrl = transfer.objectName ? this.storageClient.getObjectURL(transfer.bucket, transfer.objectName) : '';
      item = new TransferItem(transfer, objectUrl);
      this.transfersById.set(transfer.id, item);
    }
  }

  ngOnDestroy() {
    if (this.syncSubscription) {
      this.syncSubscription.unsubscribe();
    }
    if (this.queryParamSubscription) {
      this.queryParamSubscription.unsubscribe();
    }
    if (this.transferSubscription) {
      this.transferSubscription.cancel();
    }
  }
}
