﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using DbUp.Builder;
using DbUp.Engine.Output;
using DbUp.Engine.Transactions;

namespace DbUp.Oracle
{
    public static class OracleExtensions
    {
        public static UpgradeEngineBuilder OracleDatabase(this SupportedDatabases supported, string connectionString)
        {
            foreach (var pair in connectionString.Split(';').Select(s => s.Split('=')).Where(pair => pair.Length == 2).Where(pair => pair[0].ToLower() == "database"))
            {
                return OracleDatabase(new OracleConnectionManager(connectionString), pair[1]);
            }

            return OracleDatabase(new OracleConnectionManager(connectionString));
        }

        /// <summary>
        /// Creates an upgrader for MySql databases.
        /// </summary>
        /// <param name="supported">Fluent helper type.</param>
        /// <param name="connectionString">MySql database connection string.</param>
        /// <param name="schema">Which MySql schema to check for changes</param>
        /// <returns>
        /// A builder for a database upgrader designed for MySql databases.
        /// </returns>
        public static UpgradeEngineBuilder OracleDatabase(this SupportedDatabases supported, string connectionString, string schema)
        {
            return OracleDatabase(new OracleConnectionManager(connectionString), schema);
        }

        /// <summary>
        /// Creates an upgrader for MySql databases.
        /// </summary>
        /// <param name="supported">Fluent helper type.</param>
        /// <param name="connectionManager">The <see cref="MySqlConnectionManager"/> to be used during a database upgrade.</param>
        /// <returns>
        /// A builder for a database upgrader designed for MySql databases.
        /// </returns>
        public static UpgradeEngineBuilder OracleDatabase(this SupportedDatabases supported, IConnectionManager connectionManager)
            => OracleDatabase(connectionManager);

        /// <summary>
        /// Creates an upgrader for MySql databases.
        /// </summary>
        /// <param name="connectionManager">The <see cref="MySqlConnectionManager"/> to be used during a database upgrade.</param>
        /// <returns>
        /// A builder for a database upgrader designed for MySql databases.
        /// </returns>
        public static UpgradeEngineBuilder OracleDatabase(IConnectionManager connectionManager)
        {
            return OracleDatabase(connectionManager, null);
        }

        /// <summary>
        /// Creates an upgrader for MySql databases.
        /// </summary>
        /// <param name="connectionManager">The <see cref="MySqlConnectionManager"/> to be used during a database upgrade.</param>
        /// /// <param name="schema">Which MySQL schema to check for changes</param>
        /// <returns>
        /// A builder for a database upgrader designed for MySql databases.
        /// </returns>
        public static UpgradeEngineBuilder OracleDatabase(IConnectionManager connectionManager, string schema)
        {
            var builder = new UpgradeEngineBuilder();
            builder.Configure(c => c.ConnectionManager = connectionManager);
            builder.Configure(c => c.ScriptExecutor = new OracleScriptExecutor(() => c.ConnectionManager, () => c.Log, null, () => c.VariablesEnabled, c.ScriptPreprocessors, () => c.Journal));
            builder.Configure(c => c.Journal = new OracleTableJournal(() => c.ConnectionManager, () => c.Log, schema, "schemaversions"));
            builder.WithPreprocessor(new OraclePreprocessor());
            return builder;
        }
    }
}
