<?php

// This file is part of Minz.
// Copyright 2020-2025 Marien Fressinaud
// SPDX-License-Identifier: AGPL-3.0-or-later

namespace Minz\Form;

/**
 * The CsrfToken class provides a mechanism to protect against Cross-site
 * Request Forgery attacks.
 *
 * It is used alongside with the Csrf trait that must be used in Forms.
 *
 * The CSRF tokens generated by this class are always unique. They are built
 * using different variables:
 *
 * - the user session id: it is generated once per session and is created by the
 *   Csrf trait.
 * - the token name: it should be unique for each form and is passed by the
 *   Csrf trait.
 * - a random value: it allows to generate unique values each time we get a
 *   token. It is generated internally and appended to the token.
 * - a secret: this secret is used to sign the token hash. It is set globally
 *   in the configuration (see Configuration::$secret_key).
 *
 *     use Minz\Form;
 *
 *     $csrf_token = new Form\CsrfToken($some_session_id);
 *
 *     // This token must be passed in a form as a hidden input.
 *     $token = $csrf->get('my_form');
 *
 *     // Then, when validating the form:
 *     $token_from_form = $request->param('csrf_token', '');
 *     $is_valid = $csrf->validate($token_from_form, 'my_form');
 *
 * @see \Minz\Form
 * @see \Minz\Form\Csrf
 *
 * @see https://en.wikipedia.org/wiki/Cross-site_request_forgery
 * @see https://cheatsheetseries.owasp.org/cheatsheets/Cross-Site_Request_Forgery_Prevention_Cheat_Sheet.html
 */
class CsrfToken
{
    public function __construct(
        private string $session_id,
    ) {
    }

    /**
     * Return a valid CSRF token for the given name.
     *
     * @param non-empty-string $name
     * @return non-empty-string
     */
    public function get(string $name = '@global'): string
    {
        $random_value = \Minz\Random::hex(64);
        $hmac = self::generateHmac($name, $random_value);

        return "{$hmac}.{$random_value}";
    }

    /**
     * Validate that the token is correct for the given name.
     *
     * @param non-empty-string $name
     */
    public function validate(string $token, string $name = '@global'): bool
    {
        $token_parts = explode('.', $token);

        if (count($token_parts) !== 2) {
            return false;
        }

        $request_hmac = $token_parts[0];
        $request_random_value = $token_parts[1];

        if ($request_hmac === '' || $request_random_value === '') {
            return false;
        }

        $expected_hmac = self::generateHmac($name, $request_random_value);

        return hash_equals($expected_hmac, $request_hmac);
    }

    /**
     * @param non-empty-string $name
     * @param non-empty-string $random_value
     * @return non-empty-string
     */
    private function generateHmac(string $name, string $random_value): string
    {
        $session_id_length = strlen($this->session_id);
        $name_length = strlen($name);
        $random_value_length = strlen($random_value);

        $data = "{$session_id_length}:{$this->session_id}";
        $data .= ":{$name_length}:{$name}";
        $data .= ":{$random_value_length}:{$random_value}";

        return hash_hmac('sha256', $data, \Minz\Configuration::$secret_key);
    }
}
