<?php
declare(strict_types = 1);

namespace Properties\Formal\ORM;

use Formal\ORM\{
    Manager,
    Specification\Child,
};
use Fixtures\Formal\ORM\{
    User,
    AddressValue,
};
use Innmind\Specification\Sign;
use Innmind\BlackBox\{
    Set,
    Property,
    Runner\Assert,
};
use Innmind\Immutable\Either;
use Fixtures\Innmind\TimeContinuum\PointInTime;

/**
 * @implements Property<Manager>
 */
final class MatchingCollection implements Property
{
    private $createdAt;
    private string $prefix;
    private string $name1;
    private string $name2;
    private array $addresses;

    private function __construct(
        $createdAt,
        string $prefix,
        string $name1,
        string $name2,
        array $addresses,
    ) {
        $this->createdAt = $createdAt;
        $this->prefix = $prefix;
        $this->name1 = $name1;
        $this->name2 = $name2;
        $this->addresses = $addresses;
    }

    public static function any(): Set\Provider
    {
        return Set::compose(
            static fn(...$args) => new self(...$args),
            PointInTime::any(),
            Set::strings()
                ->madeOf(Set::strings()->chars()->alphanumerical())
                ->between(10, 100),
            Set::strings()
                ->madeOf(Set::strings()->chars()->alphanumerical())
                ->between(10, 100),
            Set::strings()
                ->madeOf(Set::strings()->chars()->alphanumerical())
                ->between(10, 100),
            Set::sequence(
                Set::strings()
                    ->madeOf(Set::strings()->chars()->alphanumerical())
                    ->between(1, 200),
            )->between(0, 10),
        );
    }

    public function applicableTo(object $manager): bool
    {
        return true;
    }

    public function ensureHeldBy(Assert $assert, object $manager): object
    {
        $user1 = User::new($this->createdAt);
        $user2 = User::new($this->createdAt);
        $user3 = User::new($this->createdAt);

        foreach ($this->addresses as $address) {
            $user1 = $user1->addAddress($address);
            $user2 = $user2->addAddress($address);
            $user3 = $user3->addAddress($address);
        }

        $user1 = $user1->addAddress($this->name1);
        $user2 = $user2->addAddress($this->name2);
        $user3 = $user3->addAddress($this->prefix.$this->name1);

        $repository = $manager->repository(User::class);
        $manager->transactional(
            static function() use ($repository, $user1, $user2, $user3) {
                $repository->put($user1)->unwrap();
                $repository->put($user2)->unwrap();
                $repository->put($user3)->unwrap();

                return Either::right(null);
            },
        );

        $found = $repository
            ->matching(Child::of('addresses', AddressValue::of(
                Sign::equality,
                $this->name1,
            )))
            ->map(static fn($user) => $user->id()->toString())
            ->toList();

        $assert
            ->expected($user1->id()->toString())
            ->in($found);
        $assert
            ->expected($user2->id()->toString())
            ->not()
            ->in($found);
        $assert
            ->expected($user3->id()->toString())
            ->not()
            ->in($found);

        $found = $repository
            ->matching(Child::of('addresses', AddressValue::of(
                Sign::equality,
                $this->name2,
            )))
            ->map(static fn($user) => $user->id()->toString())
            ->toList();

        $assert
            ->expected($user1->id()->toString())
            ->not()
            ->in($found);
        $assert
            ->expected($user2->id()->toString())
            ->in($found);
        $assert
            ->expected($user3->id()->toString())
            ->not()
            ->in($found);

        $found = $repository
            ->matching(Child::of(
                'addresses',
                AddressValue::of(
                    Sign::endsWith,
                    $this->name1,
                )->and(AddressValue::of(
                    Sign::startsWith,
                    $this->prefix,
                )),
            ))
            ->map(static fn($user) => $user->id()->toString())
            ->toList();

        $assert
            ->expected($user1->id()->toString())
            ->not()
            ->in($found);
        $assert
            ->expected($user2->id()->toString())
            ->not()
            ->in($found);
        $assert
            ->expected($user3->id()->toString())
            ->in($found);

        $found = $repository
            ->matching(Child::of('addresses', AddressValue::of(
                Sign::endsWith,
                $this->name1,
            )))
            ->map(static fn($user) => $user->id()->toString())
            ->toList();

        $assert
            ->expected($user1->id()->toString())
            ->in($found);
        $assert
            ->expected($user2->id()->toString())
            ->not()
            ->in($found);
        $assert
            ->expected($user3->id()->toString())
            ->in($found);

        return $manager;
    }
}
