<?php
declare(strict_types = 1);

namespace Formal\ORM\Adapter\Filesystem;

use Formal\ORM\{
    Adapter\Repository as RepositoryInterface,
    Adapter\Repository\Effectful,
    Definition\Aggregate as Definition,
    Raw\Aggregate,
    Raw\Diff,
    Sort,
    Effect,
};
use Innmind\Filesystem\{
    Name,
    Directory,
};
use Innmind\Specification\Specification;
use Innmind\Immutable\{
    Attempt,
    SideEffect,
    Maybe,
    Sequence,
    Predicate\Instance,
};

/**
 * @internal
 * @template T of object
 * @implements RepositoryInterface<T>
 */
final class Repository implements RepositoryInterface, Effectful
{
    private Transaction $transaction;
    /** @var Definition<T> */
    private Definition $definition;
    /** @var Fold<T> */
    private Fold $fold;
    private Encode $encode;
    /** @var Decode<T> */
    private Decode $decode;
    private EncodeEffect $encodeEffect;

    /**
     * @param Definition<T> $definition
     */
    private function __construct(Transaction $transaction, Definition $definition)
    {
        $this->transaction = $transaction;
        $this->definition = $definition;
        $this->fold = Fold::of($definition);
        $this->encode = Encode::new();
        $this->decode = Decode::of($definition);
        $this->encodeEffect = EncodeEffect::new();
    }

    /**
     * @internal
     * @template A of object
     *
     * @param Definition<A> $definition
     *
     * @return self<A>
     */
    public static function of(Transaction $transaction, Definition $definition): self
    {
        return new self($transaction, $definition);
    }

    #[\Override]
    public function get(Aggregate\Id $id): Maybe
    {
        return $this
            ->directory()
            ->get(Name::of($id->value()))
            ->keep(Instance::of(Directory::class))
            ->flatMap(($this->decode)($id));
    }

    #[\Override]
    public function contains(Aggregate\Id $id): bool
    {
        return $this
            ->directory()
            ->contains(Name::of($id->value()));
    }

    #[\Override]
    public function add(Aggregate $data): Attempt
    {
        $encoded = Directory::named($this->definition->name())->add(
            ($this->encode)($data),
        );

        return Attempt::of(
            fn() => $this->transaction->mutate(
                static fn($adapter) => $adapter->add($encoded)->unwrap(),
            ),
        )->map(static fn() => SideEffect::identity());
    }

    #[\Override]
    public function update(Diff $data): Attempt
    {
        $encoded = Directory::named($this->definition->name())->add(
            ($this->encode)($data),
        );

        return Attempt::of(
            fn() => $this->transaction->mutate(
                static fn($adapter) => $adapter->add($encoded)->unwrap(),
            ),
        )->map(static fn() => SideEffect::identity());
    }

    #[\Override]
    public function effect(
        Effect\Normalized $effect,
        ?Specification $specification,
    ): Attempt {
        $effect = ($this->encodeEffect)($effect);

        return $this
            ->fetch(
                $specification,
                null,
                null,
                null,
            )
            ->map($effect)
            ->sink(SideEffect::identity())
            ->attempt(fn($_, $data) => $this->update($data));
    }

    #[\Override]
    public function remove(Aggregate\Id $id): Attempt
    {
        $mutated = Directory::named($this->definition->name())->remove(
            Name::of($id->value()),
        );

        return Attempt::of(
            fn() => $this->transaction->mutate(
                static fn($adapter) => $adapter->add($mutated)->unwrap(),
            ),
        )->map(static fn() => SideEffect::identity());
    }

    #[\Override]
    public function removeAll(Specification $specification): Attempt
    {
        return $this
            ->fetch(
                $specification,
                null,
                null,
                null,
            )
            ->map(static fn($aggregate) => $aggregate->id())
            ->sink(SideEffect::identity())
            ->attempt(fn($_, $id) => $this->remove($id));
    }

    #[\Override]
    public function fetch(
        ?Specification $specification,
        null|Sort\Property|Sort\Entity $sort,
        ?int $drop,
        ?int $take,
    ): Sequence {
        $aggregates = $this->all();

        if ($specification) {
            $aggregates = $aggregates->filter(
                ($this->fold)($specification),
            );
        }

        if ($sort) {
            $compare = match ($sort->direction()) {
                Sort::asc => static fn(null|string|int|float|bool $a, null|string|int|float|bool $b) => $a <=> $b,
                Sort::desc => static fn(null|string|int|float|bool $a, null|string|int|float|bool $b) => $b <=> $a,
            };
            $pluck = match (true) {
                $sort instanceof Sort\Property => static fn(Aggregate $x): mixed => $x
                    ->properties()
                    ->find(static fn($property) => $property->name() === $sort->name())
                    ->match(
                        static fn($property) => $property->value(),
                        static fn() => throw new \LogicException("'{$sort->name()}' not found"),
                    ),
                $sort instanceof Sort\Entity => static fn(Aggregate $x): mixed => $x
                    ->entities()
                    ->find(static fn($entity) => $entity->name() === $sort->name())
                    ->flatMap(
                        static fn($entity) => $entity
                            ->properties()
                            ->find(static fn($property) => $property->name() === $sort->property()->name()),
                    )
                    ->match(
                        static fn($property) => $property->value(),
                        static fn() => throw new \LogicException("'{$sort->name()}.{$sort->property()->name()}' not found"),
                    )
            };

            $aggregates = $aggregates->sort(static fn($a, $b) => $compare(
                $pluck($a),
                $pluck($b),
            ));
        }

        if (\is_int($drop)) {
            $aggregates = $aggregates->drop($drop);
        }

        if (\is_int($take)) {
            $aggregates = $aggregates->take($take);
        }

        return $aggregates;
    }

    #[\Override]
    public function size(?Specification $specification = null): int
    {
        return $this
            ->fetch($specification, null, null, null)
            ->size();
    }

    #[\Override]
    public function any(?Specification $specification = null): bool
    {
        return $this
            ->fetch($specification, null, null, 1)
            ->first()
            ->match(
                static fn() => true,
                static fn() => false,
            );
    }

    /**
     * @return Sequence<Aggregate>
     */
    private function all(): Sequence
    {
        $decode = ($this->decode)();

        return $this
            ->directory()
            ->all()
            ->keep(Instance::of(Directory::class))
            ->flatMap(static fn($file) => $decode($file)->toSequence());
    }

    private function directory(): Directory
    {
        $name = Name::of($this->definition->name());

        return $this->transaction->get($name);
    }
}
